/*  This file is part of the OpenLB library
 *
 *  Copyright (C) 2006, 2007 Jonas Latt
 *  E-mail contact: info@openlb.net
 *  The most recent release of OpenLB can be downloaded at
 *  <http://www.openlb.net/>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this program; if not, write to the Free
 *  Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA  02110-1301, USA.
*/

/** \file
 * Interface for post-processing steps -- header file.
 */
#ifndef POST_PROCESSING_H
#define POST_PROCESSING_H

#include <vector>

#include "operator.h"
#include "blockStructure.h"
#include "blockLattice.h"
#include "io/ostreamManager.h"

namespace olb {

/////////////////// 2D Postprocessing ///////////////////////////////

/// Interface of 2D post-processing steps.
template<typename T, typename DESCRIPTOR>
class PostProcessor2D {
public:
  PostProcessor2D():
    _priority{0} { }
  virtual ~PostProcessor2D() { }
  /// Execute post-processing step
  virtual void process(BlockLattice<T,DESCRIPTOR>& blockLattice) =0;
  /// Execute post-processing step on a sublattice
  virtual void processSubDomain(BlockLattice<T,DESCRIPTOR>& blockLattice,
                                int x0_, int x1_, int y0_, int y1_) =0;
  /// Extent of application area (0 for purely local operations)
  virtual int extent() const =0;
  /// Extent of application area along a direction (0 or 1)
  virtual int extent(int direction) const =0;
  /// read and write access to name
  std::string& getName();
  /// read only access to name
  std::string const& getName() const;
  /// read only access to priority
  int getPriority() const;
protected:
  int _priority;
private:
  std::string _name;
};

template<typename T, typename DESCRIPTOR>
class PostProcessorGenerator2D {
public:
  PostProcessorGenerator2D(int x0_, int x1_, int y0_, int y1_);
  virtual ~PostProcessorGenerator2D() { }
  void shift(int deltaX, int deltaY);
  void shift(LatticeR<2> delta);
  bool extract(int x0_, int x1_, int y0_, int y1_);
  bool extract(LatticeR<2> lower, LatticeR<2> upper);
  void reset(int x0_, int x1_, int y0_, int y1_);
  void reset(LatticeR<2> lower, LatticeR<2> upper);
  virtual PostProcessor2D<T,DESCRIPTOR>* generate() const =0;
  virtual PostProcessorGenerator2D<T,DESCRIPTOR>* clone() const =0;
protected:
  int x0, x1, y0, y1;
};

template<typename T, typename DESCRIPTOR>
class LatticeCouplingGenerator2D {
public:
  LatticeCouplingGenerator2D(int x0_, int x1_, int y0_, int y1_);
  virtual ~LatticeCouplingGenerator2D() { }
  void shift(LatticeR<2> delta);
  void shift(int deltaX, int deltaY);
  bool extract(int x0_, int x1_, int y0_, int y1_);
  bool extract(LatticeR<2> lower, LatticeR<2> upper);
  void reset(int x0_, int x1_, int y0_, int y1_);
  void reset(LatticeR<2> lower, LatticeR<2> upper);
  virtual PostProcessor2D<T,DESCRIPTOR>* generate(std::vector<BlockStructureD<2>*> partners) const =0;
  virtual LatticeCouplingGenerator2D<T,DESCRIPTOR>* clone() const =0;
protected:
  int x0, x1, y0, y1;
};


template<typename T, typename DESCRIPTOR>
struct LocalPostProcessor2D : public PostProcessor2D<T,DESCRIPTOR> {
};

template<typename T, typename DESCRIPTOR>
struct GlobalPostProcessor2D : public PostProcessor2D<T,DESCRIPTOR> {
  void process(BlockLattice<T,DESCRIPTOR>& blockLattice) override =0;
  void processSubDomain(BlockLattice<T,DESCRIPTOR>& blockLattice,
                        int x0_, int x1_, int y0_, int y1_ ) override
  {
    this -> process(blockLattice);
  }
  int extent() const override
  {
    return 0;
  }
  int extent(int direction) const override
  {
    return 0;
  }
};


/////////////////// 3D Postprocessing ///////////////////////////////

template<typename T, typename DESCRIPTOR>
class PostProcessor3D {
public:
  PostProcessor3D():
    _priority{0} { }
  virtual ~PostProcessor3D() { }
  /// Execute post-processing step
  virtual void process(BlockLattice<T,DESCRIPTOR>& blockLattice) =0;
  /// Execute post-processing step on a sublattice
  virtual void processSubDomain(BlockLattice<T,DESCRIPTOR>& blockLattice,
                                int x0_, int x1_, int y0_, int y1_,
                                int z0_, int z1_ ) =0;
  /// Extent of application area (0 for purely local operations)
  virtual int extent() const =0;
  /// Extent of application area along a direction (0 or 1)
  virtual int extent(int direction) const =0;
  /// read and write access to name
  std::string& getName();
  /// read only access to name
  std::string const& getName() const;
  /// read only access to priority
  int getPriority() const;
protected:
  int _priority;
private:
  std::string _name;
};

template<typename T, typename DESCRIPTOR>
class PostProcessorGenerator3D {
public:
  PostProcessorGenerator3D( int x0_, int x1_, int y0_, int y1_,
                            int z0_, int z1_ );
  virtual ~PostProcessorGenerator3D() { }
  void shift(int deltaX, int deltaY, int deltaZ, int iC_=-1);
  void shift(LatticeR<3> delta, int iC_=-1);
  bool extract(int x0_, int x1_, int y0_, int y1_, int z0_, int z1_);
  bool extract(LatticeR<3> lower, LatticeR<3> upper);
  void reset(int x0_, int x1_, int y0_, int y1_, int z0_, int z1_);
  void reset(LatticeR<3> lower, LatticeR<3> upper);
  virtual PostProcessor3D<T,DESCRIPTOR>* generate() const =0;
  virtual PostProcessorGenerator3D<T,DESCRIPTOR>* clone() const =0;
protected:
  int x0, x1, y0, y1, z0, z1, iC;
};


template<typename T, typename DESCRIPTOR>
class LatticeCouplingGenerator3D {
public:
  LatticeCouplingGenerator3D() = delete;
  LatticeCouplingGenerator3D( int x0_, int x1_, int y0_, int y1_,
                              int z0_, int z1_ );
  virtual ~LatticeCouplingGenerator3D() { }
  void shift(LatticeR<3> delta, int iC_=-1);
  void shift(int deltaX, int deltaY, int deltaZ, int iC_=-1);
  bool extract(int x0_, int x1_, int y0_, int y1_, int z0_, int z1_);
  bool extract(LatticeR<3> lower, LatticeR<3> upper);
  void reset(LatticeR<3> lower, LatticeR<3> upper);
  void reset(int x0_, int x1_, int y0_, int y1_, int z0_, int z1_);
  virtual PostProcessor3D<T,DESCRIPTOR>* generate(std::vector<BlockStructureD<3>*> partners) const =0;
  virtual LatticeCouplingGenerator3D<T,DESCRIPTOR>* clone() const =0;
protected:
  int x0, x1, y0, y1, z0, z1, iC;
};


template<typename T, typename DESCRIPTOR>
struct LocalPostProcessor3D : public PostProcessor3D<T,DESCRIPTOR> {
};

template<typename T, typename DESCRIPTOR>
struct GlobalPostProcessor3D : public PostProcessor3D<T,DESCRIPTOR> {
  void process(BlockLattice<T,DESCRIPTOR>& blockLattice) override =0;
  void processSubDomain(BlockLattice<T,DESCRIPTOR>& blockLattice,
                        int x0_, int x1_, int y0_, int y1_,
                        int z0_, int z1_ ) override
  {
    this -> process(blockLattice);
  }
  int extent() const override
  {
    return 0;
  }
  int extent(int direction) const override
  {
    return 0;
  }
};


struct StatisticsPostProcessor {
  static constexpr OperatorScope scope = OperatorScope::PerBlock;

  int getPriority() const {
    return std::numeric_limits<int>::min();
  }

  template <typename BLOCK>
  struct type {
    void setup(BLOCK& blockLattice);
    void apply(BLOCK& blockLattice);
  };

  template <typename BLOCK>
  void setup(BLOCK& blockLattice) {
    type<BLOCK>{}.setup(blockLattice);
  }

  template <typename BLOCK>
  void apply(BLOCK& blockLattice) {
    type<BLOCK>{}.apply(blockLattice);
  }

};


}  // namespace olb

#endif
