/***********************************************************************************
 *                                                                                 *
 * Voreen - The Volume Rendering Engine                                            *
 *                                                                                 *
 * Copyright (C) 2005-2013 University of Muenster, Germany.                        *
 * Visualization and Computer Graphics Group <http://viscg.uni-muenster.de>        *
 * For a list of authors please refer to the file "CREDITS.txt".                   *
 *                                                                                 *
 * This file is part of the Voreen software package. Voreen is free software:      *
 * you can redistribute it and/or modify it under the terms of the GNU General     *
 * Public License version 2 as published by the Free Software Foundation.          *
 *                                                                                 *
 * Voreen is distributed in the hope that it will be useful, but WITHOUT ANY       *
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR   *
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.      *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License in the file   *
 * "LICENSE.txt" along with this file. If not, see <http://www.gnu.org/licenses/>. *
 *                                                                                 *
 * For non-commercial academic use see the license exception specified in the file *
 * "LICENSE-academic.txt". To get information about commercial licensing please    *
 * contact the authors.                                                            *
 *                                                                                 *
 ***********************************************************************************/

#include "voreen/core/datastructures/geometry/geometrysequence.h"

#include "voreen/core/io/serialization/serialization.h"
#include "voreen/core/utils/hashing.h"
#include "tgt/glmath.h"

#include <sstream>

namespace voreen {

const std::string GeometrySequence::loggerCat_("voreen.GeometrySequence");

GeometrySequence::GeometrySequence(bool takeOwnership)
    : Geometry()
    , takeOwnership_(takeOwnership)
{}

GeometrySequence::~GeometrySequence() {
    if(takeOwnership_) {
        for(size_t i=0; i<geometries_.size(); i++)
            delete geometries_[i];
    }

    geometries_.clear();
}

void GeometrySequence::addGeometry(Geometry* g) {
    geometries_.push_back(g);
}

bool GeometrySequence::equals(const Geometry* geometry, double epsilon) const {
    tgtAssert(geometry, "null pointer passed");
    const GeometrySequence* gs = dynamic_cast<const GeometrySequence*>(geometry);
    if(gs) {
        if(geometries_.size() != gs->geometries_.size())
            return false;

        for(size_t i=0; i<geometries_.size(); i++) {
            if(!geometries_[i]->equals(gs->geometries_[i], epsilon))
                return false;
        }
        return true;
    }
    else
        return false;
}

void GeometrySequence::clip(const tgt::plane& clipPlane, double epsilon) {
    for(size_t i=0; i<geometries_.size(); i++) 
        geometries_[i]->clip(clipPlane, epsilon);
}

void GeometrySequence::render() const {
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    tgt::multMatrix(getTransformationMatrix());

    for(size_t i=0; i<geometries_.size(); i++)
        geometries_[i]->render();

    glPopMatrix();
}

tgt::Bounds GeometrySequence::getBoundingBox(bool transformed) const {
    tgt::Bounds b;
    for(size_t i=0; i<geometries_.size(); i++)
        b.addVolume(geometries_[i]->getBoundingBox(transformed));

    if(transformed)
        return b.transform(getTransformationMatrix());
    else
        return b;
}

void GeometrySequence::serialize(XmlSerializer& s) const {
    Geometry::serialize(s);
    s.serialize("geometries", geometries_);
}

void GeometrySequence::deserialize(XmlDeserializer& s) {
    Geometry::deserialize(s);
    s.deserialize("geometries", geometries_);
}

} // namespace
