%% DPE_LINEAR_RECURSIVE calculates the generalized median of a set 
%
%  given its embedding and weighted median function
%  Parameters:
%   object_vector       - vector representation of the object to be reconstructed
%   unique_embbeding    - embedding vectors without duplicates as NxK Matrix
%   unique_set          - object set without duplicates as Cell array
%   dist_func           - distance function between objects: dist_func(o_1,o_2)
%   weighted_mean_func  - weighted mean function between objects: weighted_mean_func(o_1,o_2,alpha)
%   find_best           - 1, if the best median approximation is returned, 0 if the last is returned.
%   
%  Returns:
%   rec_obj    - The reconstructed object. If an error occures, the set_median is returned as approximation
%   best_value - Value of best_crit for rec_obj

function [ rec_obj, best_value ] = dpe_triangular_recursive(object_vector, unique_embedding, unique_set, full_set, weighted_mean_func, best_crit)


    %% stop if there is only one element in the set
    if (numel(unique_set) == 1)
        rec_obj = unique_set{1};
        best_value = best_crit(full_set,rec_obj);
        return;
    end;
   

    %% return NaN; if the embedding is incorrect
    if (size(unique_embedding,1) ~= length(unique_set))
        warning('embedding is incorrect');
        rec_obj = NaN;
        best_value = NaN;
        return;
    end
    
    %% sort by distance
    [~, index] = sort(pdist2(unique_embedding,object_vector));
    n = length(unique_set);
    
    new_set = cell(1,ceil(n/3));
    new_embedding = zeros(ceil(n/3),size(unique_embedding,2));

    rec_obj = unique_set{index(1)};
    best_value = best_crit(full_set,rec_obj);
    
    %% merge each sets of two nearest points
    for i = 1:3:n-2
        closest_vectors = unique_embedding(index(i:i+2), :);

        % project on first line
        percentage = dpe_project_point_on_line(closest_vectors(1,:),closest_vectors(2,:),object_vector);
        gen_median_vec = closest_vectors(1,:) + percentage .* (closest_vectors(2,:) - closest_vectors(1,:));
        cur_obj = dpe_get_best_weighted_mean(unique_set{index(i)}, unique_set{index(i+1)}, percentage, ...
            weighted_mean_func, full_set, best_crit);
        
        % project on second line
        percentage = dpe_project_point_on_line(gen_median_vec,closest_vectors(3,:),object_vector);
        [cur_obj, new_best_value] = dpe_get_best_weighted_mean(cur_obj, unique_set{index(i+2)}, percentage, ...
            weighted_mean_func, full_set, best_crit);
        gen_median_vec = gen_median_vec + percentage .* (closest_vectors(3,:) - gen_median_vec);


        % save result
        new_embedding((i+2)/3,:) = gen_median_vec;
        new_set((i+2)/3) = {cur_obj};

        %% check if SOD is better
        if (new_best_value < best_value)
            rec_obj = cur_obj;
            best_value = new_best_value;
        end

    end
    
    %% copy or process remaining element, if it exists
    if (rem(n,3) == 1)
        new_embedding(end,:) = unique_embedding(end,:);
        new_set(end) = unique_set(end);
    end
    
    if (rem(n,3) == 2)
        new_embedding(end,:) = dpe_median_vector(unique_embedding(end-1:end,:));
        percentage = dpe_distance_percentage(unique_embedding(end-1,:), unique_embedding(end,:),  new_embedding(end,:));
        new_set{end} = dpe_get_best_weighted_mean(unique_set{index(end-1)}, unique_set{index(end)}, percentage, ...
            weighted_mean_func, full_set, best_crit);
    end
    
    %% do recursion
    
    [cur_obj, new_best_value] = dpe_triangular_recursive(object_vector, new_embedding, new_set, full_set, weighted_mean_func, best_crit);
    
    %% check if SOD is better
    

    if (new_best_value < best_value)
        best_value = new_best_value;
        rec_obj = cur_obj;
    end

    
end
