%% DPE_LINEAR_SEARCH Finds the generalized median using linear searches
%
% Parameters: 
%   object_vector         - estimation of the geometric median in vector space
%   unique_embedding      - NxK matrix of embedding vectors
%   unique_set            - cell array of objects
%   dist_func             - distance function between objects: dist_func(o_i,o_j)
%   weighted_mean_func    - weighted mean function between objects: weighted_mean_func(o_i,o_j,alpha)
%   best crit             - criterium for the best object (see dpe_reconstruction)
%
% Optional:
%   'num-iterations',iter - maximum number of search iterations (default: 1000)
%   'start', method       - method used to calculate the first median approximation. See dpe_reconstruction (default: linear) 
%
% Returns:
%   rec_obj               - approximation of the reconstructed object
%   best_value            - best_crit value of the reconstructed object

function [rec_obj, best_value] = dpe_linear_search(object_vector, unique_embedding, unique_set, dist_func, weighted_mean_func, best_crit, varargin)


% parse options
param_names = {'num-iterations','start'};
param_defaults = {1000, 'linear'};
[num_iterations, starting_method] = internal.stats.parseArgs(param_names, param_defaults, varargin{:});

rec_obj = dpe_reconstruction(object_vector,unique_embedding, unique_set, dist_func, weighted_mean_func, starting_method, best_crit);

if (isnan(rec_obj))
    rec_obj = NaN;
    best_value = NaN;
    return
end
    
best_value = best_crit(unique_set,rec_obj);
last_best_value = inf;
best_obj = rec_obj;

% for each iteration
for iter = 1:num_iterations

% for each object
    for obj = 1:length(unique_set)
        
        % Search for best percentage that minimizes the SOD Value
        searchCrit = @(percentage)best_value_search(rec_obj,unique_set{obj}, percentage, unique_set, weighted_mean_func, best_crit);
        
        %options.MaxFunEvals = 20;
        options.TolX = last_best_value * 0.001;
        [percentage, new_best_value, ~] = fminbnd(searchCrit,0,1);%,options);

        % save minimum
        if (new_best_value < best_value)
            best_obj = weighted_mean_func(rec_obj,unique_set{obj},percentage);
            best_value = new_best_value;
        end
    end
    
    % set new median
    rec_obj = best_obj;
    
    
    % stop if there is no change
    if (last_best_value == best_value)
        return;
    end
    
    
    last_best_value = best_value;


end

end


function best_value = best_value_search(a,b, percentage, set, weighted_mean_func, best_crit)
    weighted_mean = weighted_mean_func(a,b,percentage);
    best_value = best_crit(set,weighted_mean);
end
