%% DPE_LINEAR_RECURSIVE calculates the generalized median of a set 
%  given its embedding and weighted median function
%
%  Parameters:
%   object_vector       - vector representation of the object to be reconstructed
%   unique_embbeding    - embedding vectors without duplicates as NxK Matrix
%   unique_set          - object set without duplicates as Cell array
%   weighted_mean_func  - weighted mean function between objects: weighted_mean_func(o_1,o_2,alpha)
%   find_best           - 1, if the best median approximation is returned, 0 if the last is returned.
%   
%  Returns:
%   gen_median - The calculated generalized median. If an error occures, the set_median is returned as approximation
%   best_value - Value of best_crit for rec_obj

function [ rec_obj, best_value ] = dpe_linear_recursive(object_vector, unique_embedding, unique_set, full_set, weighted_mean_func, best_crit)


    %% stop if there is only one element in the set
    if (numel(unique_set) == 1)
        rec_obj = unique_set{1};
        best_value = best_crit(full_set,rec_obj);
        return;
    end;

    %% return NaN if the embedding is incorrect
    if (size(unique_embedding,1) ~= length(unique_set))
        warning('embedding is incorrect');
        rec_obj = NaN;
        best_value = NaN;
        return;
    end
    

    
    %% find sets 
    n = length(unique_set);
    
    [~, index] = sort(pdist2(unique_embedding,object_vector));
    
     % remove duplicate
    index = unique(sort(index,2),'rows')';
    
    % find single element (always the first elementif there) and move to the end
    if index(1,1) == -1
       tmp = index(2,1);
       index(:,1) = index(:,end);
       index(1,end) = tmp;        
    end
    
    % cut extra elements caused from unmatched points
    index = index(1:n);
    
    
    %% create new sets
    new_set = cell(1,ceil(n/2));
    new_embedding = zeros(ceil(n/2),size(unique_embedding,2));
    
    
  
    rec_obj = unique_set{index(1)};
    best_value = best_crit(full_set,rec_obj);

    
    %% merge each sets of two nearest points
    for i = 1:2:n-1
        closest_vectors = unique_embedding(index(i:i+1), :);
       
        % do linear reconstruction
        percentage = dpe_project_point_on_line(closest_vectors(1,:), closest_vectors(2,:), object_vector);
        [cur_obj, new_best_value] = dpe_get_best_weighted_mean(unique_set{index(i)}, unique_set{index(i+1)}, percentage, ...
            weighted_mean_func, full_set, best_crit);
        gen_median_vec = closest_vectors(1,:) + percentage .* (closest_vectors(2,:) - closest_vectors(1,:));


        % save result for recursion
        new_embedding((i+1)/2,:) = gen_median_vec;
        new_set((i+1)/2) = {cur_obj};

        if (new_best_value < best_value)
            rec_obj = cur_obj;
            best_value = new_best_value;
        end

    end
    
    %% copy remaining element, if it exists
    if (rem(n,2))
        new_embedding(end,:) = unique_embedding(end,:);
        new_set(end) = unique_set(end);
    end
    
    %% do recursion
    
    [cur_obj, new_best_value] = dpe_linear_recursive(object_vector, new_embedding, new_set, full_set, weighted_mean_func, best_crit);
    

    if (new_best_value < best_value)
        rec_obj = cur_obj;
        best_value = new_best_value;
    end

end
