function [ gen_median, best_value, valid_result ] = dpe_best_recursive(median_vector, unique_embedding, unique_set, weighted_mean_func, best_crit)
%DPE_BEST_RECURSIVE Extension of the Recursive approach that also considers
%all intermediate results as best result. See Franek, Jiang: Ensemble
%clustering by means of clustering embedding in vector spaces
%Parameters:
%   median_vector      - the object to be reconstructed, in vector space
%   unique_embedding   - embedding consisting of only unique vectors (use dpe_group_elements)
%   unique_set         - set only consisting of unique objects
%   weighted_mean_func - weighted mean function @(a,b,alpha) between objects a and b with ratio alpha
%   best_crit          - minimizing criterium for best object (e.g. sum of distance)
%Returns:
%   gen_median   - reconstructed object
%   best_value   - value of best_crit for the reconstructed object
%   valid-result - true, if a result could be calculated
    if (isnan(unique_embedding))
      gen_median = NaN;
      best_value = NaN;
      valid_result = false;
      return;
    end

    % get a ranking of the neighbours of the euclidean median
    closest_points = knnsearch(unique_embedding, median_vector, 'k', size(unique_embedding, 1));

    % order the points from close to far as row vectors
    RPM = cleanupDims(unique_embedding(closest_points, :));

    while size(RPM,1) ~= size(RPM,2)
        % get a number of nearest points, to create a squared matrix
        if size(RPM, 1) > size(RPM, 2)
            % drop far points
            RPM = cleanupDims(RPM(1:size(RPM, 2), :));
        else
            % drop last dims
            RPM = cleanupDims(RPM(:, 1:size(RPM, 1)));
        end
    end

    if rank(RPM) > 1

        NVM = getNormalVectorMatrix(RPM');

        % computes the euclidean median in the reduced embedding.
        eukMedVecMin = dpe_median_vector(RPM);

        MVM = getMedianVectorMatrix(RPM', NVM, 2, eukMedVecMin);

        [gen_median, best_value, valid_result] = dpe_get_gen_median(RPM', MVM, unique_set, ...
            closest_points, weighted_mean_func, best_crit);
    else
        valid_result = false;
        gen_median = NaN;
        best_value = NaN;
    end

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [ NVM ] = getNormalVectorMatrix( CPM )
%GETNORMALVECTORMATRIX gets the normal vectors of the simplices. This is
%done by solving an equation system depending on the passed points. 
    
    xDim = length(CPM(:,1));
    yDim = length(CPM(1,:));
    maxHypPlaneDim = length(CPM(1,:))-1;
    NVM = zeros(xDim, maxHypPlaneDim);
    b = zeros(yDim, 1);
    b(yDim, 1) = 1;
    
    %walks through all normal-vectors
    for k = maxHypPlaneDim:-1:1
        A = zeros(yDim, xDim);
        
        %------------------------------------------------------------------
        %walks through the rows of the matrix A and filles it up
        %using three different types of equations:
        %------------------------------------------------------------------
        %this is equal to  "(pointX - point(a)) * normalCurrent = 0"
        for a = 1:k
            A(a,:) = (CPM(:,k+1) - CPM(:,a))';
        end
        %this is equal to "normalX * normalCurrent = 0"
        if (maxHypPlaneDim > k)
            for a = (k+1):maxHypPlaneDim
                A(a,:) = (NVM(:,-(a-maxHypPlaneDim-k-1)))';
            end
        end
        
        %SOLUTION BY LINEAR EQUATION-SYSTEM
        %this is equal to ||x|| = 1
        A(yDim, xDim) = 1;
        
        %SOLUTION BY NON-LINEAR EQUATION-SYSTEM
        %gets a vector filled with the equations inv
        %equations = getStrOfEquations(A, b);
        %gets the solution of the equation-system
        %x0 = ones(maxDim, 1);
        %x = lsqnonlin(inline(equations),x0);
        %x = fsolve(inline(equations),x0);
        
        %------------------------------------------------------------------
        %Solves the linear equation system
        %------------------------------------------------------------------
        %If matrix "A" has a bad condition (hint for a (nearly) singular
        %matrix) the pseudo-invert is the tool to get the solution "x".
        if (cond(A) > 1000)
            x = pinv(A) * b;
        else
            x = A\b;
        end
        %Alternative: "linsolve" is multi-threaded optimized, so it is
        %faster than "\" if there are many equations.
        %x = linsolve(A,b);
        
        %NVM has to become a matrix of normal-vectors, so x has
        %to be normalized
        NVM(:,k) = x / norm(x);
    end
   
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [ MVM ] = getMedianVectorMatrix( CPM, NVM, k, eukMedCPM )
%GETMEDIANVECTORMATRIX gets the median vectors beginning on the left column
%with the lowest median m2. In the right column is "eukMedCPM".

    if (k == (length(CPM(1,:))))
        MVM = zeros(size(NVM, 1), k); 
        MVM(:,k) = eukMedCPM;
    else
        MVM = getMedianVectorMatrix(CPM, NVM, k+1, eukMedCPM);
        %following is equal to:
        %alpha = N_k-1*(P_k-P_k+1) / N_k-1*(M_k+1-P_k+1)
        numerator = (NVM(:,k-1))' * (CPM(:,k) - CPM(:,k+1));
        denominator = (NVM(:,k-1))' * (MVM(:,k+1) - CPM(:,k+1));
        alpha = numerator / denominator;
        MVM(:,k) = CPM(:,k+1) + alpha * (MVM(:,k+1) - CPM(:,k+1));
    end
    if (k == 2)
        MVM(:,1) = [];
    end
   
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [ gen_median, best_value, valid_result ] = dpe_get_gen_median( CPM, MVM, ...
    set, nearestUniquePointsIndices, weighted_mean_func, best_crit)
%GETGENMEDIANDPE associates a string to each point included in MVM, which are
%the intersection points with the simplices.

    valid_result = true;
    gen_median = NaN;
    num_med_strings = length(MVM(1,:));
    best_value = Inf;
    %gets the first of the median strings belonging to MVM-points
    percentage = dpe_distance_percentage(CPM(:,1), CPM(:,2), MVM(:,1));
    if ~isfinite(percentage) || (percentage > 1) || (percentage < 0)
        valid_result = false;
        best_value = NaN;
        return
    else
        [new_gen_median, best_value] = dpe_get_best_weighted_mean( ...
            set{nearestUniquePointsIndices(1)}, set{nearestUniquePointsIndices(2)}, ...
            percentage, weighted_mean_func, set, best_crit);
        gen_median = new_gen_median;

        %gets all of the other median strings belonging to MVM-points
        for i = 1:num_med_strings-1
            percentage = dpe_distance_percentage( ...
                MVM(:,i), CPM(:,i+2), MVM(:,i+1));
            if ((~isfinite(percentage)) || (percentage > 1) || (percentage < 0))
                valid_result = false;
                break;
            end
            [new_gen_median, new_best_value] = dpe_get_best_weighted_mean( new_gen_median,set{nearestUniquePointsIndices(i+2)}, ...
                percentage, weighted_mean_func, set, best_crit);
            
            if (new_best_value < best_value)
                best_value = new_best_value;
                gen_median = new_gen_median;
            end
            
        end
    end
    
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [ E ] = cleanupDims(E)
% CLEANUPDIMS performs a cleanup of the dimensions of a given set of
% points. This will be done by removing dimensions which contains only 
% constant values or which are linear dependence.
%
% E:        points as rows


	% if the rank is lower than the number of dimensions, so there are
    % dimensions which contains constant values
    if rank(E) < size(E, 2)
        % remove constant column vectors (use only one iteration)
        E = constVec(E);
    end
    
    % if the rank is still lower than the number of dimensions, so there
    % are dimensions which are linear dependence
    if rank(E) < size(E, 2)
        % extract linear independence column vectors
        E = lindepVec(E);
    end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



function [newE] = lindepVec(E)
% LINDEPVEC extracts all dimensions of an vector space, which are
% linear independence.
%
% E:    row vector space


    % get the number of columns of W, which corresponds to the dimensions
    % of the space
    Ncol = size(E, 2);

    % create a new vector space which only contains linear independence 
    % column vectors of W
    newE = zeros(size(E, 1), rank(E));
    % start with the first column of the new space
    j = 1;

    % for all columns of the input space
    for i = 1:Ncol
        % add the column to the new space
        newE(:, j) = E(:, i);
        % if all columns of P are linear independence
        if rank(newE) == j
            % break if the current column is the last column
            if j == size(newE, 2)
                break;
            end
            % append the next column
            j = j + 1;
        end
    end
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


function [E] = constVec(E)
% CONSTVEC removes all dimensions of the space E, which contains
% constant values in one dimensions of all vectors.
%
% E:    row vector space

    % stores if the dimension has to remove
    removeDim = false(1, size(E, 2));
    % for column vectors
    for i = 1:size(E, 2)
        % if the current dimension is not marked as to delete
        if ~removeDim(i)
            % get the value of the first item in the current column vector
            % and check which vectors contains only this value
            constDims = all(bsxfun(@eq, ones(1, size(E, 1)) ...
                                        * E(1, i), E'), 2);
            % mark the dimension which have to remove
            removeDim(constDims) = true;
        end
    end
    % remove the marked column vector
    E(:, removeDim) = [];
end
