% This is an implementation of metricmap published in 'An Index Structure
% for Data Mining and Clustering' (2000) by Wang.
%
% O:            list of cells which contains the items
% d:            distance function (O x O -> R)
% k:            destination dimension of the embedding
function [E] = metricmap(set, dist_func, k)
    % get the number of objects
    NItems = numel(set);

    % checks the dimension of the embedding space and sets the number of
    % reference items for preembedding
    if k > floor(NItems / 2)
        warning('k is to large. reduce k to the max embedding dimension');
        k = floor(NItems / 2);
    end

    %% Create an embedding space with dimension of 2*k-1

    % m >= k (i.e. 2k)
    m = 2 * k;

    % choose m items as reference set
    S = set(randperm(NItems, m));

    % gets the first item of the reference set as origin
    S0 = S{1};
    % remove the choosen origin from the reference set
    S(1) = [];
    % now there are only m - 1 objects in the reference set
    m = m - 1;

    % computes the distances between the items in the reference set with 
    % respect to the origin
    M = zeros(m,m);
    for i=1:m-1
        for j=i+1:m
            % compute the distances in the reference set
            M(i, j) = (dist_func(S{i}, S0)^2 + dist_func(S{j}, S0)^2 ...
                - dist_func(S{i}, S{j})^2) / 2;
            % distance is symmetrical
            M(j, i) = M(i, j);
        end
    end

    % calculate an orthogonal basis of M and the eigenvalues (Q*D*Q.'=M)
    [Q, D] = eig(M);
    % extract the diag of the eigenvalues
    D = diag(D);

    %% Now the embedding space will be reduced to k

    % get an descending ordering of the absolute eigenvalues
    [~, D_order] = sort(abs(D), 'descend');
    % order the eigenvalues
    D = D(D_order);
    % update the order of the reference set
    S = S(D_order);
    % update the order of the eigenvectors
    Q = Q(:, D_order);

    % Q is orthogonal, so the transponse of Q is equal to the inverse of Q
    Qinv = Q';
    % gets a k times k submatrix of Q
    Qinv_k = Qinv(1:k, 1:k);

    % gets the squareroot of the absolute values of D
    C = sqrt(abs(D(1:k)));
    % sets all zeros of the eigenvalues to 1
    C(C == 0) = 1;
    % create a diagonal matrix as C
    Csqrt_k = diag(C);

    %% Now the objects will be projected into the embedding space

    % Embedding space
    E = zeros(NItems, k);
    % Current projection of the current objects distance
    H_k = zeros(k, 1);
    % for all items of O
    for o=1:NItems
        % for all destination dimenSons
        for ki = 1:k
            % compute the projection on the hyperplanes
            H_k(ki)= (dist_func(S0, set{o}).^2 + dist_func(S0, S{ki}).^2 ...
                - dist_func(set{o}, S{ki}).^2) / 2;
        end

        % maps the object by E : U -> R^k
        E(o, :) =  Csqrt_k \ Qinv_k * H_k;
    end
end