% FASTMAP is an implementation of fastmap published in  'A Fast Algorithm for 
% Indexing, Data-Mining and Visualization of Traditional and Multimedia 
% Datasets' (1995) by Faloutsos and Lin.
%
% set:        list of cell which contains the items
% dist_func:  distance Function (set x set -> R)
% k:          destination dimension of the embedding
function [X, PA] = fastmap(set, dist_func, k)
    % number of objects
    N = numel(set);
    
    % projected objects in target space
    X = zeros(N, k);
    % pivots
    if nargout > 1
        PA = zeros(2, k);
    end
    
    % for all k spaces
    for col=1:k
        % creates the distance function for the current space (step 6)
        dist = @(a_id, b_id) recursive_distance(dist_func, set, X, a_id, b_id, col);
        
        % choose the pivots objects
        [a_id, b_id] = choose_distant_objects(set, dist);
        
        % stores the pivots
        if nargout > 1
            PA(1, col) = a_id;
            PA(2, col) = b_id;
        end
        
        % ensure that the distance between the pivots is higher than 0
		d_ab = dist(a_id, b_id);
        if d_ab == 0
            % remove all unnecessary dimensions
            X(:, col:k)=[];
            return;
        end

        % computes the current coordinate in the embedding
        for i_id = 1:N
            d_ai = dist(a_id, i_id);
            d_bi = dist(b_id, i_id);
            X(i_id, col) = (d_ai^2 + d_ab^2 - d_bi^2) / (2 * d_ab);
        end
    end         
end

% This function calculates the recursive distance.
% 
% dist_func:  distance-Function (set x set -> R)
% set:        list of cells which contains the items
% X:          images of the objects
% a_id:       id of the first object
% b_id:       id of the second object
% i:          number of used hyperplanes
function d_ab = recursive_distance(dist_func, set, X, a_id, b_id, i)
    % if i is less or equal to 1
    if i <= 1
        % then the distance function of the original space will be used
        delta_ab = dist_func(set{a_id}, set{b_id});
    else
        % else the distance will be recursive calculated for a hyperplane
        % and the previous
        delta_ab = (recursive_distance(dist_func, set, X, a_id, b_id, i - 1))^2 ...
                        - (X(a_id, i - 1) - X(b_id, i - 1))^2;
    end
    
    % checks the delta to prevents complex values
    if delta_ab >= 0
        d_ab = sqrt(delta_ab);
    else
        d_ab = -sqrt(-delta_ab);
    end
end

% This function gets to objects with heuristically maximum distance.
%
% set:        list of cells which contains the items
% dist_func:  distance Function (set x set -> R)
function [a_id, b_id] = choose_distant_objects(set, dist_func)
    N = size(set, 1);

    % choose a random start
    b_id = randi([1, N]);

    % gets the farthest objects of the start item as pivot
    a_id = choose_farthest_object(set, b_id, dist_func);
    
    % gets the farthest objects of the first pivot as second pivot
    b_id = choose_farthest_object(set, a_id, dist_func);
end

% This function gets the object with maximum distance to a given pivot.
% set:        list of cells which contains the items
% a_id:     id of the pivot
% dist_func:     distance Function (set x set -> R)
function [b_id] = choose_farthest_object(set, a_id, dist_func)
    N = numel(set);

    % id of farthest object from a
    b_id = a_id;
    % distance of object with id a to id b
    d_ab = 0;
    % for all object
    for j_id=1:N
        % only if the current object is a different one
        if j_id ~= a_id
            % calculate the distance from a to the current object
            d_ai = dist_func(a_id, j_id);
            % if the last calculated distance is larger then the previous
            if d_ai > d_ab
                % store the id of the object
                b_id = j_id;
                % store the distance
                d_ab = d_ai;
            end
        end
    end
end
    
