function som_gui(varargin)

%SOM_GUI A GUI for initialization and training of SOM.
%
% som_gui([sD]) 
% 
%   som_gui
%   som_gui(sD)
%
%  Input and output arguments ([]'s are optional) 
%   [sD]     (struct) SOM data struct
%            (matrix) a data matrix, size dlen x dim
%
%   Actually, there are more arguments the function takes, but 
%   they are for internal action of the function only. DO NOT use
%   them.
%
% For a more throughout description, see the online documentation.
% See also PREPROCESS.

%%%%%%%%%%%% DETAILED DESCRIPTION %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% IN FILES: som_gui.html,browsewin.jpg,wspace.jpg,loadgui.jpg,initgui.jpg,questdlg.jpg,paragui.jpg,mwindow.jpg,visgui.gif,reload.gif,savemap.gif,browse.gif
%

% Contributed to SOM Toolbox vs2, February 2nd, 2000 by Mika Pollari
% Copyright (c) by Mika Pollari and SOM Toolbox Team
% http://www.cis.hut.fi/projects/somtoolbox/

% Mika Pollari 31.1.2000 vs 1.1

global NEWMAP NEWST MAPSAVED MAP DATA  LOAD_NAME LOAD_DATA;
global SAVEMAP ALGORITHM HANDLE2 STOPOLINIT INIT_TYPE;
global STRAIN1 STRAIN2 SOTHERS;

if nargin == 0
  main_gui;
  action = 'dummy';	
elseif nargin == 1
  temp = varargin{1};
  if isstruct(temp),
    DATA = temp;
    main_gui;
    action = 'input_data';
  elseif isnumeric(temp), 
    DATA = som_data_struct(temp);
    main_gui;
    action = 'input_data';
  else
    action = temp;			
  end
end 


switch(action)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% LOAD %%%%%%%%%%%%%%%%%%%%%%%%%%
  case 'load_data'
     loadgui3;		%%% Activates load GUI
  case 'workspace'
       workspace;	%%% Workspace selected
  case 'file'
        file;  		%%% File Selected
  case 'file_select'
	file_select; 
  case 'missing'	
       Handle = findobj(gcf,'Tag','Checkbox1');
       set(Handle,'Value',1);	
  case 'load_ok'	%%% <Load OK> pushed
     	load_ok;	
  case 'input_data'	%%% GUI activated with data as arg1
     	input_data;     %%% eg. som_gui(data)

  case 'browse'		%%% Activates Browse GUI		
	browse;		%%% Browse files or workspace variables

  case 'works_ok'	%%% <OK> pushed in (workspace) browse GUI
	works_ok;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%% Initialization %%%%%%%%%%%%%%%%%%%%%%%%%%%%

  case 'def_initialization'	%%% Finds default initialization ...
	def_initialization;     %%% parameters

  case 'change_initialization'	%%% Activates change (init) parameters GUI    
	change_initialization;

  case 'change_initialization_ok'%%% Set new init. parameters 	
	change_initialization_ok;

  case 'change_initialization_cancel'
	close(gcf);
	return;

  case 'map_size' %%% Checks that 'map_size' is given in correct form
	map_size;

  case 'munits'   %%% Checks that 'munits' is given in correct form
	munits;
  
  case 'init'	%%% Initialize Map
	init;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%% Train %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


  case 'def_values_others'
	def_values_others;

  case 'def_values_train'
 	STRAIN1 = som_train_struct('algorithm',ALGORITHM,'phase','rough','data',DATA);
	STRAIN2 = som_train_struct('previous',STRAIN1);

  case 'fill_fields'	%%% Fill text fields in GUI
	fill_fields;

  case 'def_train'	%%% Train Map
     	def_train;

  case 'change_def'	%%% Change default training parameters 
	change_def;	%%% Activate GUI

  case 'fill_new_defaults' 
	fill_new_defaults;

  case 'set_batch_mask'
	set_batch_mask;

  case 'set_new_parameters'
	set_new_parameters;

  case 'only_finetune' 	%%% Train only once with finetune parameters
	only_finetune;

%%%%%%% Next function check correctnes of new training parameters.

  case 'check_rough_radini'
	check_rough_radini;
  case 'check_fine_radini'
	check_fine_radini;
  case 'check_rough_radfin'
	check_rough_radfin;
  case 'check_fine_radfin'
	check_fine_radfin;
  case 'check_rough_alphaini'
	check_rough_alphaini;
  case 'check_fine_alphaini'
	check_fine_alphaini;
  case 'check_rough_trainlen'
	check_rough_trainlen;
  case 'check_fine_trainlen'
	check_fine_trainlen;



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%% Save Map %%%%%%%%%%%%%%%%%%%%%%

  case 'savemap'	%%% Save as <.cod> file
	savemap;
  case 'save_workspace'	%%% Save in workspace
	save_workspace;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%% Help & Info %%%%%%%%%%%%%%%%%%%%%%%
  case 'help'
    web file:///share/somtoolbox/vs2/html/som_GUI.html;
  case 'helpwin'
    helpwin1;
  case 'helpwin2'
    helpwin som_gui;	
  case 'data_info'
	data_info;   %%% Info about data
  case 'map_info'    %%% Info about map
	map_info;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%% Other Functions %%%%%%%%%%%%%%%%%%%%%%%

  case 'preprocess'
	preprocess_gui;		%%%%% Call preprocess GUI
  case 'visualize'
	visualize;		%%%%% Call visualization GUI

  case 'clear_all'		%%%%% Clear all filds
	clear_all;		

  case 'close'
	close_fig;		%%%%% Close active GUI 	

end 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%   END OF SWITCH-STATEMENT  %%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%















%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%% (SUB) FUNCTIONS




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     LOAD SECTION STARTS	%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	

function [] = workspace()
Handle = findobj(gcbf,'Tag','Radiobutton2');
Value = get(Handle,'Value');
HandleTemp = findobj(gcbf,'Tag','Radiobutton1');
if Value == 1	
  set(HandleTemp,'Value',0);
  HandleBar = findobj(gcbf,'Tag','PopupMenu1');
  set(HandleBar,'Enable','off');
  set(HandleBar,'Visible','off');
  Handle3 = findobj(gcbf,'Tag','StaticText3');
  set(Handle3,'Visible','off');
  Handle3 = findobj(gcbf,'Tag','Checkbox1');
  set(Handle3,'Visible','off');
  Handle3 = findobj(gcbf,'Tag','EditText3');
  set(Handle3,'Visible','off');
  Handle = findobj(gcbf,'Tag','EditText2');
  set(Handle,'String','');
end



function [] = file()
Handle = findobj(gcbf,'Tag','Radiobutton1');
Value = get(Handle,'Value');
HandleTemp = findobj(gcbf,'Tag','Radiobutton2');
if Value == 1
  set(HandleTemp,'Value',0);
  HandleBar = findobj(gcbf,'Tag','PopupMenu1');
  set(HandleBar,'Enable','on');
  set(HandleBar,'Visible','on');
  Handle3 = findobj(gcbf,'Tag','StaticText3');
  set(Handle3,'Visible','on');
  Handle3 = findobj(gcbf,'Tag','Checkbox1');
  set(Handle3,'Visible','on');
  Handle3 = findobj(gcbf,'Tag','EditText3');
  set(Handle3,'Visible','on'); 
  Handle = findobj(gcbf,'Tag','EditText1');
  set(Handle,'String','');
end

function [] = file_select() 
Handle = findobj(gcbf,'Tag','PopupMenu1');
temp = get(Handle,'String');
val = get(Handle,'Value');
Handle1 = findobj(gcbf,'Tag','Checkbox1');
Handle2 = findobj(gcbf,'Tag','EditText3');
if strcmp(temp{val},'dat file')
   set(Handle2,'String','x');
   set(Handle1,'Enable','on');
   set(Handle2,'Enable','on');
   set(Handle1,'Visible','on');
   set(Handle2,'Visible','on');
  else 
   set(Handle1,'Value',0);
   set(Handle1,'Enable','off');
   set(Handle2,'Enable','off');
   set(Handle1,'Visible','off');
   set(Handle2,'Visible','off');
end

function [] = load_ok()
global MAP DATA LOAD_DATA LOAD_NAME;
Handle1 = findobj(gcbf,'Tag','EditText1');
Handle2 = findobj(gcbf,'Tag','EditText2');
Name1 = get(Handle1,'String');
Name2 = get(Handle2,'String');
if isempty(Name1) && not(isempty(Name2))
  Handle = findobj(gcbf,'Tag','PopupMenu1')
  type = get(Handle,'String');
  val = get(Handle,'Value');
  type = type{val};
  if strcmp(type,'mat file')
      ltemp = 'load:::';
      ltemp = strcat(ltemp,Name2);
      ltemp = strrep(ltemp,':::',' ');
      evalin('base',ltemp);
      DATA = evalin('base','sD');
      LOAD_DATA = evalin('base','sD.data');
      LOAD_NAME = evalin('base','sD.name');
      LOAD_NAME = strrep(LOAD_NAME,'.','_');
      load_labels = evalin('base','sD.labels');
      load_comp_names = evalin('base','sD.comp_names');
      DATA = som_data_struct(LOAD_DATA);
      DATA.name = LOAD_NAME;
      DATA.comp_names = load_comp_names;
      DATA.labels = load_labels;
   else
      Handle = findobj(gcbf,'Tag','Checkbox1');
      value = get(Handle,'Value');
      if value == 0	
         temp = 'som_read_data(''';
         temp = strcat(temp,Name2,''');');
      else 
          Handle = findobj(gcbf,'Tag','EditText3');
          missing = get(Handle,'String');
          if not(isempty(missing))
              temp = 'som_read_data(''';
              temp = strcat(temp,Name2,'''',',','''',missing,''');');
          else
	      temp = 'som_read_data(''';
              temp = strcat(temp,Name2,''');');
          end   
      end
     evalin('base',temp);
     DATA = evalin('base','ans');
     name = DATA.name;
     temp = findstr('/',name);
     if not(isempty(temp))	
         name = name(temp(end)+1:end);
     end
     name = strrep(name,'.','_');		
     LOAD_NAME = name;
     DATA.name = name;	
   end
elseif isempty(Name2) && not(isempty(Name1))
   LOAD_DATA = evalin('base',Name1);
   if not(isstruct(LOAD_DATA))
      DATA = som_data_struct(LOAD_DATA);
      LOAD_NAME = Name1;
      DATA.name = Name1;
   else
      DATA = LOAD_DATA;
      name = DATA.name;
      temp = findstr('/',name);
      if not(isempty(temp))	
	 name = name(temp(end)+1:end);
      end
      name = strrep(name,'.','_');		
      LOAD_NAME = name;
      DATA.name = name;
   end
else 
   errmsg = {'Give name of data before loading'};
   errordlg(errmsg,'Empty data name!');
   return;
end
close(gcbf);			     			
if not(isempty(MAP))
   clear MAP;
   global MAP;
   str1 = 'Map:  <empty>';
   str2 = 'Train';
   Handle = findobj(gcf,'Tag','StaticText3');
   set(Handle,'String',str1);
   Handle = findobj(gcf,'Tag','StaticText8');
   set(Handle,'String',str2);
end	
temp =  'Data:';
temp = strcat(temp,' <',LOAD_NAME,'>');	
Handle = findobj(gcf,'Tag','StaticText4');
set(Handle,'String',temp);
som_gui('def_initialization');
Handle = findobj(gcf,'Tag','Pushbutton2');
set(Handle,'Enable','off');
Handle = findobj(gcf,'Tag','Pushbutton4');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','Pushbutton9');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','Subuimenu2');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','&Help/InfoHelp windowuimenu1');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','&Init&Trainuimenu1');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','&Init&TrainInitialize1');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','Subuimenu1');
set(Handle,'Enable','off'); %%%%%%????????	
Handle = findobj(gcf,'Tag','StaticText10');
set(Handle,'String','Status <data loaded>');	
 


function [] = input_data()
global DATA; 
name = DATA.name;
newname = strrep(name,'.','_');
DATA.name = newname;
temp = strcat('Data:  <',newname,'>');	
Handle = findobj(gcf,'Tag','StaticText4');
set(Handle,'String',temp);
som_gui('def_initialization');
Handle = findobj(gcf,'Tag','Pushbutton2');
set(Handle,'Enable','off');
Handle = findobj(gcf,'Tag','Pushbutton4');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','Pushbutton9');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','Subuimenu2');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','&Help/InfoHelp windowuimenu1');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','&Init&Trainuimenu1');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','&Init&TrainInitialize1');
set(Handle,'Enable','on');
Handle = findobj(gcf,'Tag','Subuimenu1');
set(Handle,'Enable','off'); %%%%%%????????	
Handle = findobj(gcf,'Tag','StaticText10');
set(Handle,'String','Status <data loaded>');	


function [] = browse()
global HANDLE2;
HandleWorkspace = findobj(gcbf,'Tag','Radiobutton2');
HandleFile = findobj(gcbf,'Tag','Radiobutton1');
WorkspaceVal = get(HandleWorkspace,'Value');
FileVal = get(HandleFile,'Value');
if FileVal == 1
   Handle = findobj(gcbf,'Tag','PopupMenu1');
   str = get(Handle,'String');
   value = get(Handle,'Value');
   str = str{value};
   if strcmp(str,'mat file')
      filtter = '*.mat';
   else
      filtter = '*.dat*';
   end 
   [filename pathname] = uigetfile(filtter,'Load file.');
   temp = strcat(pathname,filename);
   Handle = findobj(gcbf,'Tag','EditText2');
   set(Handle,'String',temp);
elseif WorkspaceVal == 1
   HANDLE2 = gcf;
   works;
   temp = evalin('base','who');
   index2 = 1;
   names = '';
   for index = 1:length(temp)
      if isnumeric(evalin('base',temp{index}))
	test = size(evalin('base',temp{index}));
	if test(1) ~= 1 && test(2) ~= 1
   	   names{index2} = temp{index};	
	   index2 = index2 + 1;
	end
      end
   end
   for index = 1:length(temp)
	variable = evalin('base',temp{index});
	if isstruct(variable)
	   fnames = fieldnames(variable);
	   if size(fnames,1) == 6 && strcmp(fnames(1),'type') && strcmp(variable.type,'som_data')
	     names{index2} = temp{index};	
	     index2 = index2 + 1;
	   end
	end 
   end	
   Handle = findobj(gcf,'Tag','Listbox1');
   %%%%%% if is empty string#%%%
   set(Handle,'String',names);
else
   errmsg = 'Select browse type: Workspace or file.';
   errordlg(errmsg,'Browse error!');
   return;
end


function [] = works_ok()
global HANDLE2;
Handle = findobj(gcbf,'Tag','Listbox1');
temp = get(Handle,'String');
val = get(Handle,'Value');
data = temp{val};
Handle = findobj(HANDLE2,'Tag','EditText1');
set(Handle,'String',data);
close;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     END OF LOAD SECTION	%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     START OF INITIALIZATION	%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	

function [] = def_initialization()
global DATA STOPOLINIT INIT_TYPE;
sTopol = som_topol_struct('data',DATA);
Handle = findobj(gcf,'Tag','StaticText5');
temp = num2str(sTopol.msize);
temp = strcat('map size:',' [',temp,']');
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText6');
set(Handle,'String','type: linear');
Handle = findobj(gcf,'Tag','StaticText20');
temp = strcat('lattice:',sTopol.lattice);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText21');
temp = strcat('shape:',sTopol.shape);
set(Handle,'String',temp);
STOPOLINIT = sTopol;
INIT_TYPE = 'linear';

function [] = change_initialization()
global INIT_TYPE STOPOLINIT; 
initialization2;
Handle = findobj(gcf,'Tag','PopupMenu1');
temp = get(Handle,'String');
val = loop(temp,INIT_TYPE);
set(Handle,'Value',val);
Handle = findobj(gcf,'Tag','PopupMenu2');
temp = get(Handle,'String');
val = loop(temp,STOPOLINIT.lattice);
set(Handle,'Value',val);
Handle = findobj(gcf,'Tag','PopupMenu3');
temp = get(Handle,'String');
val = loop(temp,STOPOLINIT.shape);
set(Handle,'Value',val);
Handle = findobj(gcf,'Tag','EditText1');
temp = num2str(STOPOLINIT.msize);
msize = strcat('[',temp,']');
set(Handle,'String',msize);

function [] = change_initialization_ok()
Handle = findobj(gcbf,'Tag','PopupMenu1');
temp = get(Handle,'String');
val = get(Handle,'Value');
INIT_TYPE = temp{val};
Handle = findobj(gcbf,'Tag','PopupMenu2');
temp = get(Handle,'String');
val = get(Handle,'Value');
lattice = temp{val};
Handle = findobj(gcbf,'Tag','PopupMenu3');
temp = get(Handle,'String');
val = get(Handle,'Value');
shape = temp{val};
Handle = findobj(gcbf,'Tag','EditText1');
temp = get(Handle,'String');
msize = str2num(temp);
STOPOLINIT = som_set('som_topol','msize',msize,'lattice',lattice,'shape',shape);
close(gcf);
Handle = findobj(gcf,'Tag','StaticText5');
temp = num2str(STOPOLINIT.msize);
temp = strcat('map size:',' [',temp,']');
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText6');
temp = strcat('type:',INIT_TYPE);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText20');
temp = strcat('lattice:',STOPOLINIT.lattice);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText21');
temp = strcat('shape:',STOPOLINIT.shape);
set(Handle,'String',temp);


function [] = def_values_others()
global SOTHERS;
Handle = findobj(gcf,'Tag','StaticText19');
temp = strcat('tracking:',SOTHERS.tracking);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText12');
temp = strcat('order:',SOTHERS.oder);
set(Handle,'String',temp); 
Handle = findobj(gcf,'Tag','StaticText14');
temp = strcat('length_type:',SOTHERS.length_type);
set(Handle,'String',temp);



function [] = fill_fields()
global STRAIN1 STRAIN2 ALGORITHM

neigh = STRAIN1.neigh;
mask = STRAIN1.mask;
rad_ini1 = STRAIN1.radius_ini;
rad_ini2 = STRAIN2.radius_ini;
rad_fin1 = STRAIN1.radius_fin;
rad_fin2 = STRAIN2.radius_fin;
trainlen1 = num2str(STRAIN1.trainlen);
trainlen2 = num2str(STRAIN2.trainlen);
alpha_ini1 = num2str(STRAIN1.alpha_ini);
alpha_ini2 = num2str(STRAIN2.alpha_ini);
if strcmp(ALGORITHM,'seq')
        alpha_type = STRAIN1.alpha_type; %%% only in sequential
        Handle = findobj(gcf,'Tag','StaticText28');
        temp = strcat('alpha type:',alpha_type);
        set(Handle,'String',temp);
end
Handle = findobj(gcf,'Tag','StaticText11');
temp = strcat('neigh: ',neigh);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText22');
temp = num2str(rad_fin1);
temp = strcat('radius final:',temp);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText25');
temp = num2str(rad_fin2);
temp = strcat('radius final:',temp);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText11');
temp = strcat('neigh: ',neigh);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText17');
temp = num2str(rad_ini1);
temp = strcat('radius initial:',temp);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText24');
temp = num2str(rad_ini2);
temp = strcat('radius initial:',temp);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText16');
temp = num2str(trainlen1);
temp = strcat('training length:',temp);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText23');
temp = num2str(trainlen2);
temp = strcat('training length:',temp);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText26');
temp = strcat('alpha initial:',alpha_ini1);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText27');
temp = strcat('alpha initial:',alpha_ini2);
set(Handle,'String',temp);


function [] = init()
global INIT_TYPE MAP NEWMAP ALGORITHM SOTHERS DATA STOPOLINIT;
if strcmp(INIT_TYPE,'random')
   MAP = som_randinit(DATA,STOPOLINIT);
else 
   MAP = som_lininit(DATA,STOPOLINIT);
end
NEWMAP = MAP;	
temp = 'Map:';
temp = strcat(temp,' <',MAP.name,'>');
Handle = findobj(gcbf,'Tag','StaticText3');
set(Handle,'String',temp);
Handle = findobj(gcbf,'Tag','StaticText10');
set(Handle,'String','Status <map initialized>');
ALGORITHM = 'batch';
Handle = findobj(gcbf,'Tag','Pushbutton4');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','Pushbutton6');
set(Handle,'Enable','on');
Handle = findobj(gcbf,'Tag','Pushbutton5');
set(Handle,'Enable','on');		
SOTHERS.tracking = '1';
SOTHERS.length_type = 'epochs';
SOTHERS.oder = 'random';			
som_gui('def_values_topol');
som_gui('def_values_train');
som_gui('def_values_others');
som_gui('fill_fields');
Handle = findobj(gcbf,'Tag','Pushbutton4');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','Pushbutton9');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','Radiobutton1');
set(Handle,'Enable','on');
Handle = findobj(gcbf,'Tag','&Init&TrainChange initialization valuesuimenu1');
set(Handle,'Enable','on');
Handle = findobj(gcbf,'Tag','&Init&TrainTrain1');
set(Handle,'Enable','on');
Handle = findobj(gcbf,'Tag','&Help/InfoData infouimenu1');
set(Handle,'Enable','on');	
Handle = findobj(gcbf,'Tag','Subuimenu2');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','&Init&Trainuimenu1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','&Init&TrainInitialize1');
set(Handle,'Enable','off');  %%%%%%%%%%%??????????			
Handle = findobj(gcbf,'Tag','StaticText9');
set(Handle,'String','training type: batch');	



function [] = set_batch_mask()
Handle = findobj(gcbf,'Tag','Listbox2');
temp = get(Handle,'String');
mask = str2num(temp);
Handle = findobj(gcbf,'Tag','Listbox1');
replace = get(Handle,'Value');
Handle = findobj(gcbf,'Tag','EditText2');
temp = get(Handle,'String');
value = str2num(temp);
if not(isempty(value))
  mask(replace) = value;
  Handle = findobj(gcbf,'Tag','Listbox2');
  temp = num2str(mask);
  set(Handle,'String',temp);
end

function [] = munits()
global DATA STOPOLINIT;
msgs = {'Correct map units is number';'Correct map units is number'};
[msgs_nro, value] = check_ok('EditText2');
if msgs_nro > 0
   errordlg({msgs{msgs_nro}},'Incorrect map units!')
   return;
end
STOPOLINIT = som_topol_struct('munits',value,'data',DATA);
Handle = findobj(gcbf,'Tag','EditText1');
temp = num2str(STOPOLINIT.msize);
msize = strcat('[',temp,']');
set(Handle,'String',msize);

function [] = map_size()
 global STOPOLINIT;
 msgs = {'Map size must be in form [x y]';...
		'Map size must be in form [x y]'};
 [msgs_nro, value, Handle] = msize_ok('EditText1');
 if msgs_nro > 0
   errordlg({msgs{msgs_nro}},'Incorrect map size!');
   temp = num2str(STOPOLINIT.msize);
   temp = strcat('[',temp,']');
   set(Handle,'String',temp);	
   return;
 end
 STOPOLINIT.msize = value;
 Handle = findobj(gcbf,'Tag','EditText2');
 set(Handle,'String','');


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     END OF INITIALIZATION	%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     START OF TRAINING		%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	


function  [] = def_train()
global SOTHERS ALGORITHM MAP NEWST DATA STRAIN1 STRAIN2 MAPSAVED;
tlen_type = SOTHERS.length_type;
sample_order = SOTHERS.oder;	
tracking = SOTHERS.tracking;
test = str2num(tracking);
Handle = findobj(gcbf,'Tag','Radiobutton1');
tempval = get(Handle,'Value');
if strcmp(ALGORITHM,'seq')
    if tempval ~= 1
       [MAP NEWST] = som_seqtrain(MAP,DATA,'train',STRAIN1,tlen_type,sample_order);
    end
    if test > 1
	figure;
	set(gcf,'Name',MAP.name);
	 set(gcf,'NumberTitle','off');
    end
    [NEWMAP NEWST] = som_seqtrain(MAP,DATA,'train',STRAIN2,'tracking',test,tlen_type,sample_order); 
else
    if tempval ~= 1
       [MAP NEWST] = som_batchtrain(MAP,DATA,'train',STRAIN1);
    end
    if test > 1
       figure;
       set(gcf,'Name',MAP.name);
       set(gcf,'NumberTitle','off');
    end
        [NEWMAP NEWST] = som_batchtrain(MAP,DATA,'train',STRAIN2,'tracking',test);
end
MAP = NEWMAP;
clear MAPSAVED;	
Handle = findobj(gcbf,'Tag','StaticText10');
set(Handle,'String','Status <map trained>');
Handle = findobj(gcbf,'Tag','Load/SaveSubuimenu1');
set(Handle,'Enable','on');		
Handle = findobj(gcbf,'Tag','Load/SaveSave mapuimenu1');
set(Handle,'Enable','on');	 
Handle = findobj(gcbf,'Tag','&Load/SaveSave mapSave in workspaceuimenu1');
set(Handle,'Enable','on');
Handle = findobj(gcbf,'Tag','&ToolsSubuimenu1');
set(Handle,'Enable','on');
Handle = findobj(gcbf,'Tag','&Init&TrainChange initialization valuesuimenu1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','&Init&TrainTrain1');
set(Handle,'Enable','off');		


function [] = change_def()
global ALGORITHM STRAIN1 DATA; 

ButtonName = questdlg('Select training type!',...
	'Change values.',...
	'Batch','Sequential','Cancel',...
        'Batch');	
if strcmp(ButtonName,'Sequential')
   Handle = findobj(gcbf,'Visible','off');
   set(Handle,'Visible','on');
   ALGORITHM = 'seq';
   Handle = findobj(gcf,'Tag','StaticText9');
   set(Handle,'String','training type: sequential');
   new_para2_2;
   Handle = findobj(gcf,'Tag','StaticText1');
   set(Handle,'String','Change parameters for sequential training');
   Handle = findobj(gcf,'Enable','off');
   set(Handle,'Enable','on');
   Handle = findobj(gcf,'Visible','off');
   set(Handle,'Visible','on');
elseif strcmp(ButtonName,'Batch')
   ALGORITHM = 'batch';
   Handle = findobj(gcbf,'Tag','StaticText26');
   set(Handle,'Visible','off');
   Handle = findobj(gcbf,'Tag','StaticText27');
   set(Handle,'Visible','off');
   Handle = findobj(gcf,'Tag','StaticText9');
   set(Handle,'String','training type: batch');
   Handle = findobj(gcf,'Tag','StaticText12');
   set(Handle,'Visible','off');
   Handle = findobj(gcf,'Tag','StaticText28');
   set(Handle,'Visible','off');
   Handle = findobj(gcf,'Tag','StaticText14');
   set(Handle,'Visible','off');
   new_para2_2;	
   Handle = findobj(gcf,'Tag','StaticText1');
   set(Handle,'String','Change parameters for batch training');
   Handle = findobj(gcf,'Tag','PopupMenu3');
   set(Handle,'Enable','off');
   set(Handle,'Visible','off');
   Handle = findobj(gcf,'Tag','PopupMenu4');
   set(Handle,'Enable','off');
   set(Handle,'Visible','off');
   Handle = findobj(gcf,'Tag','PopupMenu5');
   set(Handle,'Enable','off');
   set(Handle,'Visible','off');
   Handle = findobj(gcf,'Tag','StaticText17');
   set(Handle,'Visible','off');
   Handle = findobj(gcf,'Tag','StaticText18');
   set(Handle,'Visible','off');
   Handle = findobj(gcf,'Tag','StaticText19');
   set(Handle,'Visible','off');
   Handle = findobj(gcf,'Tag','StaticText13');
   set(Handle,'Visible','off'); 
   Handle = findobj(gcf,'Tag','StaticText14');
   set(Handle,'Visible','off');
   Handle = findobj(gcf,'Tag','EditText6');
   set(Handle,'Visible','off');
   set(Handle,'Enable','off');
   Handle = findobj(gcf,'Tag','EditText10');
   set(Handle,'Visible','off');
   set(Handle,'Enable','off');    
else 
   return;	
end
som_gui('def_values_train');	
mask = STRAIN1.mask;
Handle = findobj(gcf,'Tag','Listbox1');
set(Handle,'String',DATA.comp_names);
som_gui('fill_new_defaults');


function [] = fill_new_defaults()
global STRAIN1 STRAIN2 SOTHERS ALGORITHM;

Handle = findobj(gcf,'Tag','EditText4');
temp = num2str(STRAIN1.radius_ini);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','EditText8');
temp = num2str(STRAIN2.radius_ini);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','EditText5');
temp = num2str(STRAIN1.radius_fin);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','EditText9');
temp = num2str(STRAIN2.radius_fin);
set(Handle,'String',temp);	
Handle = findobj(gcf,'Tag','EditText6');
temp = num2str(STRAIN1.alpha_ini);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','EditText10');
temp = num2str(STRAIN2.alpha_ini);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','EditText7');
temp = num2str(STRAIN1.trainlen);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','EditText11');
temp = num2str(STRAIN2.trainlen);
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','Listbox2');
temp = num2str(STRAIN1.mask');
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','PopupMenu2');
string = get(Handle,'String');
val = loop(string,SOTHERS.tracking);
set(Handle,'Value',val);
Handle = findobj(gcf,'Tag','PopupMenu1');
string = get(Handle,'String');
val = loop(string,STRAIN1.neigh);
set(Handle,'Value',val);
if strcmp(ALGORITHM,'seq')
   Handle = findobj(gcf,'Tag','PopupMenu3');
   string = get(Handle,'String');
   val = loop(string,SOTHERS.length_type);
   set(Handle,'Value',val);
   Handle = findobj(gcf,'Tag','PopupMenu4');
   string = get(Handle,'String');
   val = loop(string,SOTHERS.oder);
   set(Handle,'Value',val);
   Handle = findobj(gcf,'Tag','PopupMenu5');
   string = get(Handle,'String');
   val = loop(string,STRAIN1.alpha_type);
   set(Handle,'Value',val);
end			



function [] = set_new_parameters()
global STRAIN1 STRAIN2 ALGORITHM SOTHERS; 

Handle = findobj(gcbf,'Tag','Listbox2');
temp = get(Handle,'String');
mask = str2num(temp); %%%%%%%%%%%%% Do somthing
mask = mask';
Handle = findobj(gcbf,'Tag','PopupMenu1');
temp = get(Handle,'String');
val = get(Handle,'Value');
neigh = temp{val};
Handle = findobj(gcbf,'Tag','PopupMenu2');
temp = get(Handle,'String');
val = get(Handle,'Value');
SOTHERS.tracking = temp{val};  %%%%% finetune phase!
Handle = findobj(gcbf,'Tag','EditText4');
temp = get(Handle,'String');
rad_ini1 = str2num(temp);
Handle = findobj(gcbf,'Tag','EditText8');
temp = get(Handle,'String');
rad_ini2 = str2num(temp);
Handle = findobj(gcbf,'Tag','EditText5');
temp = get(Handle,'String');
rad_fin1 = str2num(temp);
Handle = findobj(gcbf,'Tag','EditText9');
temp = get(Handle,'String');
rad_fin2 = str2num(temp);
Handle = findobj(gcbf,'Tag','EditText6');
temp = get(Handle,'String');
alpha_ini1 = str2num(temp);
Handle = findobj(gcbf,'Tag','EditText10');
temp = get(Handle,'String');
alpha_ini2 = str2num(temp);
Handle = findobj(gcbf,'Tag','EditText7');
temp = get(Handle,'String');
train_length1 = str2num(temp);
Handle = findobj(gcbf,'Tag','EditText11');
temp = get(Handle,'String');
train_length2 = str2num(temp);
if strcmp(ALGORITHM,'seq')
   Handle = findobj(gcbf,'Tag','PopupMenu3');
   temp = get(Handle,'String');
   val = get(Handle,'Value');
   SOTHERS.length_type = temp{val};
   Handle = findobj(gcbf,'Tag','PopupMenu4');
   temp = get(Handle,'String');
   val = get(Handle,'Value');
   SOTHERS.oder= temp{val};
   Handle = findobj(gcbf,'Tag','PopupMenu5');
   temp = get(Handle,'String');
   val = get(Handle,'Value');
   alpha_type = temp{val};
else 
   alpha_type = 'inv';
end
STRAIN1.neigh = neigh;
STRAIN2.neigh = neigh;
STRAIN1.mask = mask;
STRAIN2.mask = mask;
STRAIN1.radius_ini = rad_ini1;
STRAIN2.radius_ini = rad_ini2;
STRAIN1.radius_fin = rad_fin1;
STRAIN2.radius_fin = rad_fin2;
STRAIN1.alpha_ini = alpha_ini1;
STRAIN2.alpha_ini = alpha_ini2;
STRAIN1.alpha_type = alpha_type;
STRAIN2.alpha_type = alpha_type;
STRAIN1.trainlen = train_length1;
STRAIN2.trainlen = train_length2;
close(gcbf);
som_gui('fill_fields');
som_gui('def_values_others');

function [] = only_finetune()

Handle = findobj(gcbf,'Tag','Radiobutton1');
test = get(Handle,'Value');
if test == 1
   Handle = findobj(gcbf,'Tag','StaticText16');
   set(Handle,'Enable','off');
   Handle = findobj(gcbf,'Tag','StaticText17');
   set(Handle,'Enable','off');
   Handle = findobj(gcbf,'Tag','StaticText22');
   set(Handle,'Enable','off');
   Handle = findobj(gcbf,'Tag','StaticText26');
   set(Handle,'Enable','off');
else 
   Handle = findobj(gcbf,'Tag','StaticText16');
   set(Handle,'Enable','on');
   Handle = findobj(gcbf,'Tag','StaticText17');
   set(Handle,'Enable','on');
   Handle = findobj(gcbf,'Tag','StaticText22');
   set(Handle,'Enable','on');
   Handle = findobj(gcbf,'Tag','StaticText26');
   set(Handle,'Enable','on');
end


function [] = check_rough_radini()
global STRAIN1;
msgs = {'Initial radius must be number!';...
        'Initial radius must be single valued number!'};
[msgs_nro, value, Handle] = check_ok('EditText4');
if msgs_nro > 0
   errordlg({msgs{msgs_nro}},'Incorrect initial radius!')
   temp = num2str(STRAIN1.radius_ini);
   set(Handle,'String',temp);
   return;
end


function [] = check_fine_radini()
global STRAIN2;
msgs = {'Initial radius must be number!';...
        'Initial radius must be single valued number!'};
[msgs_nro, value, Handle] = check_ok('EditText8');
if msgs_nro > 0
   errordlg({msgs{msgs_nro}},'Incorrect initial radius!')
   temp = num2str(STRAIN2.radius_ini);
   set(Handle,'String',temp);
   return;
end

function [] = check_rough_radfin()
global STRAIN1;
msgs = {'Final radius must be number!';...
	'Final radius must be single valued number!'};
[msgs_nro, value, Handle] = check_ok('EditText5');
if msgs_nro > 0
   errordlg({msgs{msgs_nro}},'Incorrect final radius!')
   temp = num2str(STRAIN1.radius_fin);
   set(Handle,'String',temp);
   return;
end

function [] = check_fine_radfin()
global STRAIN2;
msgs = {'Final radius must be number!';...
	'Final radius must be single valued number!'};
[msgs_nro, value, Handle] = check_ok('EditText9');
if msgs_nro > 0
   errordlg({msgs{msgs_nro}},'Incorrect final radius!')
   temp = num2str(STRAIN2.radius_fin);
   set(Handle,'String',temp);
   return;
end

function [] = check_rough_alphaini()
global STRAIN1;	
msgs = {'Alpha initial must be number!';...
	'Alpha initial must be single valued number!'};
[msgs_nro, value, Handle] = check_ok('EditText6');
if msgs_nro > 0
   errordlg({msgs{msgs_nro}},'Incorrect initial alpha!')
   temp = num2str(STRAIN1.alpha_ini);
   set(Handle,'String',temp);
   return;
end

function [] = check_fine_alphaini()
global STRAIN2;
msgs = {'Alpha initial must be number!';...
	'Alpha initial must be single valued number!'};
[msgs_nro, value, Handle] = check_ok('EditText10');
if msgs_nro > 0
   errordlg({msgs{msgs_nro}},'Incorrect initial alpha!')
   temp = num2str(STRAIN2.alpha_ini);
   set(Handle,'String',temp);
   return;
end

function [] = check_rough_trainlen()
global STRAIN1;
msgs = {'Training length must be number!';...
	'Training length must be single valued number!'};
[msgs_nro, value, Handle] = check_ok('EditText7');
if msgs_nro > 0
   errordlg({msgs{msgs_nro}},'Incorrect training length!')
   temp = num2str(STRAIN1.trainlen);
   set(Handle,'String',temp);
   return;
end

function [] = check_fine_trainlen()
global STRAIN2;

msgs = {'Training length must be number!';...
	'Training length must be single valued number!'};
[msgs_nro, value, Handle] = check_ok('EditText11');
if msgs_nro > 0
   errordlg({msgs{msgs_nro}},'Incorrect training length!')
   temp = num2str(STRAIN2.trainlen);
   set(Handle,'String',temp);
   return;
end	

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     END OF TRAINING		%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     START OF SAVING		%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	


function [] = savemap()
global MAP MAPSAVED;
if isempty(MAP)
  str = {'There is no map to be saved! Train map before saving.'};
  helpdlg(str,'Empty map!');	
  return;
end
[FileName Path] = uiputfile('*.cod','Save file!');
if FileName ~= 0
   temp = strcat(Path,FileName);
   som_write_cod(MAP,temp);
   MAPSAVED = 'SAVED';
end
Handle = findobj(gcf,'Tag','StaticText10');
set(Handle,'String','Status <map saved>');	 

function [] = save_workspace()
global MAP MAPSAVED;
if isempty(MAP)
   str = {'There is no map to be saved! Train map before saving.'};
   helpdlg(str,'Empty map!');
   return;
else
   prompt = {'Save map as?'};
   title = 'Save map!';
   lineNo = 1;
   answer = inputdlg(prompt,title,lineNo);
   if isempty(answer)
      return;
   end
   if not(isempty(answer{1}))
      ws_variable = evalin('base','who');
      max_length = 0;
      for index = 1:size(ws_variable,1)
         if max_length < size(ws_variable{index},2)
	    max_length = size(ws_variable{index},2);
         end
      end
      length = max_length + 1;
      tempfoo(1:1:length) = 'A'; 
      assignin('base',tempfoo,answer{1});
      str = ['exist(' tempfoo ')'];	
      temp = evalin('base',str); %%%%%%%%%%@@@@@@@@@
      evalin('base',['clear ' tempfoo ]) 
      if temp == 0
         assignin('base',answer{1},MAP);
         MAPSAVED = 'SAVED';
      elseif temp ~= 0
         Questmsg = strcat('Variable',' ''',answer{1},'''',...
         ' exist. Overwrite?');
         ButtonName = questdlg(Questmsg);
         switch(ButtonName)
         case 'Yes'
            assignin('base',answer{1},MAP);
            MAPSAVED = 'SAVED';
         case 'No' 
             som_gui('save_workspace');		
         end
     end	
   else
      helpmsg = {'There cannot be any empty field in ''save'''};
      helpdlg(helpmsg,'Help Save!');
      som_gui('save'); 
   end
end
Handle = findobj(gcf,'Tag','StaticText10');
set(Handle,'String','Status <map saved>');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     END OF SAVING		%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     START OF HELP & INFO	%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	

%HEREXX

function [] = data_info()
global DATA;
if isempty(DATA)
   helpmsg = 'Load data first!';
   helpdlg(helpmsg,'Empty data!');
   return;
end
file_name = tempname;
file_name = strcat(file_name,'.m');
fid = fopen(file_name,'w');
fprintf(fid,'%% %+35s\n','DATA INFO');
fprintf(fid,'%%\n');
print_info(DATA,2,fid);
directory = tempdir;
addpath (directory);
helpwin (file_name);
fclose(fid);
delete(file_name);
rmpath (directory);


function [] = map_info()
global MAP;
if isempty(MAP)
   helpmsg = 'There is no map!';
   helpdlg(helpmsg,'Empty map!');
   return;
end
file_name = tempname;
file_name = strcat(file_name,'.m');
fid = fopen(file_name,'w');
fprintf(fid,'%% %+35s\n','MAP INFO');
fprintf(fid,'%%\n');
print_info(MAP,2,fid);
directory = tempdir;
addpath (directory);
helpwin (file_name);
fclose(fid);
delete(file_name);
rmpath (directory); 


function [] = helpwin1()
file1 = tempname;
file1 = strcat(file1,'.m');
directory = tempdir;	
html2tex('file:///share/somtoolbox/vs2/html/som_GUI.html',file1);
addpath (directory);
helpwin (file1);
rmpath (directory);
delete (file1);		




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     END OF HELP & INFO	%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     START OF OTHER FUNC	%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	



function [msgs_nro, value, Handle] = check_ok(Tag)
Handle = findobj(gcbf,'Tag',Tag);
temp = get(Handle,'String');
value = str2num(temp);
if isempty(value)
   msgs_nro = 1;
   return;
end
[test1 test2] = size(value);
if test1 ~= 1 || test2 ~= 1
   msgs_nro = 2;	  
   return; 
end
msgs_nro = 0;	


function [msgs_nro, value, Handle] = msize_ok(Tag)
Handle = findobj(gcbf,'Tag',Tag);
temp = get(Handle,'String');
value = str2num(temp);
if isempty(value)
   msgs_nro = 1;
   return;
end
[test1 test2] = size(value);
if test1 ~= 1 || test2 ~= 2
   msgs_nro = 2;	  
   return; 
end
msgs_nro = 0;


%%% Changed 1.2.2000

function [] = visualize()
global MAP;
if isempty(MAP)
   helpmsg = {'Train map before tryinig to visualize it!'};
   helpdlg(helpmsg,'Empty Map!');
   return;
end

dim = size(MAP.codebook,2);
odim = 2;
[P,V] = pcaproj(MAP.codebook,odim);
ccode = som_colorcode(MAP, 'rgb1');

figure; 
som_show(MAP,'umat','all','comp',1:dim,'norm','d');  
figure;
subplot(1,2,1) 
som_grid(MAP,'Coord',P,'MarkerColor',ccode,'Markersize',5, ...
	 'Linewidth',1,'Linecolor','k');
xlabel('PC1'), ylabel('PC2')
title('PCA-projection (on the left), color coding (on the right)')
axis tight, axis equal
subplot(1,2,2) 
som_cplane(MAP.topol.lattice,MAP.topol.msize,ccode);

%msgbox('Save map in workspace. Load it from there.');
%som_gui('save_workspace');  
%som_comp_vis;

%%%%%%%%%%%%%%%%


function [] = clear_all()

Handle = findobj(gcbf,'Enable','off');
set(Handle,'Enable','on');
Handle = findobj(gcbf,'Tag','Radiobutton1');
set(Handle,'Value',0);
Handle = findobj(gcbf,'Tag','StaticText10');
set(Handle,'String','Status <no action>');
Handle = findobj(gcbf,'Tag','StaticText3');
set(Handle,'String','Map: <empty>');
Handle = findobj(gcbf,'Tag','StaticText4');
set(Handle,'String','Data: <empty>');
Handle = findobj(gcbf,'Tag','StaticText20');
set(Handle,'String','lattice:');
Handle = findobj(gcbf,'Tag','StaticText11');
set(Handle,'String','neigh:');
Handle = findobj(gcbf,'Tag','StaticText16');
set(Handle,'String','training length:');
Handle = findobj(gcbf,'Tag','StaticText23');
set(Handle,'String','training length:');
Handle = findobj(gcbf,'Tag','StaticText17');
set(Handle,'String','radius initial:');
Handle = findobj(gcbf,'Tag','StaticText24');
set(Handle,'String','radius initial:');
Handle = findobj(gcbf,'Tag','StaticText5');
set(Handle,'String','map size:');
Handle = findobj(gcbf,'Tag','StaticText21');
set(Handle,'String','shape:');
Handle = findobj(gcbf,'Tag','StaticText12');
set(Handle,'String','order:');
set(Handle,'Visible','off');
Handle = findobj(gcbf,'Tag','StaticText14');
set(Handle,'String','length type:');
set(Handle,'Visible','off');
Handle = findobj(gcbf,'Tag','StaticText22');
set(Handle,'String','radius final:');
Handle = findobj(gcbf,'Tag','StaticText25');
set(Handle,'String','radius final:');
Handle = findobj(gcbf,'Tag','StaticText19');
set(Handle,'String','tracking:');
Handle = findobj(gcbf,'Tag','StaticText7');
set(Handle,'String','Initialization');
Handle = findobj(gcbf,'Tag','StaticText28');
set(Handle,'String','alpha type:');
set(Handle,'Visible','off');
Handle = findobj(gcbf,'Tag','StaticText26');
set(Handle,'String','alpha initial:');
Handle = findobj(gcbf,'Tag','StaticText27');
set(Handle,'String','alpha initial:');
Handle = findobj(gcbf,'Tag','StaticText6');
set(Handle,'String','type:');
Handle = findobj(gcbf,'Tag','StaticText9');
set(Handle,'String','training type:');
Handle = findobj(gcbf,'Tag','Pushbutton9');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','Pushbutton6');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','Pushbutton4');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','Pushbutton5');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','Pushbutton2');
set(Handle,'Enable','on');
Handle = findobj(gcbf,'Tag','Radiobutton1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','Load/SaveSave mapuimenu1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','&Load/SaveSave mapSave in workspaceuimenu1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','Subuimenu2');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','&ToolsSubuimenu1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','&Help/InfoHelp windowuimenu1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','&Help/InfoData infouimenu1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','&Init&Trainuimenu1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','&Init&TrainInitialize1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','&Init&TrainChange initialization valuesuimenu1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','&Init&TrainTrain1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'Tag','Load/SaveSubuimenu1');
set(Handle,'Enable','off');
Handle = findobj(gcbf,'String','alpha initial:');
set(Handle,'Visible','off');
clear;
clear global;


function [] = close_fig()
global MAPSAVED NEWMAP;
if isempty(MAPSAVED)
  if not(isempty(NEWMAP))
     quest = 'Save map before closing?';
     ButtonName = questdlg(quest);
     switch ButtonName
        case 'Yes'
	  som_gui('savemap');		
	  som_gui('clear');
	  clear global;
	  close(gcbf);
	 case 'No'
	   som_gui('clear');	
	   clear global;
	   close(gcbf);
	 case 'Cancel'
      end
  else
     som_gui('clear');
     clear global;
     close(gcbf);	
  end	
else
  som_gui('clear');
  clear global;
  close(gcbf);
end


function [] = preprocess_gui()
global DATA; 
if isempty(DATA)
   helpmsg = {'Load data before tryinig to preprocess!'};
   helpdlg(helpmsg,'Empty Data!');
   return;
end
preprocess(DATA);
waitfor(gcf);
prompt = {'Name of preprocessed data in workspace?'};
tittle = 'Reload preprocessed data!';
lineNo = 1;
def = {DATA.name};
answer = inputdlg(prompt,tittle,lineNo,def);
if isempty(answer)
  return;
end
data = answer{1};
new_name = retname;
assignin('base',new_name,data);
str = ['exist(' new_name ')'];	
temp = evalin('base',str); 
if temp ~= 1
 temp = strcat('Variable ''',data,''' doesn''t exist in workspace.',...
	'Old Data which is not preprocessed will be used.');
  errordlg(temp,'Unknown variable!');
  return;
end
evalin('base',['clear ' new_name ]) 
Handle = findobj(gcf,'Tag','StaticText4');
temp = strcat('Data: <',data,'>');
set(Handle,'String',temp);
Handle = findobj(gcf,'Tag','StaticText10');
set(Handle,'String','Status <data preprocessed>');
temp = evalin('base',data);
DATA.data = temp;
som_gui('def_initialization');


function [val] = loop(cell_data, search_data)
for val = 1: length(cell_data) 
  if strcmp(cell_data{val},search_data)
    break;
  end
end
if not(strcmp(cell_data{val},search_data))
  val = -1;
end


function [] = comp_names(names,fid)
last = size(names);
for index=1:last
	fprintf(fid,'%% %s\n',names{index})
end	

function [] = fill_field(names,mask,fid)
last = size(mask);
for index=1:last
	num = num2str(mask(index))
	fprintf(fid,'%% %-15s %-2s\n',names{index},num)
end	


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%     END OF OTHER FUNC		%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%				%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	


      

function fig = main_gui()

v = version;
ver_53_or_newer = (str2num(v(1:3)) >= 5.3);

h0 = figure('Units','normalized', ...
	'Color',[0.85 0.85 0.85], ...
	'Name','SOM Toolbox -- Initialization & Training', ...
	'NumberTitle','off', ...
	'PaperPosition',[18 180 576 432], ...
	'PaperUnits','points', ...
	'Position',[0.3296875 0.28125 0.3828125 0.576171875], ...
	'Tag','Fig1'); 
if ver_53_or_newer, set(h0,'ToolBar','none'); end

h1 = uimenu('Parent',h0, ...
	'Label','&Load/Save', ...
	'Tag','uimenu1');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''load_data'');',...
	'Label','Load Data', ...
	'Tag','Subuimenu1');
h2 = uimenu('Parent',h1, ...
	'Label','Save map', ...
	'Enable','off',...
	'Tag','Load/SaveSubuimenu1');
h3 = uimenu('Parent',h2, ...
	'Callback','som_gui(''save_workspace'');', ...
	'Enable','off', ...
	'Label','Save in workspace', ...
	'Tag','Load/SaveSave mapuimenu1');
h3 = uimenu('Parent',h2, ...
	'Callback','som_gui(''savemap'');', ...
	'Enable','off', ...
	'Label','Write cod-file', ...
	'Tag','&Load/SaveSave mapSave in workspaceuimenu1');
h1 = uimenu('Parent',h0, ...
	'Label','&Utilities', ...
	'Tag','uimenu2');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''preprocess'');', ...
	'Enable','off', ...
	'Label','Preprocess Data', ...
	'Tag','Subuimenu2');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''visualize'');', ...
	'Enable','off', ...
	'Label','Visualize Map', ...
	'Tag','&ToolsSubuimenu1');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''clear_all'');', ...
	'Label','Clear all', ...
	'Tag','&ToolsSubuimenu2');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''close'');', ...
	'Label','Close Figure', ...
	'Tag','&ToolsClear alluimenu1');
h1 = uimenu('Parent',h0, ...
	'Label','&Info', ...
	'Tag','&ToolsClose Figureuimenu1');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''help'');', ...
	'Label','WWW Help', ...
	'Tag','Helpuimenu1');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''helpwin'');', ...
	'Label','Help window', ...
	'Tag','Helpuimenu2');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''helpwin2'');', ...
	'Label','About GUI', ...
	'Tag','&Help/InfoHelp windowuimenu2');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''data_info'');', ...
	'Enable','off', ...
	'Label','Data info', ...
	'Tag','&Help/InfoHelp windowuimenu1');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''map_info'');', ...
	'Enable','off', ...
	'Label','Map info', ...
	'Tag','&Help/InfoData infouimenu1');
h1 = uimenu('Parent',h0, ...
	'Label','&Init/Train', ...
	'Tag','&Init/Train1');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''change_initialization'');', ...
	'Enable','off', ...
	'Label','Change initialization values', ...
	'Tag','&Init&Trainuimenu1');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''init'');', ...
	'Enable','off', ...
	'Label','Initialize', ...
	'Tag','&Init&TrainInitialize1');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''change_def'');', ...
	'Enable','off', ...
	'Label','Change training values', ...
	'Tag','&Init&TrainChange initialization valuesuimenu1');
h2 = uimenu('Parent',h1, ...
	'Callback','som_gui(''def_train'');', ...
	'Enable','off', ...
	'Label','Train', ...
	'Tag','&Init&TrainTrain1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.04081632653061224 0.01129943502824859 0.7619047619047619 0.9717514124293786], ...
	'Style','frame', ...
	'Tag','Frame1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.06802721088435373 0.7909604519774012 0.7074829931972788 0.1807909604519774], ...
	'Style','frame', ...
	'Tag','Frame2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.09523809523809523 0.8527570621468927 0.6530612244897959 0.03389830508474576], ...
        'FontUnits','normalized',...
	'String','Map <empty>', ...
	'Style','text', ...
	'Tag','StaticText3');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.09523809523809523 0.8075593220338984 0.6530612244897959 0.03389830508474576], ...
	'String','Data <empty>', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText4');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.06802721088435373 0.5988700564971752 0.7074829931972788 0.1694915254237288], ...
	'Style','frame', ...
	'Tag','Frame3');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.1041 0.7356 0.6286 0.0271], ...
	'String','Initialization', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'FontWeight','bold', ...
	'Tag','StaticText7');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.4489795918367346 0.7005649717514124 0.2993197278911565 0.03389830508474576], ...
	'String','map size:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText5');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.09523809523809523 0.6553672316384182 0.2993197278911565 0.03389830508474576], ...
	'String','lattice:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText20');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.09523809523809523 0.7000000000000001 0.2993197278911565 0.03389830508474576], ...
	'String','type:', ...
   'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText6');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.4489795918367346 0.6553672316384182 0.2993197278911565 0.03389830508474576], ...
	'String','shape:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText21');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.752941176470588 0.752941176470588 0.752941176470588], ...
	'ListboxTop',0, ...
	'Position',[0.3129251700680272 0.6101694915254238 0.217687074829932 0.03389830508474576], ...
	'String','Change values', ...
        'FontUnits','normalized',...	       
	'Callback','som_gui(''change_initialization'');', ...
	'Enable','off', ...
	'Tag','Pushbutton9');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.06802721088435373 0.02259887005649718 0.7074829931972788 0.5536723163841808], ...
	'Style','frame', ...
	'Tag','Frame4');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.1041 0.5316 0.6429 0.0339], ...
	'String','Training', ...
   'FontUnits','normalized',...	       
	'Style','text', ...
	'FontWeight','bold', ...
	'Tag','StaticText8');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'ListboxTop',0, ...
	'Position',[0.09523809523809523 0.4971751412429379 0.6530612244897959 0.03389830508474576], ...
	'String','training type', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText9');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.4489795918367346 0.4519774011299435 0.2993197278911565 0.03389830508474576], ...
	'String','tracking:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText19');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.09523809523809523 0.4519774011299435 0.2993197278911565 0.03389830508474576], ...
	'String','neigh:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText11');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'HorizontalAlignment','left', ...	       
	'BackgroundColor',[0.9 0.9 0.9], ...
	'ListboxTop',0, ...
	'Position',[0.09523809523809523 0.36519774011299435 0.2993197278911565 0.03389830508474576], ...	       
	'String','alpha type:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Visible','off',...
	'Tag','StaticText28');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'ListboxTop',0, ...
	'HorizontalAlignment','left', ...	       
	'Position',[0.09523809523809523 0.4067796610169492 0.2993197278911565 0.03389830508474576], ...	       
	'String','length type:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Visible','off',...
	'Tag','StaticText14');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.4489795918367346 0.4067796610169492 0.2993197278911565 0.03389830508474576], ...
	'String','order:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Visible','off',...
	'Tag','StaticText12');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.09523809523809523 0.07909604519774012 0.2993197278911565 0.2711864406779661], ...
	'Style','frame', ...
	'Tag','Frame5');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.4353741496598639 0.07909604519774012 0.2993197278911565 0.2711864406779661], ...
	'Style','frame', ...
	'Tag','Frame6');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.108843537414966 0.3050847457627119 0.2721088435374149 0.03389830508474576], ...
	'String','Rough', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText13');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.4489795918367346 0.3050847457627119 0.2721088435374149 0.03389830508474576], ...
	'String','Finetune', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText15');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.108843537414966 0.1807909604519774 0.2721088435374149 0.03389830508474576], ...
	'String','training length:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText16');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.108843537414966 0.2694915254237288 0.2714285714285714 0.03389830508474576], ...
	'String','radius initial:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText17');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.1088 0.2260 0.2721 0.0339], ...
	'String','radius final:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText22');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'ListboxTop',0, ...
	'Position',[0.108843537414966 0.13694915254237288 0.2714285714285714 0.03389830508474576], ...	       
	'String','alpha initial:', ...
        'FontUnits','normalized',...	       
	'HorizontalAlignment','left', ...	       
	'Style','text', ...
	'Visible','off',...
	'Tag','StaticText26');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.4489795918367346 0.1807909604519774 0.2721088435374149 0.03389830508474576], ...
	'String','training length:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText23');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.4489795918367346 0.2711864406779661 0.2721088435374149 0.03389830508474576], ...
	'String','radius initial:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText24');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.4490 0.2260 0.2721 0.0339], ...
	'String','radius final:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText25');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'ListboxTop',0, ...
	'Position',[0.4489795918367346  0.13694915254237288 0.2721088435374149 0.03389830508474576], ...	       
	'String','alpha initial:', ...
        'FontUnits','normalized',...	       
	'HorizontalAlignment','left', ...	       
	'Style','text', ...
	'Visible','off',...
	'Tag','StaticText27');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.752941176470588 0.752941176470588 0.752941176470588], ...
	'ListboxTop',0, ...
	'Position',[0.3129251700680272 0.03389830508474576 0.217687074829932 0.03389830508474576], ...
	'String','Change values', ...
        'FontUnits','normalized',...	       
	'Callback','som_gui(''change_def'');', ...
	'Enable','off', ...
	'Tag','Pushbutton6');
if ver_53_or_newer, set(h1,'TooltipString','Change default values in training.'); end

h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.752941176470588 0.752941176470588 0.752941176470588], ...
	'ListboxTop',0, ...
	'Position',[0.8163265306122448 0.8152542372881356 0.163265306122449 0.05593220338983051], ...
	'String','LOAD', ...
        'FontUnits','normalized',...	       
	'Callback','som_gui(''load_data'');', ...
	'Tag','Pushbutton2');
if ver_53_or_newer, set(h1,'TooltipString','Load data file.'); end

h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.752941176470588 0.752941176470588 0.752941176470588], ...
	'ListboxTop',0, ...
	'Position',[0.8163265306122448 0.6457627118644068 0.163265306122449 0.05593220338983051], ...
	'String','INITIALIZE', ...
        'FontUnits','normalized',...	       
	'Callback','som_gui(''init'');', ...
	'Enable','off', ...
	'Tag','Pushbutton4');
if ver_53_or_newer, set(h1,'TooltipString','Initialize map.'); end

h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.752941176470588 0.752941176470588 0.752941176470588], ...
	'ListboxTop',0, ...
	'Position',[0.8163265306122448 0.384180790960452 0.163265306122449 0.05649717514124294], ...
	'String','TRAIN', ...
        'FontUnits','normalized',...	       
	'Callback','som_gui(''def_train'');', ...
	'Enable','off', ...
	'Tag','Pushbutton5');
if ver_53_or_newer, set(h1,'TooltipString','Train map whit default values.'); end

h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.752941176470588 0.752941176470588 0.752941176470588], ...
	'ListboxTop',0, ...
	'Position',[0.8163265306122448 0.06779661016949153 0.163265306122449 0.05649717514124294], ...
	'Callback','som_gui(''close'');', ...
	'String','CLOSE', ...
        'FontUnits','normalized',...	       
	'Tag','Pushbutton8');
if ver_53_or_newer, set(h1,'TooltipString','Close figure.'); end

h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.9 0.9 0.9], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.09387755102040815 0.897954802259887 0.6530612244897959 0.03389830508474576], ...
	'String','Status <no action>', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText10');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.108843537414966 0.0903954802259887 0.2721088435374149 0.03389830508474576], ...
	'String','Only finetune', ...
        'FontUnits','normalized',...	       
	'Callback','som_gui(''only_finetune'');', ...
	'Enable','off', ...
	'Style','radiobutton', ...
	'Tag','Radiobutton1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.09523809523809523 0.9418531073446328 0.6530612244897959 0.0259887005649718], ...
   'String','Information', ...
   'FontUnits','normalized',...
	'FontWeight','bold', ...
	'Style','text', ...
	'Tag','StaticText18');
if nargout > 0, fig = h0; end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function fig = loadgui3()

temp = {'dat file';'mat file'};

h0 = figure('Units','normalized', ...
        'Color',[0.8 0.8 0.8], ...
	'Name','Load data!', ...
	'NumberTitle','off', ...
	'PaperType','a4letter', ...
	'Position',[0.3828125 0.5 0.3421875 0.189453125], ...
	'Tag','Fig1');

h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.02853881278538813 0.06443298969072164 0.7705479452054794 0.8698453608247422], ...
	'Style','frame', ...
	'Tag','Frame1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.04337899543378995 0.547680412371134 0.7420091324200913 0.354381443298969], ...
	'Style','frame', ...
	'Tag','Frame2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.04280821917808219 0.09664948453608246 0.7420091324200913 0.4188144329896907], ...
	'Style','frame', ...
	'Tag','Frame3');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'FontWeight','bold', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.05717762557077625 0.7881958762886597 0.2853881278538812 0.09664948453608246], ...
	'String','From', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'Callback','som_gui(''workspace'');', ...
	'ListboxTop',0, ...
	'Position',[0.05107762557077625 0.7087628865979381 0.1997716894977169 0.09664948453608246], ...	       	       
	'String','Ws', ...
   'FontUnits','normalized',...	       
	'Style','radiobutton', ...
	'Tag','Radiobutton2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'Callback','som_gui(''file'');', ...
	'ListboxTop',0, ...
	'Position',[0.05107762557077625 0.5773195876288659 0.2009132420091324 0.09793814432989689], ...
	'String','File', ...
        'FontUnits','normalized',...	       
	'Style','radiobutton', ...
	'Tag','Radiobutton1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','Handle = findobj(gcbf,''Tag'',''EditText2'');set(Handle,''String'','''');',...
        'FontUnits','normalized',...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.2893881278538812 0.7087628865979381 0.3139269406392694 0.09664948453608246], ...	       
	'Style','edit', ...
	'Tag','EditText1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','Handle = findobj(gcbf,''Tag'',''EditText1'');set(Handle,''String'','''');',...
        'FontUnits','normalized',...	       
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.2893881278538812 0.5798969072164948 0.3139269406392694 0.09664948453608246], ...	       
	'Style','edit', ...
	'Tag','EditText2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'Callback','som_gui(''browse'');', ...
	'ListboxTop',0, ...
	'Position',[0.6279 0.5799 0.1427 0.2255], ...
	'String','Browse', ...
        'FontUnits','normalized',...	       
	'Tag','Pushbutton1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'Callback','som_gui(''load_ok'');', ...
	'ListboxTop',0, ...
	'Position',[0.8276 0.5577 0.1427 0.2255], ...
	'String','Load', ...
        'FontUnits','normalized',...	       
	'Tag','Pushbutton2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'Callback','close;',...
	'ListboxTop',0, ...
	'Position',[0.8276 0.2577 0.1427 0.2255], ...
	'String','Cancel', ...
	'FontUnits','normalized',...       
	'Tag','Pushbutton3');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'Callback','som_gui(''file_select'');', ...
	'ListboxTop',0, ...
	'Max',2, ...
	'Min',1, ...
	'String',temp,...
   'FontUnits','normalized',... 
	'Position',[0.3995433789954338 0.2977319587628866 0.1997716894977169 0.08664948453608246], ...	       
	'Style','popupmenu', ...
	'Tag','PopupMenu1', ...
	'Value',1);
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'FontWeight','bold', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.05707762557077625 0.3865979381443299 0.7134703196347032 0.09664948453608246], ...
	'String','Parameters for file', ...
        'FontUnits','normalized',...
	'Style','text', ...
	'Tag','StaticText2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.05707762557077625 0.2777319587628866 0.2568493150684931 0.09664948453608246], ...
	'String','File type ', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText3');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.05707762557077625 0.1288659793814433 0.2996575342465753 0.09664948453608246], ...
	'String','Missing value', ...
	'Style','checkbox', ...
        'FontUnits','normalized',...	       
	'Tag','Checkbox1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''missing'');',...
	'ListboxTop',0, ...
	'Position',[0.5136986301369862 0.1258659793814433 0.08561643835616438 0.10664948453608246], ...	       
	'String','x', ...
        'FontUnits','normalized',...	       
	'Style','edit', ...
	'Tag','EditText3');
if nargout > 0, fig = h0; end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%




function fig = works()

v = version;
ver_53_or_newer = (str2num(v(1:3)) >= 5.3);

h0 = figure('Units','normalized', ...
	'Color',[0.8 0.8 0.8], ...
	'Name','Load from workspace!', ...
	'NumberTitle','off', ...
	'PaperPosition',[18 180 576 432], ...
	'PaperType','a4letter', ...
	'PaperUnits','points', ...
	'Position',[0.5390625 0.2490234375 0.203125 0.251953125], ...
	'Tag','Fig1');
if ver_53_or_newer, set(h0,'ToolBar','none'); end

h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.05384615384615385 0.1472868217054263 0.9076923076923078 0.8255813953488372], ...
	'Style','frame', ...
	'Tag','Frame1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'Callback','som_gui(''works_ok'');', ...
	'ListboxTop',0, ...
	'Position',[0.1077 0.0194 0.2885 0.1202], ...
	'String','OK', ...
        'FontUnits','normalized',...	       
	'Tag','Pushbutton1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'Callback','close;', ...
	'ListboxTop',0, ...
	'Position',[0.6115 0.0155 0.2885 0.1202], ...
	'String','Cancel', ...
        'FontUnits','normalized',...	       
	'Tag','Pushbutton2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Position',[0.1192 0.1977 0.7692 0.6395], ...
	'String',' ', ...
        'FontUnits','normalized',...	       
	'Style','listbox', ...
	'Tag','Listbox1', ...
	'Value',1);
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'FontWeight','bold', ...
	'ListboxTop',0, ...
	'Position',[0.2115384615384616 0.8720930232558139 0.576923076923077 0.06976744186046512], ...
	'String','Your options', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText1');
if nargout > 0, fig = h0; end






function fig = initialization2()


temp1 = {'random';'linear'};
temp2 = {'hexa';'rect'};
temp3 = {'sheet';'cyl';'toroid'};



% position bug in following corrected 1.12.04 KimmoR
h0 = figure('Units','normalized', ...
        'Color',[0.8 0.8 0.8], ...
	'Name','Change initialization parameters!', ...
	'NumberTitle','off', ...
	'PaperType','a4letter', ...
	'Position',[0.48828125 0.4267578125 0.3515625 0.146484375], ...
	'Tag','Fig1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.02777777777777778 0.08333333333333333 0.8055555555555556 0.8333333333333334], ...	       
	'Style','frame', ...
	'Tag','Frame1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'Callback','som_gui(''change_initialization_ok'');', ...
	'ListboxTop',0, ...
	'Position',[0.8472222222222222 0.55 0.125 0.25], ...
        'FontUnits','normalized',...	       
	'String','OK', ...
	'Tag','Pushbutton1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'Callback','som_gui(''change_initialization_cancel'');', ...
	'ListboxTop',0, ...
	'Position',[0.8472222222222222 0.25 0.125 0.25], ...	       
        'FontUnits','normalized',...
	'String','Cancel', ...
	'Tag','Pushbutton2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'FontWeight','bold', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.08333333333333334 0.6666666666666666 0.7066666666666667 0.1933333333333333], ...
	'String','Initialization parameters:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.0556 0.200 0.1667 0.1250],...       
	'String','type:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'ListboxTop',0, ...
	'Max',2, ...
	'Min',1, ...
	'Position',[0.2500 0.200 0.1667 0.1250], ...
	'String',temp1, ...
        'FontUnits','normalized',...	       
	'Style','popupmenu', ...
	'Tag','PopupMenu1', ...
	'Value',1);
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.05555555555555556 0.6 0.1666666666666667 0.125], ...
	'String','map size:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''map_size'');', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.25 0.6 0.1666666666666667 0.125], ...
        'FontUnits','normalized',...	       
	'Style','edit', ...
	'Tag','EditText1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.05555555555555556 0.4033333333333333 0.1666666666666667 0.125], ...
	'String','lattice:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText3');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Max',2, ...
	'Min',1, ...
	'Position',[0.25 0.4333333333333333 0.1666666666666667 0.125], ...
	'String',temp2, ...
        'FontUnits','normalized',...	       
	'Style','popupmenu', ...
	'Tag','PopupMenu2', ...
	'Value',2);
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.4444444444444445 0.4033333333333333 0.1666666666666667 0.125], ...	       
	'String','shape:', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText4');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Max',3, ...
	'Min',1, ...
	'Position',[0.638888888888889 0.4333333333333333 0.1666666666666667 0.125], ...	       
	'String',temp3, ...
        'FontUnits','normalized',...	       
	'Style','popupmenu', ...
	'Tag','PopupMenu3', ...
	'Value',2);
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.4444444444444445 0.6 0.1666666666666667 0.125], ...
        'FontUnits','normalized',...
	'String','munits:', ...
	'Style','text', ...
	'Tag','StaticText5');	       
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''munits'');', ...
	'ListboxTop',0, ...
	'Position',[0.638888888888889 0.6 0.1666666666666667 0.125], ...
	'Style','edit', ...
        'FontUnits','normalized',...	       
	'Tag','EditText2');
if nargout > 0, fig = h0; end







function fig = new_para2_2()

temp1 = {'0';'1';'2';'3'};
temp2 = {'gaussian';'cutgauss';'ep';'bubble'};
temp3 = {'epochs';'samples'};
temp4 = {'random';'ordered'};
temp5 = {'inv';'linear';'power'};

v = version;
ver_53_or_newer = (str2num(v(1:3)) >= 5.3);

h0 = figure('Units','normalized', ...
	'Color',[0.8 0.8 0.8], ...
	'Name','Change training parameters!', ...
	'NumberTitle','off', ...
	'PaperPosition',[18 180 576 432], ...
	'PaperType','a4letter', ...
	'PaperUnits','points', ...
	'Position',[0.59140625 0.4560546875 0.3046875 0.4619140625], ...
	'Tag','Fig3');
if ver_53_or_newer, set(h0,'ToolBar','none'); end
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.02051282051282051 0.08456659619450317 0.9641025641025641 0.8921775898520086], ...
	'Style','frame', ...
	'Tag','Frame1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.5308 0.1374 0.4000 0.3742], ...
	'Style','frame', ...
	'Tag','Frame3');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'ListboxTop',0, ...
	'Position',[0.08012820512820512 0.1416490486257928 0.4102564102564102 0.3699788583509514], ...
	'Style','frame', ...
	'Tag','Frame2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'Callback','close(gcbf);', ...
	'ListboxTop',0, ...
	'Position',[0.6410 0.0036 0.2897 0.0740], ...
        'FontUnits','normalized',...	       
	'String','Cancel', ...
	'Tag','Pushbutton2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'Callback','som_gui(''set_new_parameters'');', ...
	'ListboxTop',0, ...
	'Position',[0.1026 0.0036 0.2897 0.0740], ...
	'String','Set parameters', ...
        'FontUnits','normalized',...	       
	'Tag','Pushbutton1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'ListboxTop',0, ...
	'Max',4, ...
	'Min',1, ...
	'Position',[0.7051282051282051 0.6723044397463003 0.1923076923076923 0.040169133192389], ...
	'String',temp1, ...
        'FontUnits','normalized',...	       
	'Style','popupmenu', ...
	'Tag','PopupMenu2', ...
	'Value',1);
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'ListboxTop',0, ...
	'Max',4, ...
	'Min',1, ...
	'Position',[0.2948717948717949 0.6670190274841438 0.1923076923076923 0.03964059196617336], ...
	'String',temp2, ...
	'FontUnits','normalized',...       
	'Style','popupmenu', ...
	'Tag','PopupMenu1', ...
	'Value',1);
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'Callback','som_gui(''batch_cancel'');', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.5076923076923077 0.6575052854122622 0.1923076923076923 0.05285412262156448], ...
	'String','tracking', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText6');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'Callback','som_gui(''batch_cancel'');', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.09615384615384615 0.6553911205073996 0.1923076923076923 0.05285412262156448], ...
	'String','neigh.', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText5');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.09615384615384615 0.7526427061310783 0.09487179487179487 0.04228329809725159], ...
	'String','mask:', ...
	'FontUnits','normalized',... 
	'Style','text', ...
	'Tag','StaticText2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Position',[0.2948717948717949 0.7399577167019028 0.6025641025641025 0.07399577167019028], ...
	'String',' ', ...
	'FontUnits','normalized',...  
	'Style','listbox', ...
	'Tag','Listbox2', ...
	'Value',1);
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.0962 0.8060 0.1154 0.0529], ...
        'FontUnits','normalized',...
	'String','Set', ...
	'Style','text', ...
	'Tag','StaticText3');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''set_batch_mask'');', ...
	'Position',[0.2948717948717949 0.8165961945031712 0.3205128205128205 0.05285412262156448], ...
	'String',' ', ...
        'FontUnits','normalized',...	       
	'Style','listbox', ...
	'Tag','Listbox1', ...
	'Value',1);
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.6250 0.8060 0.1603 0.0529], ...
	'String','to value', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText4');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''set_batch_mask'');', ...
	'ListboxTop',0, ...
	'Position',[0.7923076923076923 0.8181818181818182 0.09487179487179487 0.05285412262156448], ...
	'Style','edit', ...
        'FontUnits','normalized',...	       
	'Tag','EditText2');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''check_fine_trainlen'');', ...
	'ListboxTop',0, ...
	'Position',[0.7923 0.2352 0.0974 0.0402], ...
        'FontUnits','normalized',...	       
	'Style','edit', ...
	'Tag','EditText11');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''check_fine_alphaini'');', ...
	'Enable','off', ...
	'ListboxTop',0, ...
	'Position',[0.7923076923076923 0.1664904862579281 0.09743589743589742 0.03805496828752643], ...
	'Style','edit', ...
        'FontUnits','normalized',...	       
	'Tag','EditText10', ...
	'Visible','off');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''check_fine_radfin'');', ...
	'ListboxTop',0, ...
	'Position',[0.7923076923076923 0.3002114164904862 0.09743589743589742 0.040169133192389], ...
	'Style','edit', ...
	'FontUnits','normalized',...       
	'Tag','EditText9');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''check_fine_radini'');', ...
	'ListboxTop',0, ...
	'Position',[0.7923076923076923 0.3657505285412262 0.09743589743589742 0.040169133192389], ...
	'Style','edit', ...
	'FontUnits','normalized',...     
	'Tag','EditText8');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.8 0.8 0.8], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.5590 0.2326 0.2179 0.0402], ...
	'String','training length', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText16');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.8 0.8 0.8], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.5590 0.1665 0.2179 0.0381], ...
	'String','alpha initial', ...
	'FontUnits','normalized',...     
	'Style','text', ...
	'Tag','StaticText14', ...
	'Visible','off');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.8 0.8 0.8], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.5590 0.2981 0.2179 0.0402], ...
	'String','radius final', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText12');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.8 0.8 0.8], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.5590 0.3636 0.2179 0.0402], ...
	'String','radius initial', ...
	'FontUnits','normalized',...      
	'Style','text', ...
	'Tag','StaticText10');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''check_rough_trainlen'');', ...
	'ListboxTop',0, ...
	'Position',[0.3590 0.2352 0.0949 0.0402], ...
	'Style','edit', ...
	'FontUnits','normalized',...     
	'Tag','EditText7');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''check_rough_alphaini'');', ...
	'Enable','off', ...
	'ListboxTop',0, ...
	'Position',[0.3590 0.1691 0.0949 0.0381], ...
	'Style','edit', ...
        'FontUnits','normalized',...	       
	'Tag','EditText6', ...
	'Visible','off');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''check_rough_radfin'');', ...
	'ListboxTop',0, ...
	'Position',[0.358974358974359 0.3044397463002114 0.09487179487179487 0.040169133192389], ...
	'Style','edit', ...
	'FontUnits','normalized',...      
	'Tag','EditText5');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[1 1 1], ...
	'Callback','som_gui(''check_rough_radini'');', ...
	'ListboxTop',0, ...
	'Position',[0.358974358974359 0.3699788583509514 0.09487179487179487 0.040169133192389], ...
	'Style','edit', ...
        'FontUnits','normalized',...	       
	'Tag','EditText4');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.8 0.8 0.8], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.0962 0.2326 0.2179 0.0402], ...
	'String','training length', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText15');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.8 0.8 0.8], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.0962 0.1691 0.2179 0.0381], ...
	'String','alpha initial', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText13', ...
	'Visible','off');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.8 0.8 0.8], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.0962 0.3023 0.2179 0.0402], ...
	'String','radius final', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText11');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.8 0.8 0.8], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.0962 0.3679 0.2179 0.0402], ...
        'FontUnits','normalized',...
	'String','radius initial', ...
	'Style','text', ...
	'Tag','StaticText9');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'ListboxTop',0, ...
	'Position',[0.5948717948717949 0.4291754756871036 0.2871794871794872 0.05285412262156448], ...
	'String','Finetune', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText8');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'ListboxTop',0, ...
	'Position',[0.1205128205128205 0.4355179704016914 0.3153846153846154 0.04862579281183932], ...
	'String','Rough', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText7');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'FontWeight','bold', ...
	'ListboxTop',0, ...
	'Position',[0.1641025641025641 0.8900634249471459 0.7025641025641025 0.05285412262156448], ...
	'String','Change parameters for batch training', ...
	'Style','text', ...
        'FontUnits','normalized',...	       
	'Tag','StaticText1');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.09615384615384615 0.6025369978858351 0.1743589743589744 0.040169133192389], ...
	'String','length type:', ...
	'Style','text', ...
        'FontUnits','normalized',...	       
	'Tag','StaticText17');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'ListboxTop',0, ...
	'Max',2, ...
	'Min',1, ...
	'Position',[0.2948717948717949 0.6062367864693446 0.1923076923076923 0.03964059196617336], ...
	'String',temp3, ...
        'FontUnits','normalized',...	       
	'Style','popupmenu', ...
	'Tag','PopupMenu3', ...
	'Value',1);
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.5102564102564102 0.6004228329809724 0.1641025641025641 0.040169133192389], ...
	'String','order', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText18');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Max',2, ...
	'Min',1, ...
	'Position',[0.7051282051282051 0.6109936575052853 0.1923076923076923 0.040169133192389], ...
	'String',temp4, ...
        'FontUnits','normalized',...	       
	'Style','popupmenu', ...
	'Tag','PopupMenu4', ...
	'Value',1);
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'HorizontalAlignment','left', ...
	'ListboxTop',0, ...
	'Position',[0.09615384615384615 0.5369978858350951 0.2051282051282051 0.040169133192389], ...
	'String','learning func', ...
        'FontUnits','normalized',...	       
	'Style','text', ...
	'Tag','StaticText19');
h1 = uicontrol('Parent',h0, ...
	'Units','normalized', ...
	'BackgroundColor',[0.701960784313725 0.701960784313725 0.701960784313725], ...
	'ListboxTop',0, ...
	'Max',3, ...
	'Min',1, ...
	'Position',[0.2948717948717949 0.5454545454545455 0.1923076923076923 0.03964059196617336], ...
	'String',temp5, ...
        'FontUnits','normalized',...	       
	'Style','popupmenu', ...
	'Tag','PopupMenu5', ...
	'Value',1);
if nargout > 0, fig = h0; end


function print_info(sS,level,fid)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% check arguments


%error(nargchk(1, 2, nargin))  % check no. of input args is correct

if ~isstruct(sS),
  if ~iscell(sS) || ~isstruct(sS{1}), 
    error('Input argument is not a struct or a cell array of structs.')
  end
  csS = sS;
else
  csS = {sS};
end

if nargin<2 || isempty(level) || isnan(level), level = 1; end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% print struct information

for c=1:length(csS), 
 sS = csS{c};

 switch sS.type, 
 case 'som_map', 
  mdim = length(sS.topol.msize);
  [munits dim] = size(sS.codebook);
  t    = length(sS.trainhist);  
  if t==0, st='uninitialized'; 
  elseif t==1, st = 'initialized';
  else st = sprintf('initialized, trained %d times',t-1);
  end

  % level 1
  fprintf(fid,'%% Struct type                           : %s\n', sS.type);
  fprintf(fid,'%% Map name                              : %s\n', sS.name);
  fprintf(fid,'%% Input dimension                       : %d\n', dim);
  fprintf(fid,'%% Map grid size                         : ');
  for i = 1:mdim - 1, fprintf(fid,'%d x ',sS.topol.msize(i)); end
  fprintf(fid,'%d\n', sS.topol.msize(mdim));
  fprintf(fid,'%% Lattice type (rect/hexa)              : %s\n', sS.topol.lattice);
  fprintf(fid,'%% Shape (sheet/cyl/toroid)              : %s\n', sS.topol.shape);
  fprintf(fid,'%% Neighborhood type                     : %s\n', sS.neigh);
  fprintf(fid,'%% Mask                                  : ');
  if dim,
    for i = 1:dim-1, fprintf(fid,'%d ',sS.mask(i)); end; 
    fprintf(fid,'%d\n',sS.mask(dim));
  else fprintf(fid,'%% \n');
  end
  fprintf(fid,'%% Training status                       : %s\n', st);
   
  % level 1,
  status = cell(dim,1);
  for i=1:dim, 
    n = length(sS.comp_norm{i});
    if n, 
      uninit = strcmp('uninit',{sS.comp_norm{i}.status});
      done   = strcmp('done',{sS.comp_norm{i}.status});
      undone = strcmp('undone',{sS.comp_norm{i}.status});
      if sum(uninit)==n, status{i} = 'no normalization';
      elseif sum(done)==n, status{i} = 'normalized';
      elseif sum(undone)==n, status{i} = 'denormalized';
      else status{i} = 'partial';
      end
    else status{i} = 'no normalization'; end
  end
  if level>1, 
    fprintf(fid,'%% Vector components\n');
    M = sS.codebook;
    fprintf(fid,'%%  #   name          mask     min    mean     max     std  status\n');
    fprintf(fid,'%%  --- ------------  ----  ------  ------  ------  ------  ------\n');
    for i = 1:dim,
      fprintf(fid,'%%  %-3d %-12s  %-4.2f  %6.1g  %6.1g  %6.1g  %6.1g  %s\n', ...
              i,sS.comp_names{i}, sS.mask(i), ...
              min(M(:,i)),mean(M(:,i)),max(M(:,i)),std(M(:,i)),status{i});
    end
  end

  % level 3
  if level>2,
    fprintf(fid,'%% Vector component normalizations\n');
    fprintf(fid,'%%  #   name          method (i=uninit,u=undone,d=done)\n');
    fprintf(fid,'%%  --- ------------  ---------------------------------------\n');
    for i=1:dim,  
      fprintf(fid,'%%  %-3d %-12s  ',i,sS.comp_names{i});
      n = length(sS.comp_norm{i}); 
      for j=1:n, 
        m = sS.comp_norm{i}(j).method;
        s = sS.comp_norm{i}(j).status;
        if strcmp(s,'uninit'), c='i'; 
        elseif strcmp(s,'undone'), c='u'; 
        else c='d';
        end
        fprintf(fid,'%% %s[%s] ',m,c);
      end
      fprintf(fid,'%% \n');
    end
  end
  
  % level 4
  if level>3,
    fprintf(fid,'%% Training history\n');
    for i=1:t, 
      sT = sS.trainhist(i);
      fprintf(fid,'%% * Algorithm: %8s   Data: %13s  Trainlen:  %8d\n',...
	      sT.algorithm,sT.data_name,sT.trainlen);
      %if i>1, 
        fprintf(fid,'%%   Neighborh: %8s  Mask:  ',sT.neigh);
        for i = 1:dim-1, fprintf(fid,'%% %d ',sT.mask(i)); end; 
        fprintf(fid,'%% %d\n',sT.mask(mdim));
        fprintf(fid,'%%   Radius:  %4.2f->%4.2f  Alpha: %5.3f (%s)\n', ...
		sT.radius_ini,sT.radius_fin,sT.alpha_ini,sT.alpha_type);
      %end
      fprintf(fid,'%%   Time:  %s\n',sT.time);
    end
  end

 case 'som_data',

  [dlen dim] = size(sS.data);
  if dlen*dim
    ind      = find(~isnan(sum(sS.data),2));
  else ind = []; end
  complete = size(sS.data(ind,:),1);
  partial  = dlen - complete;
  values   = numel(sS.data);
  missing  = sum(sum(isnan(sS.data))); 

  % level 1  
  fprintf(fid,'%% Struct type             : %s\n', sS.type);
  fprintf(fid,'%% Data name               : %s\n', sS.name);
  fprintf(fid,'%% Vector dimension        : %d\n', dim);
  fprintf(fid,'%% Number of data vectors  : %d\n', dlen);
  fprintf(fid,'%% Complete data vectors   : %d\n', complete);
  fprintf(fid,'%% Partial data vectors    : %d\n', partial);  
  if values, r = floor(100 * (values - missing) / values); else r = 0; end
  fprintf(fid,'%% Complete values         : %d of %d (%d%%)\n', ...
          values-missing, values, r); 

  % level 2,
  status = cell(dim,1);
  for i=1:dim, 
    n = length(sS.comp_norm{i});
    if n, 
      uninit = strcmp('uninit',{sS.comp_norm{i}.status});
      done   = strcmp('done',{sS.comp_norm{i}.status});
      undone = strcmp('undone',{sS.comp_norm{i}.status});
      if sum(uninit)==n, status{i} = 'no normalization';
      elseif sum(done)==n, status{i} = 'normalized';
      elseif sum(undone)==n, status{i} = 'denormalized';
      else status{i} = 'partial';
      end
    else status{i} = 'no normalization'; end
  end
  if level>1, 
    fprintf(fid,'%% Vector components\n');
    D = sS.data;
    fprintf(fid,'%%  #   name            min     mean     max     std  missing      status\n');
    fprintf(fid,'%%  --- ------------  ------  ------  ------  ------  -----------  ------\n');
    for i = 1:dim,
      known = find(~isnan(D(:,i))); 
      miss = dlen-length(known);
      fprintf(fid,'%%  %-3d %-12s  %6.1g  %6.1g  %6.1g  %6.1g  %5d (%2d%%)  %s\n', ...
              i,sS.comp_names{i}, ...
              min(D(known,i)),mean(D(known,i)),max(D(known,i)),std(D(known,i)), ...
              miss,floor(100*miss/dlen),status{i});
    end
  end

  % level 3
  if level>2,
    fprintf(fid,'%% Vector component normalizations\n');
    fprintf(fid,'%%  #   name          method (i=uninit,u=undone,d=done)\n');
    fprintf(fid,'%%  --- ------------  ---------------------------------------\n');
    for i=1:dim,  
      fprintf(fid,'%%  %-3d %-12s  ',i,sS.comp_names{i});
      n = length(sS.comp_norm{i});         
      for j=1:n, 
        m = sS.comp_norm{i}(j).method;
        s = sS.comp_norm{i}(j).status;
        if strcmp(s,'uninit'), c='i'; 
        elseif strcmp(s,'undone'), c='u'; 
        else c='d';
        end
        fprintf(fid,'%% %s[%s] ',m,c);
      end
      fprintf(fid,'%% \n');
    end
  end

 case 'som_topol', 

  mdim = length(sS.msize);
 
  % level 1
  fprintf(fid,'%% Struct type                           : %s\n',sS.type);
  fprintf(fid,'%% Map grid size                         : ');
  for i = 1:mdim - 1, fprintf(fid,'%% %d x ',sS.msize(i)); end
  fprintf(fid,'%% %d\n', sS.msize(mdim));
  fprintf(fid,'%% Lattice type (rect/hexa)              : %s\n', sS.lattice);
  fprintf(fid,'%% Shape (sheet/cyl/toroid)              : %s\n', sS.shape);

 case 'som_train', 

  % level 1
  fprintf(fid,'%% Struct type                           : %s\n',sS.type);
  fprintf(fid,'%% Training algorithm                    : %s\n',sS.algorithm);
  fprintf(fid,'%% Training data                         : %s\n',sS.data_name);
  fprintf(fid,'%% Neighborhood function                 : %s\n',sS.neigh);
  fprintf(fid,'%% Mask                                  : ');
  dim = length(sS.mask);
  if dim, 
    for i = 1:dim-1, fprintf(fid,'%% %d ',sS.mask(i)); end; 
    fprintf(fid,'%% %d\n',sS.mask(end));
  else fprintf(fid,'%% \n'); end
  fprintf(fid,'%% Initial radius                        : %-6.1f\n',sS.radius_ini);
  fprintf(fid,'%% Final radius                          : %-6.1f\n',sS.radius_fin);
  fprintf(fid,'%% Initial learning rate (alpha)         : %-6.1f\n',sS.alpha_ini);
  fprintf(fid,'%% Alpha function type (linear/inv)      : %s\n',sS.alpha_type);
  fprintf(fid,'%% Training length                       : %d\n',sS.trainlen);
  fprintf(fid,'%% Average quantization error            : %-6.1f\n',sS.qerror);
  fprintf(fid,'%% When training was done                : %s\n',sS.time);

 case 'som_norm', 
  
  % level 1
  fprintf(fid,'%% Struct type                           : %s\n',sS.type);
  fprintf(fid,'%% Normalization method                  : %s\n',sS.method);
  fprintf(fid,'%% Status                                : %s\n',sS.status);

  % level 2
  if level>1, 
    fprintf(fid,'%% Parameters:\n');
    sS.params
  end
 end
end

function [] = html2tex(html_addres,texfile)

tempfile = tempname;
fid = fopen(texfile,'w');
eval(['!lynx -dump ' html_addres ' > ' tempfile]);
fid2 = fopen(tempfile,'r');
while not(feof(fid2))
	line = fgets(fid2);
	line = strcat('%',line);
	fprintf(fid,'%s',line);
end
	fclose(fid);
	fclose(fid2);
	delete (tempfile);


function [name] = retname
resnames = who;
if size(resnames,1) > 0
  max_length = size(resnames{1},2);
  for index = 1:size(resnames,1)
    if size(resnames{index},2) > max_length
	max_length =  size(resnames{index},2);
    end
  end
 length = max_length + 1;
 name(:,1:1:length) = 'A' 				
else
 name = 'A';
end	
%%



