%DPE_RECONSTRUCTION calculates the reconstruction of the generalized median
%   Parameters:
%       object_vector       - euclidean vector of the object to be reconstructed
%       embedding           - euclidean embedding of the objects in the set. MxN Matrix, M Objects in N Dimensions
%       set                 - set of objects corresponding to the embedding as cell array. embedding(i,:) is the vector of object set{i}
%       dist_func           - distance function dist_func(o,p) between objects o and p
%       weighted_mean_func  - weighted mean function weighted_mean_func(o,p,alpha) between objects o and p with ratio alpha
%       method              - reconstruction method
%       best_crit           - minimizing criterium best_crit(set,obj) for the best object if several are possible. Default: SOD
%
%   Returns:
%       rec_obj             - reconstructed object
%       best_value          - value of best_cirt for rec_obj

function [rec_obj, best_value] = dpe_reconstruction( object_vector, embedding, set, dist_func, weighted_mean_func , method, best_crit)

    %% check input

    if (nargin < 7)
        best_crit = @(set,obj) sum(dpe_pdist(set,{obj},dist_func));
    end
   
    if (~isvector(object_vector))
        error('dpe_reconstruction: object_vector has to be a vector!')
    end
    
    if (~ismatrix(embedding))
        error('dpe_reconstruction: embedding has to be a matrix!')
    end
    
    if (~iscell(set))
        error('dpe_reconstruction: set has to be a cell array of objects!')
    end
    
    
    % remove duplicates
    [unique_embedding, unique_set] = dpe_group_elements(embedding, set, @(set) dpe_set_median(set, dist_func));
    

    %% do reconstruction
    switch method
        case 'linear'
            %[rec_obj, best_value] = dpe_linear(object_vector, unique_embedding, unique_set, weighted_mean_func,best_crit);
            [rec_obj, best_value] = dpe_linear(object_vector, embedding, set, weighted_mean_func,best_crit);

        case 'triangular'
           %[rec_obj, best_value] = dpe_triangular(object_vector, unique_embedding, unique_set, weighted_mean_func,best_crit);
            [rec_obj, best_value] = dpe_triangular(object_vector, embedding, set, weighted_mean_func,best_crit);
        case 'recursive'
            valid_result = false;
            counter = 0;
            % try reconstruction several times to avoid random errors in recursive reconstruction
            while (counter < 3 && ~valid_result);
                [rec_obj, best_value, valid_result] = dpe_recursive(object_vector, unique_embedding, unique_set, weighted_mean_func, best_crit);
                %[rec_obj, best_value, valid_result] = dpe_recursive(object_vector, embedding, set, weighted_mean_func, best_crit);
                counter = counter + 1;
            end
        case 'best_recursive'
            valid_result = false;
            counter = 0;
            % try reconstruction several times to avoid random errors in recursive reconstruction
            while (counter < 3 && ~valid_result);
                [rec_obj, best_value, valid_result] = dpe_best_recursive(object_vector, unique_embedding, unique_set, weighted_mean_func, best_crit);
                %[rec_obj, best_value, valid_result] = dpe_best_recursive(object_vector, embedding, set, weighted_mean_func, best_crit);
                counter = counter + 1;
            end
            
        case 'linear_recursive'
            [rec_obj, best_value] = dpe_linear_recursive(object_vector, embedding, set, set, weighted_mean_func, best_crit);

        case 'triangular_recursive'
            [rec_obj, best_value] = dpe_triangular_recursive(object_vector, embedding, set, set, weighted_mean_func, best_crit);
   
        case 'linear_search'
            [rec_obj, best_value] = dpe_linear_search(object_vector, embedding, set, dist_func, weighted_mean_func, best_crit,'start','linear');
            
        case 'linear_search_best_recursive'
            [rec_obj, best_value] = dpe_linear_search(object_vector, embedding, set, dist_func, weighted_mean_func, best_crit,'start','best_recursive');
            
        case 'linear_search_linear_recursive'
            [rec_obj, best_value] = dpe_linear_search(object_vector, embedding, set, dist_func, weighted_mean_func, best_crit,'start','linear_recursive');

      
        otherwise
            error([method ' is not supported']);
        
    end
end

