%% DPE_EMBEDDING is a wrapper for the different embedding methods.
%Parameters:
% set           = vector of cells which contains the items
% k             = destination dimension of the embedding
% dist_func     = distance function ((set x, set x) -> R)
% method        = embedding method
% varargin      = optional parameters for some embedding methods
%
%Returns:
% E             = the embedding vectors as Nxk matrix
% D_orig        = distance matrix of the original objects

function [E,D_orig] = dpe_embedding(set, method, num_dim, dist_func,varargin)

%% check input
    
    if (~iscell(set))
        error('dpe_embedding: set has to be a cell array of objects!')
    end
    
    if (~ischar(method))
        error('dpe_embedding: method has to be a string!')
    end
    
    if (~isscalar(num_dim))
        error('dpe_embedding: num_dim has to be a scalar value!')
    end
    
    
%% Remove double values
    D_cleaned = dpe_pdist(set, dist_func);
    D_orig = D_cleaned;
    size_D = size(D_cleaned,1);

    % remove rows in which there are duplicates and save the indices
    % find zero elements that are in the lower triangle and not on the diagonal
    [row, col] = find(tril(tril(D_cleaned)==0,-1)==1);
    ind = [col,row];
    
    % remove rows if they were found
    if (~isempty(ind))
        %% remove redundancy (e.g. ind = [3 5] [3 6] [5 6] -> [5,6] is redundant)
        % find duplicate row indices
        [n, bin] = histc(ind(:,1), unique(ind(:,1)));
        multiple = find(n > 1);
        index    = find(ismember(bin, multiple));
        remove_ind = [];



        %% iterate duplicate rows and remember them
        for double_row_1 = 1:size(index,1)-1
            for double_row_2 = double_row_1+1:size(index,1)
                for set_row = 1:size(ind,1)
                    if (sum(ind(set_row,:) == [ind(index(double_row_1),2), ind(index(double_row_2),2)]) == 2)
                        remove_ind = [remove_ind set_row];
                    end
                end
            end
        end


        %% remove values        
        ind(remove_ind,:) = [];
        D_cleaned(:,ind(:,2)) = [];
        D_cleaned(ind(:,2),:) = [];
        set(ind(:,2)) = [];
    end

    

%% Do Embedding

% warning if too many dimensions
if (size(D_cleaned,1) < num_dim)
    warning('dpe_embedding: embedding dimension reduced')
    num_dim = size(D_cleaned,1);
end

counter = 0;
max_tries = 1;
E_temp = NaN;
E = NaN;
while (counter < max_tries && any(isnan(E_temp(:))))
    try    

        switch method

            % FastMap
            case 'fastmap'
                E_temp = fastmap(set, dist_func, num_dim);

            % MetricMap
            case 'metricmap'
                E_temp = metricmap(set, dist_func, num_dim);

            % SparseMap
            case 'sparsemap'
                % The parameter optimization results in this setting for
                % sparsemap which is a good compromise between calculationtime
                % and accuracy.
                E_temp = sparsemap(set, dist_func, 0.2, 0.2, num_dim);

            % Multidimensional Scaling
            case 'mds'
                E_temp = mdscale(D_cleaned, num_dim,varargin{:});

            % Nonlinear Sammon mapping
            case 'sammon'
                E_temp = mdscale(D_cleaned,num_dim,'Criterion','sammon',varargin{:});


            % T-SNE
            case 't-sne'
                E_temp = tsne_d(D_cleaned,[],num_dim,varargin{:});
                
            % SPE
            case 'spe'
                E_temp = spe(D_cleaned,num_dim);

            % MVU
            case 'mvu'
                E_temp = mvu(D_cleaned,max(round(num_dim/4),3),'outdim',num_dim,varargin{:})';

            % LLE
            case 'lle'
                E_temp = lle(D_cleaned,round(size(set(:),1)/4),num_dim)';
             
            % IsoMap
            case 'isomap'
                options.verb = 0;
                options.display = 0;
                %options.dims = num_dim;
                %options.distance_mode = 'global';
                E_temp = isomap(D_cleaned,num_dim,options);
                %E_temp = isomap2(D,'k',max(round(num_dim/4),3),options);
                

            case 'cca'
                E_temp = cca(D_cleaned, num_dim, 1000, D_cleaned);
                
            % Prototype Embedding
            case 'prototype'
                E_temp = prototype(set,dist_func,num_dim,varargin{:});

            otherwise
                error(['method ' method ' not found']);

        end
    catch ER
        % catch some mathematical errors and try again
        warning([method ': ' ER.message]);
        counter = counter +1;
    end  
end
    
%% Test if the result is real 
% some methods may create complex results accidentially
if (~isreal(E_temp) || counter == max_tries)
   E = NaN;
   return
end  
    
%% Reinsert Values
if size(E_temp) < size_D
    for num_ind = 1:size(ind,1)
       ind_new = ind - [zeros(size(ind,1),1), ones(size(ind,1),1)];
       E_temp = insertrows(E_temp,E_temp(ind_new(num_ind,1),:),ind_new(num_ind,2));
    end

    % transform insertion indices to account for now missing values
%         num_ind = size(ind,1);
%         ind_new = ind - [zeros(num_ind,1), (1:num_ind)'];
%         E_temp = insertrows(E_temp,E_temp(ind_new(:,1),:),ind_new(:,2));
% 

     if (size(E_temp,1) ~= size_D)
        error('The reconstruction has the wrong number of Elements!')
     end
end

E = E_temp;
    

end

