/*
 * JPhyloIO - Event based parsing and stream writing of multiple sequence alignment and tree formats. 
 * Copyright (C) 2015-2019  Ben Stöver, Sarah Wiechers
 * <http://bioinfweb.info/JPhyloIO>
 * 
 * This file is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package info.bioinfweb.jphyloio.formats.newick;


import info.bioinfweb.jphyloio.ReadWriteConstants;
import info.bioinfweb.jphyloio.events.ConcreteJPhyloIOEvent;
import info.bioinfweb.jphyloio.events.JPhyloIOEvent;
import info.bioinfweb.jphyloio.events.meta.LiteralContentSequenceType;
import info.bioinfweb.jphyloio.events.meta.LiteralMetadataContentEvent;
import info.bioinfweb.jphyloio.events.meta.LiteralMetadataEvent;
import info.bioinfweb.jphyloio.events.meta.URIOrStringIdentifier;
import info.bioinfweb.jphyloio.events.type.EventContentType;
import info.bioinfweb.jphyloio.exception.JPhyloIOReaderException;
import info.bioinfweb.jphyloio.formats.text.TextReaderStreamDataProvider;
import info.bioinfweb.jphyloio.objecttranslation.InvalidObjectSourceDataException;
import info.bioinfweb.jphyloio.objecttranslation.implementations.ListTranslator;

import java.util.Collection;
import java.util.LinkedList;

import javax.xml.namespace.QName;



/**
 * Reads meta information from a hot comment in a Newick string like it is generated by MrBayes or BEAST.
 * 
 * @author Ben St&ouml;ver
 */
public class HotCommentDataReader implements NewickConstants, ReadWriteConstants {
	private static class Value {
		public String stringValue;
		public Object objectValue;
		
		public Value(String stringValue, Object objectValue) {
			super();
			this.stringValue = stringValue;
			this.objectValue = objectValue;
		}
		
		public Value(String stringValue) {
			this(stringValue, stringValue);
		}
	}
	
	
	private Value readTextElementData(String text) {
		char nameDelimiter = ' ';
		if (text.startsWith(Character.toString(NAME_DELIMITER)) && text.endsWith(Character.toString(NAME_DELIMITER))) {
			nameDelimiter = NAME_DELIMITER;
		}
		else if (text.startsWith(Character.toString(ALTERNATIVE_NAME_DELIMITER)) 
				&& text.endsWith(Character.toString(ALTERNATIVE_NAME_DELIMITER))) {
			nameDelimiter = ALTERNATIVE_NAME_DELIMITER;
		}
		if (nameDelimiter != ' ') {  // The implementation would also be able to read single delimiters inside strings. Since there is no complete formal definition for metacomments, it is not clear if and how delimiters should actually be masked inside a string. 
			return new Value(text.substring(1, text.length() - 1).replaceAll(  // Values like "100" should also be read as a string.
					"\\" + nameDelimiter + "\\" + nameDelimiter, "" + nameDelimiter));  // Replace e.g. 'A''B'.
		}
		else {
			try {  //TODO Should parsing a long be tried before?
				return new Value(text, new Double(text));
			}
			catch (NumberFormatException e) {
				return new Value(text);
			}
		}
	}
	
	
	private int findNameEnd(String text, int pos, char nameDelimiter) {
		pos++;  // Skip leading name delimiter.
		while (pos < text.length()) {
			if (text.charAt(pos) == nameDelimiter) {
				return pos;
			}
			else {
				pos++;
			}
		}
		return -1;
	}
	
	
	private int findFieldEnd(String text, int pos) {
		while (pos < text.length()) { 
			switch (text.charAt(pos)) {
				case NAME_DELIMITER:
				case ALTERNATIVE_NAME_DELIMITER:
					pos = findNameEnd(text, pos, text.charAt(pos));
					if (pos == -1) {
						return -1;
					}
					break;
				case FIELD_END_SYMBOL:
					return pos;
			}
			pos++;  // Also skips found name end
		}
		return -1;
	}
	
	
	private int findAllocationEnd(String text, int start) {
		if (start >= text.length()) {
			return -1;
		}
		else {
			int pos = start;
			while (pos < text.length()) {
				switch (text.charAt(pos)) {
					case FIELD_START_SYMBOL:
						pos = findFieldEnd(text, pos);
						break;
					case NAME_DELIMITER:
					case ALTERNATIVE_NAME_DELIMITER:
						pos = findNameEnd(text, pos, text.charAt(pos));
						break;
					case ALLOCATION_SEPARATOR_SYMBOL:
						return pos;
				}
				if (pos == -1) {
					return -1;
				}
				pos++;  // Also skips found field or name end.
			}
			return pos;
		}
	}
	
	
	private void addLiteralMetaStart(String key, QName predicate, URIOrStringIdentifier dataType, 
			TextReaderStreamDataProvider<?> streamDataProvider,	Collection<JPhyloIOEvent> eventQueue) {
		
		eventQueue.add(new LiteralMetadataEvent(DEFAULT_META_ID_PREFIX + streamDataProvider.getIDManager().createNewID(), 
				key, new URIOrStringIdentifier(key, predicate), dataType, LiteralContentSequenceType.SIMPLE));
	}
	
	
	private void addLiteralMetaContent(Value value, Collection<JPhyloIOEvent> eventQueue) {
		eventQueue.add(new LiteralMetadataContentEvent(value.objectValue, value.stringValue));
	}
	
	
	private void addLiteralMetaEnd(Collection<JPhyloIOEvent> eventQueue) {
		eventQueue.add(ConcreteJPhyloIOEvent.createEndEvent(EventContentType.LITERAL_META));
	}
	
	
	/**
	 * Processed comments according to 
	 * <a href="https://code.google.com/archive/p/beast-mcmc/wikis/NexusMetacommentFormat.wiki">this</a> definition.
	 * 
	 * @param comment
	 * @param eventQueue
	 * @throws JPhyloIOReaderException 
	 */
	private void processMetacomments(final String comment, TextReaderStreamDataProvider<?> streamDataProvider, 
			Collection<JPhyloIOEvent> eventQueue) throws JPhyloIOReaderException {
		
		ListTranslator translator = new ListTranslator();  //TODO Could be a static class field.
		int start = 1;
		int end = findAllocationEnd(comment, start);
		while (end != -1) {
			String allocation = comment.substring(start, end);
			int allocationSymbolPos = allocation.indexOf(ALLOCATION_SYMBOL);  // split() cannot be used, since text values may contain additional '='.
			if (allocationSymbolPos >= 0) {
				String valueText = allocation.substring(allocationSymbolPos + 1).trim();
				Value value;
				URIOrStringIdentifier dataType = null;
				if (valueText.startsWith("" + FIELD_START_SYMBOL) && valueText.endsWith("" + FIELD_END_SYMBOL)) {
					try {
						value = new Value(valueText, translator.representationToJava(valueText, streamDataProvider));
						dataType = new URIOrStringIdentifier(null, DATA_TYPE_NEWICK_ARRAY);
					}
					catch (InvalidObjectSourceDataException e) {
						throw new JPhyloIOReaderException("The following excpetion occurred when trying to read a list from a Newick hot comment: " + 
								e.getMessage(), streamDataProvider.getDataReader(), e);
					}
				}
				else {
					value = readTextElementData(valueText);
				}
				addLiteralMetaStart(allocation.substring(0, allocationSymbolPos).trim(), PREDICATE_HAS_LITERAL_METADATA, dataType, streamDataProvider, eventQueue);
				addLiteralMetaContent(value, eventQueue);
				addLiteralMetaEnd(eventQueue);
			}
			
			start = end + 1;
			end = findAllocationEnd(comment, start);
		}
	}
	
	
	/**
	 * Processes comments according to 
	 * <a href="https://sites.google.com/site/cmzmasek/home/software/forester/nhx">this</a> definition.
	 * 
	 * @param comment the text of the hot comment without the comment start or end tokens
	 * @param streamDataProvider the stream data provider associated with the parent event reader
	 */
	private void processNHX(final String comment, TextReaderStreamDataProvider<?> streamDataProvider, Collection<JPhyloIOEvent> eventQueue) {
		Collection<JPhyloIOEvent> newEvents = new LinkedList<JPhyloIOEvent>();
		//TODO Was it necessary to buffer events?
		String[] parts = comment.substring(NHX_START.length()).split("" + NHX_VALUE_SEPARATOR_SYMBOL);
		for (int i = 0; i < parts.length; i++) {
			int splitPos = parts[i].indexOf(ALLOCATION_SYMBOL);
			if (splitPos > 0) {
				String key = parts[i].substring(0, splitPos);
				addLiteralMetaStart(NHX_KEY_PREFIX + key, NHXUtils.getInstance().predicateByKey(key), null, streamDataProvider, newEvents);
				addLiteralMetaContent(readTextElementData(parts[i].substring(splitPos + 1, parts[i].length())), newEvents);
				addLiteralMetaEnd(newEvents);
			}
			else {  // If the part starts with '=' or there is no '='.
				throw new IllegalArgumentException("\"" + parts[i] + "\" is not a legal NHX metadata definition.");
			}
		}
		eventQueue.addAll(newEvents);
	}
	
	
	/**
	 * Reads metadata from the specified hot comment and adds according events to the queue.
	 * 
	 * @param comment the text of the hot comment without the comment start or end tokens
	 * @param streamDataProvider the stream data provider associated with the parent event reader
	 * @param isOnNode Specifies {@code true} here, if the hot comment was attached to a node or
	 *        {@code false} if it was attached to an edge
	 * @throws IllegalArgumentException if the specified comment cannot be parsed in the TreeAnnotator or the NHX format.
	 * @throws JPhyloIOReaderException 
	 */
	public void read(String comment, TextReaderStreamDataProvider<?> streamDataProvider, Collection<JPhyloIOEvent> eventQueue, boolean isOnNode) throws IllegalArgumentException, JPhyloIOReaderException {
		if (comment.startsWith(NHX_START)) {  // Needs to be checked first.
			processNHX(comment, streamDataProvider, eventQueue);
		}
		else if (comment.startsWith("" + HOT_COMMENT_START_SYMBOL)) {
			processMetacomments(comment, streamDataProvider, eventQueue);
		}
		// The following case is currently unused, because JPhyloIO creates comment events from unnamed hot comments 
		// and its up to the application, whether these shall be interpreted as metainformation or not.
//		else if (comment.length() > 0) {  // Read unformatted comment
//			String name = UNNAMED_EDGE_DATA_NAME;
//			if (isOnNode) {
//				name = UNNAMED_NODE_DATA_NAME;
//			}
//			addMetaInformation(name, readTextElementData(comment), eventQueue, true);
//		}
	}
}