/*
 * JPhyloIO - Event based parsing and stream writing of multiple sequence alignment and tree formats. 
 * Copyright (C) 2015-2019  Ben Stöver, Sarah Wiechers
 * <http://bioinfweb.info/JPhyloIO>
 * 
 * This file is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package info.bioinfweb.jphyloio;


import info.bioinfweb.commons.log.ApplicationLogger;
import info.bioinfweb.jphyloio.dataadapters.DocumentDataAdapter;
import info.bioinfweb.jphyloio.dataadapters.MatrixDataAdapter;
import info.bioinfweb.jphyloio.dataadapters.OTUListDataAdapter;
import info.bioinfweb.jphyloio.events.LinkedLabeledIDEvent;
import info.bioinfweb.jphyloio.formats.text.AbstractTextEventWriter;
import info.bioinfweb.jphyloio.formats.text.TextWriterStreamDataProvider;

import java.io.IOException;
import java.io.Writer;
import java.util.Iterator;



/**
 * Default implementation for writers that can only write a single matrix (multiple sequence alignment) to their
 * target format, because that format does not support additional data.
 * <p>
 * This abstract implementation will log according warnings for the other data that has not been written and than
 * calls {@link #writeSingleMatrix(DocumentDataAdapter, MatrixDataAdapter, Iterator, Writer, ReadWriteParameterMap)}
 * which must be implemented by inherited classes accordingly.
 * 
 * @author Ben St&ouml;ver
 * @since 0.0.0
 */
public abstract class AbstractSingleMatrixEventWriter<P extends TextWriterStreamDataProvider<? extends AbstractTextEventWriter<P>>> 
		extends AbstractTextEventWriter<P> {
	
	private String formatName;
	

	/**
	 * Creates a new instance of this class.
	 * 
	 * @param formatName the name of the format the inherited implementation writes to, which will be used for the log
	 *        messages generated by this abstract implementation.
	 */
	public AbstractSingleMatrixEventWriter(String formatName) {
		super();
		this.formatName = formatName;
	}
	
	
	/**
	 * Implementations of this method should write a single matrix to their target format.
	 * 
	 * @param document the document containing the matrix
	 * @param matrix the (non-empty) matrix to be written
	 * @param sequenceIDIterator an iterator oder all sequences in {@code matrix} positioned before the first sequence
	 * @param writer the writer to write the data to
	 * @param parameters the parameter map for the writer implementation
	 * @throws Exception if the implementing writer throws an exception
	 */
	protected abstract void writeSingleMatrix(DocumentDataAdapter document, MatrixDataAdapter matrix, 
			Iterator<String> sequenceIDIterator,	ReadWriteParameterMap parameters) throws IOException;


	protected String editSequenceOrNodeLabel(final LinkedLabeledIDEvent event, final ReadWriteParameterMap parameters, 
			OTUListDataAdapter otuList) {
		
		return createUniqueLinkedOTULabel(parameters,
				new UniqueLabelHandler() {
					@Override
					public boolean isUnique(String label) {
						return !parameters.getLabelEditingReporter().isLabelUsed(event.getType().getContentType(), label);
					}

					@Override
					public String editLabel(String label) {
						return maskReservedLabelCharacters(label);
					}
				}, 
				event, otuList, false);  // Already considers possible maximum length.
	}
	
	
	protected abstract String maskReservedLabelCharacters(String label);

	
	@Override
	protected void doWriteDocument(DocumentDataAdapter document, Writer writer,	ReadWriteParameterMap parameters) throws IOException {
		super.doWriteDocument(document, writer, parameters);
		
		ApplicationLogger logger = parameters.getLogger();
		logIngnoredOTULists(document, logger, parameters, formatName, "sequences");
		Iterator<MatrixDataAdapter> matrixIterator = document.getMatrixIterator(parameters);
		if (matrixIterator.hasNext()) {
			MatrixDataAdapter matrixDataAdapter = matrixIterator.next();
			Iterator<String> sequenceIDIterator = matrixDataAdapter.getSequenceIDIterator(parameters);
			if (sequenceIDIterator.hasNext()) {
				writeSingleMatrix(document, matrixDataAdapter, sequenceIDIterator, parameters);
			}
			else {
				logger.addWarning("An empty " + formatName + 
						" file was written since the first matrix model adapter did not provide any sequences.");
			}
			
			if (matrixIterator.hasNext()) {
				logger.addWarning("The specified document adapter contained more than one character matrix adapter. Since the "
						 + formatName	+ " format does not support multiple alignments in one file, only the first matrix was written.");
			}
		}
		else {
			logger.addWarning("An empty " + formatName + 
					" file was written since the specified document adapter contained contained no matrices.");
		}
		
		if (document.getTreeNetworkGroupIterator(parameters).hasNext()) {
			logger.addWarning("The specified tree or network definitions will not be written, since the " + formatName + 
					" format does not support this."); 
		}
	}	
}
