/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.syscir.engine.typetransformer;

import java.io.IOException;
import java.util.LinkedList;
import java.util.List;

import javax.xml.parsers.ParserConfigurationException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import de.tub.pes.syscir.engine.Engine;
import de.tub.pes.syscir.engine.Environment;
import de.tub.pes.syscir.engine.util.IOUtil;
import de.tub.pes.syscir.sc_model.SCClass;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.variables.SCKnownType;

public class AbstractTypeTransformer implements KnownTypeTransformer {

	private static Logger logger = LogManager.getLogger(AbstractTypeTransformer.class
			.getName());

	protected String name = null;
	protected String impl = null;

	public String getImplementation() {
		return impl;
	}

	public void setImplementation(String file) {
		this.impl = file;
	}

	public void createType(Environment e) {
		// check if it has been created yet
		if (!existsType(name, e)) {
			// there are two possibilities: either the type has a dedicated
			// implementation
			if (impl != null) {
				// in this case we have to read the implementation
				try {
					Document doc = IOUtil.readXML(impl);
					// Parsing the implementation file
					Environment temp = Engine.parseImplementation(doc);
					// Integrating the newly parsed environment into the already
					// existing
					e.integrate(temp);

				} catch (SAXException exc) {
					exc.printStackTrace();
				} catch (IOException exc) {
					exc.printStackTrace();
				} catch (ParserConfigurationException exc) {
					exc.printStackTrace();
				}
			}
			// or it has no dedicated implementation
			// we have to handle this case in the specific knownType classes.
		}
	}
	
	/**
	 * Usually types generated by type transformers are used as known types. Overwrite this method if necessary.
	 * @param name
	 * @param e
	 * @return
	 */
	public boolean existsType(String name, Environment e) {
		return e.getKnownTypes().containsKey(name);
	}

	@Override
	public SCKnownType createInstance(String instName, Environment e,
			boolean stat, boolean cons, List<String> other_mods) {
		SCClass type = e.getKnownTypes().get(name);
		SCKnownType kt = null;
		if (type != null) {
			kt = new SCKnownType(instName, type, e.getCurrentClass(), null,
					stat, cons, other_mods, e.getLastInitializer());
		} else {
			logger.error(
					"Configuration error: type {} cannot be instantiated.",
					name);
		}
		return kt;
	}

	public SCKnownType initiateInstance(String instName,
			List<Expression> params, Environment e, boolean stat, boolean cons,
			List<String> other_mods) {
		SCClass type = e.getKnownTypes().get(name);
		SCKnownType kt = null;
		if (type != null) {
			if (e.getKnownTypes().get(name).getConstructor() != null
					&& e.getKnownTypes().get(name).getConstructor()
							.getParameters().size() != params.size()) {
				logger.error("ERROR: "
						+ this
						+ ": not the right number of parameters to initiate "
						+ instName
						+ ". Expected was "
						+ e.getKnownTypes().get(name).getConstructor()
								.getParameters().size()
						+ " parameters but got " + params.size() + ".");
				return null;
			} else {
				kt = new SCKnownType(instName, type, e.getCurrentClass(),
						new LinkedList<Expression>(), stat, cons, other_mods,
						e.getLastInitializer());
			}
		} else {
			logger.error(
					"Configuration error: cannot initiate an instance of type {}.",
					name);
		}
		return kt;
	}

}
