/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.util;

import java.io.FileNotFoundException;
import java.io.IOException;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.ExpressionBlock;

/**
 * Utility class to write an UPPAAL Model. The writer can either write tags
 * ("<...>") or text (content between tags). Special chars in text get escaped.
 * Terminology: empty tag = "<myTag .../>" start tag = "<myTag ...>"; start tag
 * (not closed) = "<myTag ..."; end tag = "</myTag>";
 * 
 * @author rschroeder
 * 
 */
public class XMLWriter extends SC2uppaalFileWriter {

	public static String getStdUppaalHeader() {
		return "<?xml version='1.0' encoding='utf-8'?>"
				+ Constants.END_LINE
				+ "<!DOCTYPE nta PUBLIC '-//Uppaal Team//DTD Flat System 1.1//EN' 'http://www.it.uu.se/research/group/darts/uppaal/flat-1_1.dtd'>";
	}

	public XMLWriter(String fileName) throws FileNotFoundException {
		super(fileName);
	}

	private XMLWriter writeCloseTagSymbol() throws IOException {
		getWriter().write(">");
		return this;
	}

	private XMLWriter writeOpenTagSymbol() throws IOException {
		getWriter().write("<");
		return this;
	}

	private XMLWriter writeEndTagSymbol() throws IOException {
		getWriter().write("/");
		return this;
	}

	public XMLWriter writeStdUppaalHeader() throws IOException {
		getWriter().write(getStdUppaalHeader());
		return this;
	}

	public XMLWriter startTag(String tagContent, boolean closeCurrentTag)
			throws IOException {
		writeOpenTagSymbol();
		getWriter().write(tagContent);
		if (closeCurrentTag) {
			writeCloseTagSymbol();
			getWriter().write(Constants.END_LINE);
		}
		return this;
	}

	public XMLWriter startTag(String tagContent) throws IOException {
		return startTag(tagContent, true);
	}

	public XMLWriter endTag(String tagContent) throws IOException {
		writeOpenTagSymbol();
		writeEndTagSymbol();
		getWriter().write(tagContent);
		writeCloseTagSymbol();
		getWriter().write(Constants.END_LINE);
		return this;
	}

	public XMLWriter writeEmptyTag(String tagContent) throws IOException {
		startTag(tagContent, false);
		writeEndTagSymbol();
		writeCloseTagSymbol();
		getWriter().write(Constants.END_LINE);
		return this;
	}

	public XMLWriter writeText(String text, boolean addNewLine)
			throws IOException {
		getWriter().write(escape(text));
		if (addNewLine) {
			getWriter().write(Constants.END_LINE);
		}
		return this;
	}

	public XMLWriter writeText(String text) throws IOException {
		return writeText(text, false);
	}

	public XMLWriter writeText(Expression expr, boolean addNewLine,
			boolean breakLogicalOperators) throws IOException {
		String out = expr.toStringNoSem();
		if (breakLogicalOperators) {
			// insert line breaks for logical operators in order to break guards
			out = out.replace(" && ", " && " + Constants.END_LINE)
					.replace(" || ", " || " + Constants.END_LINE)
					.replace(" & ", " & " + Constants.END_LINE)
					.replace(" | ", " | " + Constants.END_LINE);

		}
		if (expr instanceof ExpressionBlock) {
			out = out.replace("\n", ",\n");
		}
		writeText(out, addNewLine);
		return this;
	}

	public XMLWriter writeText(Expression expr) throws IOException {
		return writeText(expr, false, false);
	}

	public XMLWriter writeTextLine(String text) throws IOException {
		return writeText(text, true);
	}

	public XMLWriter writeTextLine(Expression expr) throws IOException {
		return writeText(expr, true, false);
	}

	public XMLWriter writeTextLine(Expression expr,
			boolean breakLogicalOperators) throws IOException {
		return writeText(expr, true, breakLogicalOperators);
	}

	public XMLWriter writeTextLine() throws IOException {
		return writeText("");
	}

	public XMLWriter writeHeader(String text) throws IOException {
		XMLWriter ret = writeText("\n");
		ret = writeText("///////////////////\n");
		ret = ret.writeText("// " + text + "\n");
		ret = ret.writeText("///////////////////\n");
		ret = writeText("\n");
		return ret;
	}

	public XMLWriter writeExpressionInfo(Expression expr) throws IOException {
		getWriter().write(getExpressionInfo(expr));
		return this;
	}

	public static String escape(String in) {
		String out = in;
		out = out.replaceAll("&", "&amp;");
		out = out.replace("<", "&lt;");
		out = out.replace(">", "&gt;");
		return out;
	}

	public static String getExpressionInfo(Expression expr) {
		String out = escape("nodeID: " + expr.getNodeId() + ", file: "
				+ expr.getFile() + ", line: " + expr.getLine());
		return out;
	}
}
