/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.engine.Engine;
import de.tub.pes.syscir.analysis.VariableUsageAnalyzer;
import de.tub.pes.syscir.sc_model.SCFunction;
import de.tub.pes.syscir.sc_model.SCParameter;
import de.tub.pes.syscir.sc_model.SCSystem;
import de.tub.pes.syscir.sc_model.SCVariable;
import de.tub.pes.syscir.sc_model.expressions.ArrayAccessExpression;
import de.tub.pes.syscir.sc_model.expressions.BinaryExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableExpression;
import de.tub.pes.syscir.sc_model.variables.SCArray;
import de.tub.pes.syscir.sc_model.variables.SCPointer;

/**
 * 
 * @author Timm Liebrenz
 *
 */
public class VariableConverter {

	private static Logger logger = LogManager.getLogger(VariableConverter.class
			.getName());
	private static VariableUsageAnalyzer vua = null;

	/**
	 * Private constructor to prohibit instantiation via default constructor.
	 */
	private VariableConverter() {
	}

	public static void init(SCSystem system) {
		vua = new VariableUsageAnalyzer();
		vua.analyze(system);
	}

	public static TAVariable getTAVarFromType(String name, String type) {
		if (type.equals("int")) {
			return new TAInteger(name);
		} else if (type.equals("bool")) {
			return new TABoolean(name);
		}
		return new TAGenericTypeVariable(name, type);
	}

	public static TAVariable getTmplParFromFuncPar(SCParameter par,
			String prefix) {
		SCVariable scvar = par.getVar();
		SCFunction func = par.getFunction();
		String name = getTemplParamName(func.getName(), scvar.getName(), prefix);
		if (scvar instanceof SCPointer) {
			// if the param is a pointer, the transport var is an addr ptr
			return getAddrPtr(name, scvar.getTypeWithoutSize());
		} else {
			return getTAVarFromType(name, scvar.getTypeWithoutSize());
		}
	}

	public static TAVariable getTrnsprtVarForFuncReturn(SCFunction func,
			String prefix) {
		if (func == null) {
			//fix for optimized method transformer
			return null;
		}
		String retType = func.getReturnType();
		String name = getTrnsprtVarNameForFuncReturn(func, prefix);
		if (!retType.equals("void")) {
			if (retType.indexOf("*") != -1) {
				retType = retType.replace("*", "");
				return VariableConverter.getAddrPtr(name, retType);
			}
			return getTAVarFromType(name, retType);
		}
		return null;
	}

	public static TAVariable getTrnsprtVarForFuncReturn(SCFunction func) {
		return getTrnsprtVarForFuncReturn(func, "");
	}

	public static String getTrnsprtVarNameForFuncReturn(SCFunction func,
			String prefix) {
		return prefix + func.getName() + Constants.PREFIX_DELIMITER
				+ Constants.LOCAL_FUNCTION_RETURN_KEYWORD;
	}

	public static String getTrnsprtVarNameForFuncReturn(SCFunction func) {
		return getTrnsprtVarNameForFuncReturn(func, "");
	}

	public static TAVariable getAddrPtr(String name, String type) {
		return new TAAddressPointer(name, type);
	}

	public static TAVariable getAddrPtr(SCVariable scvar) {
		String name = getAddrPtrName(scvar);
		return getAddrPtr(name, scvar.getTypeWithoutSize());
	}

	public static String getAddrPtrName(SCVariable scvar) {
		return scvar.getName();
	}

	public static String getAddrPtrType(String type) {
		return TAAddressPointer.typeAsAddrPtr(type);
	}

	public static Expression derefAddrPtr(SCVariable scvar, Expression offset) {
		String type = scvar.getTypeWithoutSize();
		SCArray memArr = new SCArray(TAMemoryType.getMemArrName(type), type);
		Expression access = new SCVariableExpression(null, scvar);
		if (offset != null && !"".equals(offset.toStringNoSem())) {
			access = new BinaryExpression(null, access, "+", offset);
		}
		Expression ret = new ArrayAccessExpression(null, memArr, access);
		return ret;
	}

	public static Expression derefAddrPtr(SCVariable scvar, String offset) {
		return derefAddrPtr(scvar, new ConstantExpression(null, offset));
	}

	public static Expression derefAddrPtr(SCVariable scvar) {
		return derefAddrPtr(scvar, "");
	}

	public static String getTemplParamName(String funcName, String parName,
			String prefix) {
		return prefix + funcName + Constants.PREFIX_DELIMITER
				+ Constants.LOCAL_FUNCTION_PARAM_KEYWORD
				+ Constants.PREFIX_DELIMITER + parName;
	}

	public static String getTemplParamName(String funcName, String parName) {
		return getTemplParamName(funcName, parName, "");
	}

	public static TAVariable convertVariable(SCVariable scvar) {
		if (scvar instanceof SCPointer || useMemModel(scvar)) {
			return getAddrPtr(scvar);
		} else {
			TAVariable tavar = getTAVarFromType(scvar.getName(),
					scvar.getTypeWithoutSize());
			tavar.setConstant(scvar.isConst());
			return tavar;
		}
	}

	/**
	 * Don't allocate space for certain vars, depending on class and type.
	 * 
	 * @param scvar
	 * @return
	 */
	public static boolean useMemModel(SCVariable scvar) {
		if (vua == null) {
			logger.error("Call init() before using this function to enable variable reference analysis.");
		}
		if (scvar.isSCModule() || scvar.isChannel()) {
			return false;
		} else if (scvar instanceof SCPointer || scvar.isConst()) {
			return false;
		} else if (TAMemoryType.isForbiddenType(scvar.getTypeWithoutSize())) {
			return false;
		} else if (Engine.ALWAYS_USE_MEM_MODEL) {
			return true;
		} else if (vua.getVars().contains(scvar)) {
			return false;
		}
		return true;
	}
}
