/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.io.IOException;

import de.tub.pes.sc2uppaal.tamodel.expressions.TAVariableAssignmentExpression;
import de.tub.pes.sc2uppaal.util.XMLWriter;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;

/**
 * 
 * @author Joachim Fellmuth, Paula Herber
 * 
 */

public abstract class TAVariable {

	/**
	 * Name of the variable.
	 */
	protected String name;

	/**
	 * The variable type. Any type string is allowed that would work as a type
	 * declaration in Uppaal, including interval boundaries of integer
	 * variables.
	 */
	protected String type;

	protected Expression initValue;

	protected boolean isConstant;

	protected boolean isReference;

	/**
	 * Marks this variable as volatile. Can be used to control optimizer
	 * influence on this variables state.
	 */
	protected boolean isVolatile;

	public TAVariable(String name, String type, Expression initValue) {
		super();
		this.name = name;
		this.type = type;
		this.initValue = initValue;
		this.isConstant = false;
		this.isReference = false;
		this.isVolatile = false;
	}

	public TAVariable(String name, String type) {
		this(name, type, null);
	}

	public Expression getInitValue() {
		return initValue;
	}

	public void setInitValue(Expression initValue) {
		this.initValue = initValue;
	}

	public void setInitValue(String initValue) {
		this.initValue = new ConstantExpression(null, initValue);
	}

	public boolean hasInitValue() {
		return (initValue != null);
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getName() {
		return name;
	}

	/**
	 * @return The type string of this variable.
	 */
	public String getType() {
		return type;
	}

	/**
	 * Sets a new type string.
	 * 
	 * @param type
	 */
	public void setType(String type) {
		this.type = type;
	}

	public abstract String getDummyName(TAModel ta);

	/**
	 * Prints the variable. This method is used when creating the declaration of
	 * the TAVariable.
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void print(XMLWriter writer) throws IOException {
		String output = createDeclarationString();
		writer.writeText(output);
	}

	/**
	 * Returns a printable representation of the Variable which can be used in
	 * inside declarations. Simple variables (as int or bool) should return the
	 * same string which is printed when calling
	 * {@link TAVariable#print(XMLWriter)}.
	 */
	public String createDeclarationString() {
		StringBuffer out = new StringBuffer();
		out.append(isConst2Str());
		out.append(getType());
		out.append(" ");
		out.append(getName());
		if (hasInitValue()) {
			out.append(" = " + getInitValue().toStringNoSem());
		}
		out.append(";" + Constants.END_LINE);
		return out.toString();
	}

	public void printAsParam(XMLWriter writer) throws IOException {
		StringBuffer out = new StringBuffer();
		out.append(isConst2Str()).append(getType()).append(" ")
				.append(getRefParamPrefix()).append(getName());
		writer.writeText(out.toString());
	}

	protected String getRefParamPrefix() {
		return (isConstant ? "" : "&"); // if const, dont use references
	}

	public abstract TAVariable createCopy(String prefix);

	protected void copyAttributesExceptName(TAVariable other) {
		other.type = type;
		other.initValue = initValue;
		other.isConstant = isConstant;
		other.isReference = isReference;
	}

	/**
	 * Returns an Expression which resets the variable to a predefined value.
	 * 
	 * @return
	 */
	public Expression getResetExpr() {
		Expression expr = new TAVariableAssignmentExpression(null, this,
				Constants.getDefaultValueName(type));
		return expr;
	}

	public boolean isConstant() {
		return isConstant;
	}

	public void setConstant(boolean isConstant) {
		this.isConstant = isConstant;
	}

	protected String isConst2Str() {
		return (isConstant ? "const " : "");
	}

	public boolean isReference() {
		return isReference;
	}

	public void setReference(boolean isReference) {
		this.isReference = isReference;
	}

	public boolean isVolatile() {
		return isVolatile;
	}

	public void setVolatile(boolean isVolatile) {
		this.isVolatile = isVolatile;
	}

	@Override
	public String toString() {
		return name;
	}

	public static String addPrefix(String name, String prefix) {
		if (!("".equals(prefix))) {
			String myDelim = Constants.PREFIX_DELIMITER;
			if (prefix.lastIndexOf(Constants.PREFIX_DELIMITER) == prefix
					.length() - 1) {
				// prevent double delimiter
				myDelim = "";
			}
			name = prefix + myDelim + name;
		}
		return name;
	}

	public void addPrefix(String prefix) {
		this.name = addPrefix(this.name, prefix);
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((initValue == null) ? 0 : initValue.hashCode());
		result = prime * result + (isConstant ? 1231 : 1237);
		result = prime * result + (isReference ? 1231 : 1237);
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		result = prime * result + ((type == null) ? 0 : type.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		TAVariable other = (TAVariable) obj;
		if (initValue == null) {
			if (other.initValue != null)
				return false;
		} else if (!initValue.equals(other.initValue))
			return false;
		if (isConstant != other.isConstant)
			return false;
		if (isReference != other.isReference)
			return false;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		if (type == null) {
			if (other.type != null)
				return false;
		} else if (!type.equals(other.type))
			return false;
		return true;
	}

}
