/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.io.IOException;
import java.util.LinkedList;
import java.util.List;

import de.tub.pes.sc2uppaal.tamodel.expressions.TASendRecvExpression;
import de.tub.pes.sc2uppaal.util.XMLWriter;
import de.tub.pes.syscir.sc_model.expressions.Expression;

/**
 * A transition in Uppaal is an edge in a TA template. It is connects two
 * locations and is directed. It can also contain four different labels, which
 * manipulate variables or the control flow. See the Uppaal documentation for
 * more details on transitions.
 * 
 * @author Joachim Fellmuth, Paula Herber, Marcel Pockrandt
 * 
 */
public class TATransition {
	/**
	 * Start location of the transition.
	 */
	private TALocation start;

	/**
	 * End location of the transition.
	 */
	private TALocation end;

	/**
	 * A List of selection expressions to generate transient random variables.
	 */
	private List<Expression> select;

	/**
	 * The guard expression. Must evaluate to an integer variable. No
	 * side-effects allowed.
	 */
	private Expression guard = null;

	/**
	 * The synchronization expression. Must evaluate to a channel variable. No
	 * side-effects allowed.
	 */
	private TASendRecvExpression sync = null;

	/**
	 * A list of update expressions.
	 */
	private List<Expression> update;

	/**
	 * Standard constructor. Be aware that the start and end location MUST be
	 * set at some point.
	 */
	public TATransition() {
		select = new LinkedList<Expression>();
		update = new LinkedList<Expression>();

	}

	/**
	 * Creates a regular transition that connects the two given location.
	 * 
	 * @param start
	 * @param end
	 */
	public TATransition(TALocation start, TALocation end) {
		this();
		this.start = start;
		this.end = end;
	}

	/**
	 * Creates a transition that connects the two given locations and set the
	 * given expression as synchronization label. Lazy constructor for the most
	 * common transition type.
	 * 
	 * @param start
	 * @param end
	 * @param sync
	 *            Synchronization label.
	 */
	public TATransition(TALocation start, TALocation end,
			TASendRecvExpression sync) {
		this(start, end);
		this.sync = sync;
	}

	/**
	 * @return Starting location
	 */
	public TALocation getStart() {
		return start;
	}

	/**
	 * Sets starting location.
	 * 
	 * @param start
	 */
	public void setStart(TALocation start) {
		this.start = start;
	}

	/**
	 * @return End location
	 */
	public TALocation getEnd() {
		return end;
	}

	/**
	 * Sets end location.
	 * 
	 * @param end
	 */
	public void setEnd(TALocation end) {
		this.end = end;
	}

	/**
	 * Returns the guard label expression. Null if not set.
	 * 
	 * @return Guard
	 */
	public Expression getGuard() {
		return guard;
	}

	/**
	 * Sets the guard label expression. Must evaluate to integer and be
	 * side-effect free.
	 * 
	 * @param guard
	 */

	public void setGuard(Expression guard) {
		this.guard = guard;
	}

	/**
	 * Returns the selection label expression list.
	 * 
	 * @return Select expression list.
	 */
	public List<Expression> getSelect() {
		return select;
	}

	/**
	 * Adds a single select expression to the select expression list. See Uppaal
	 * documentation for details.
	 * 
	 * @param expression
	 */
	public void addSelectExpression(Expression expression) {
		select.add(expression);
	}

	/**
	 * Returns the synchronization label expression. Null if not set.
	 * 
	 * @return synchronization expression
	 */
	public TASendRecvExpression getSync() {
		return sync;
	}

	/**
	 * Sets the synchronization expression. Must evaluate to a channel variable
	 * and be side-effect free.
	 * 
	 * @param sync
	 */
	public void setSync(TASendRecvExpression sync) {
		this.sync = sync;
	}

	/**
	 * Returns the update label expression list.
	 * 
	 * @return Update
	 */
	public List<Expression> getUpdate() {
		return update;
	}

	/**
	 * Adds an update expression to the end of the update expression list of
	 * this transition.
	 * 
	 * @param exp
	 */
	public void addUpdateExpression(Expression exp) {
		((LinkedList<Expression>) this.update).addLast(exp);
	}

	/**
	 * Adds an update expression at the specified index to the update expression
	 * list of this transition.
	 * 
	 * @param exp
	 * @param index
	 */
	public void addUpdateExpression(Expression exp, int index) {
		((LinkedList<Expression>) this.update).add(index, exp);
	}

	/**
	 * Removes the given update expression
	 * 
	 * @param exp
	 */
	public void deleteUpdateExpression(Expression exp) {
		((LinkedList<Expression>) this.update).remove(exp);
	}

	/**
	 * Adds an update expression to the front of the update expression list of
	 * this transition.
	 * 
	 * @param exp
	 */
	public void addUpdateExpressionFront(Expression exp) {
		((LinkedList<Expression>) this.update).addFirst(exp);
	}

	/**
	 * Prints the transition data to the given stream writer. Format is XML,
	 * Uppaal modeling language.
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void print(XMLWriter writer) throws IOException {
		writer.startTag("transition");
		writer.writeEmptyTag("source ref=\"id" + start.getId() + "\"");
		writer.writeEmptyTag("target ref=\"id" + end.getId() + "\"");

		printSelect(writer);
		printGuard(writer);
		printSync(writer);
		printUpdate(writer);
		printComments(writer);

		writer.endTag("transition");
	}

	/**
	 * Prints the select expressions in Uppaal modeling language XML format to
	 * the given writer.
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void printSelect(XMLWriter writer) throws IOException {
		if (select != null && select.size() > 0) {
			writer.startTag("label kind=\"select\"");
			writer.writeTextLine(select.get(0));
			for (int i = 1; i < select.size(); i++) {
				writer.writeTextLine(select.get(1));
			}
			writer.endTag("label");
		}
	}

	/**
	 * 
	 * Prints the guard expression in Uppaal modeling language XML format to the
	 * given writer.
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void printGuard(XMLWriter writer) throws IOException {
		if (guard != null) {
			writer.startTag("label kind=\"guard\"");
			writer.writeTextLine(guard, true);
			writer.endTag("label");
		}

	}

	/**
	 * 
	 * Prints the synchronisation expression in Uppaal modeling language XML
	 * format to the given writer.
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void printSync(XMLWriter writer) throws IOException {
		if (sync != null) {
			writer.startTag("label kind=\"synchronisation\"");
			writer.writeTextLine(sync);
			writer.endTag("label");
		}
	}

	/**
	 * 
	 * Prints the update expression list in Uppaal modeling language XML format
	 * to the given writer.
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void printUpdate(XMLWriter writer) throws IOException {
		if (update != null && update.size() > 0) {
			writer.startTag("label kind=\"assignment\"");
			writer.writeText(update.get(0));
			for (int i = 1; i < update.size(); i++) {
				writer.writeTextLine(",");
				writer.writeText(update.get(i));
			}
			writer.writeTextLine();
			writer.endTag("label");
		}
	}

	/**
	 * 
	 * Prints the node of the expression list as Uppaal comment to the given
	 * writer.
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void printComments(XMLWriter writer) throws IOException {
		if (select != null || guard != null || update != null) {
			writer.startTag("label kind=\"comments\"");
			if (select != null && select.size() > 0) {
				writer.writeTextLine("Select:");
				writer.writeExpressionInfo(select.get(0));
				for (int i = 1; i < select.size(); i++) {
					writer.writeTextLine(",");
					writer.writeExpressionInfo(select.get(i));
				}
				writer.writeTextLine();
			}
			if (guard != null) {
				writer.writeTextLine("Guard:");
				writer.writeExpressionInfo(guard);
				writer.writeTextLine();
			}
			if (update != null && update.size() > 0) {
				writer.writeTextLine("Update:");
				writer.writeExpressionInfo(update.get(0));
				for (int i = 1; i < update.size(); i++) {
					writer.writeTextLine(",");
					writer.writeExpressionInfo(update.get(1));
				}
			}
			writer.endTag("label");
		}
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		// Begin and end
		sb.append("From: " + start + ", To: " + end);
		// the guards
		sb.append(guard != null ? ", guard = " + guard : "");
		// the synchronization
		sb.append(sync != null ? ", sync = " + sync : "");
		// all updates
		if (update != null && !update.isEmpty()) {
			sb.append(", update = ");
			sb.append(update.get(0));
			for (int i = 1; i < update.size(); i++) {
				sb.append(", ");
				sb.append(update.get(i));
			}
		}
		return sb.toString();
	}
}
