/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.engine.Engine;
import de.tub.pes.sc2uppaal.util.XMLWriter;

/**
 * The Uppaal system declaration consists of a number of template instances,
 * which are defined like variables of certain types, i.e. TA templates. At the
 * definition of template instances, arguments are assigned to the template
 * parameters. Hence a template instance simply consists of a name, a
 * parameter-&gt;argument map and a link to the template it is an instance of.
 * 
 * @author Joachim Fellmuth, Paula Herber, Marcel Pockrandt, Björn Beckmann
 * 
 */
public class TATemplateInstance {

	private static Logger logger = LogManager
			.getLogger(TATemplateInstance.class.getName());

	/**
	 * The template this is an instance of.
	 */
	private final TATemplate template;

	/**
	 * Name of this instance.
	 */
	private String name = "";

	/**
	 * Parameter-&gt;argument map.
	 */
	private Map<String, String> params;

	/**
	 * Locale Variable bindings.
	 */
	private final Map<String, String> localVars;

	/**
	 * 
	 * @param template
	 * @param name
	 */
	public TATemplateInstance(TATemplate template, String name) {
		// if(name.equals("AHB_Bus#ahb_simple_bus#ahb_simple_bus#nb_transport_bw"))
		// System.out.println("break");
		if (template == null)
			logger.warn(
					"trying to create template instance {} with null template",
					name);

		this.template = template;
		this.name = name;

		params = new HashMap<String, String>();
		localVars = new HashMap<String, String>();

	}

	/**
	 * Returns the name of this instance.
	 * 
	 * @return
	 */
	public String getName() {
		return name;
	}

	/**
	 * Returns the template this is an instance of.
	 * 
	 * @return template
	 */
	public TATemplate getTemplate() {
		return template;
	}

	/**
	 * Returns the name of the template this is an instance of.
	 * 
	 * @return
	 */
	public String getTemplateName() {
		return template.getName();
	}

	/**
	 * Connects a parameter with an argument variable or value. It is not
	 * required that the connected parameter actually exists, only that all
	 * existing parameters are eventualy connected.
	 * 
	 * @param localName
	 *            Parameter
	 * @param argName
	 *            Argument
	 */
	public void connectParameter(String localName, String argName) {
		params.put(localName, argName);
	}

	/**
	 * Connects a local variable with an argument variable or value. It is not
	 * required that the connected parameter actually exists, only that all
	 * existing parameters are eventualy connected.
	 * 
	 * @param localName
	 *            Parameter
	 * @param argName
	 *            Argument
	 */
	public void connectLocalVariable(String localName, String argName) {
		localVars.put(localName, argName);
	}

	public Map<String, String> getLocalVariableBindings() {
		return localVars;
	}

	/**
	 * Sets a new argument map. Keys are parameter names.
	 * 
	 * @param params
	 */
	public void setParamConnections(Map<String, String> params) {
		this.params = params;
	}

	public Map<String, String> getParamConnections() {
		return params;
	}

	/**
	 * Returns the argument connected to a template parameter, i.e. the global
	 * variable that is assigned to the local parameter.
	 * 
	 * @param localParam
	 *            Parameter
	 * @return Argument
	 */
	public String getGlobalParam(String localParam) {
		return params.get(localParam);
	}

	/**
	 * Simple test if the minimum requirement for full connection is met: are
	 * there as many parameter connections as parameters? Does not guarantee
	 * that all parameters are connected, as there can be connections for
	 * nonexistent parameters.
	 * 
	 * @return True if enough parameter connection available, else false.
	 */
	// public boolean isConnected() {
	// return template.getParameters().size() == params.size();
	// }

	/**
	 * Prints the template instance to the given stream writer. The format is
	 * similar to the system declarations in the Uppaal modeling suite and the
	 * Uppaal modeling language.
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void print(XMLWriter writer) throws IOException {
		String out = name + " = " + template.getName() + "(";
		Iterator<TAVariable> paramIterator = template.getParameters()
				.iterator();

		List<TAVariable> localVarsAsParamsList = template
				.getLocalVarsAsParameters();

		while (paramIterator.hasNext()) {
			TAVariable param = paramIterator.next();
			String binding = params.get(param.getName());
			if (binding == null) {
				logger.error(
						"Found no binding for parameter {} of TATemplate {}.",
						param.getName(), name);
			}
			out += binding;
			if (paramIterator.hasNext()) {
				out += ", ";
			}
		}

		// If we use defragmentation, all address pointers of the template are
		// created as global variables and we have to bind them to their
		// corresponding parameters
		if (Engine.USE_DEFRAGMENTATION) {
			for (int i = 0; i < localVarsAsParamsList.size(); i++) {
				if (i == 0 && template.getParameters().size() != 0) {
					out += ", ";
				} else if (i != 0) {
					out += ", ";
				}
				out += Constants.createDelimitedString(this.name,
						localVarsAsParamsList.get(i).getName());
			}
		}

		out += ");";
		writer.writeTextLine(out);
	}

	/**
	 * Creates a copy of this instance, with the given string as name prefix.
	 * Copies parameter connection map, too.
	 * 
	 * @param prefix
	 *            Name prefix
	 * @return Copy
	 */
	public TATemplateInstance createCopy(String prefix) {
		TATemplateInstance copy = new TATemplateInstance(this.template, prefix
				+ this.name);

		Iterator<String> iter = params.keySet().iterator();
		while (iter.hasNext()) {
			String key = iter.next();
			copy.connectParameter(key, prefix + params.get(key));
		}

		iter = localVars.keySet().iterator();
		while (iter.hasNext()) {
			String key = iter.next();
			copy.connectParameter(key, prefix + localVars.get(key));
		}

		return copy;
	}

	@Override
	public String toString() {
		return name;
	}

}
