/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.io.IOException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.engine.Engine;
import de.tub.pes.sc2uppaal.optimization.POR;
import de.tub.pes.sc2uppaal.tamodel.expressions.TASendExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TASendRecvExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TAVariableExpression;
import de.tub.pes.sc2uppaal.util.XMLWriter;
import de.tub.pes.syscir.sc_model.SCFunction;
import de.tub.pes.syscir.sc_model.expressions.AccessExpression;
import de.tub.pes.syscir.sc_model.expressions.ArrayAccessExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.SCPortSCSocketExpression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableExpression;

//import de.tub.pes.syscir.sc_model.expressions.VariableExpression;

/**
 * A TA template is a template for the construction of a timed automaton in
 * Uppaal. It contains the locations and transitions that make up the control
 * flow of the system. This class represents a TA template in this internal
 * declaration and consists of all the parts a template in Uppaal has:
 * transitions, locations, local variables and parameter variables.
 * 
 * @author fellmuth, Verena Klös, Björn Beckmann
 */
public class TATemplate {

	private static Logger logger = LogManager.getLogger(TATemplate.class
			.getName());

	/**
	 * Name of the template
	 */
	private String name;

	/**
	 * SystemC function this template implements
	 */
	private SCFunction function = null;

	/**
	 * Set of the template parameters.
	 */
	private Set<TAVariable> parameters;

	/**
	 * List of local variables of this template.
	 */
	private final List<TAVariable> localVars;

	/**
	 * List of local variables of this template which should be used as
	 * parameters. This is used for defragmentation as we have to create global
	 * variables for each address pointer variable.
	 */
	private final List<TAVariable> localVarsAsParameters;

	/**
	 * List of function parameters which are call by value (note: thes are not
	 * the _template_ parameters)
	 */
	private final List<TAVariable> cbvParsAsLocalVars;

	private final List<TAFunction> localFuns;

	/**
	 * List of locations.
	 */
	private List<TALocation> locations;

	/**
	 * List of transitions
	 */
	private List<TATransition> transitions;

	/**
	 * Partial Order Reduction
	 * 
	 * contains necessary informations to support partial order reduction
	 * 
	 * e.g. results from data race analysis
	 * 
	 */
	private POR por;

	private boolean porActive = false;

	private int waitOffset;

	/**
	 * The initial location of the template. It must be set to a location that
	 * exists in this template.
	 */
	private TALocation initLocation = null;

	private boolean testTemplate = false;
	private boolean monitorTemplate = false;

	/**
	 * Creates a TA template with the given name. Initializes the containers.
	 * 
	 * @param name
	 */
	public TATemplate(String name) {
		this.name = name;
		this.parameters = new HashSet<TAVariable>();
		this.localVars = new LinkedList<TAVariable>();
		this.localVarsAsParameters = new LinkedList<TAVariable>();
		this.cbvParsAsLocalVars = new LinkedList<TAVariable>();
		this.locations = new LinkedList<TALocation>();
		this.transitions = new LinkedList<TATransition>();
		this.localFuns = new LinkedList<TAFunction>();
	}

	public TATemplate(String name, POR por) {
		this(name);
		this.setPor(por);
		TAInteger cln = new TAInteger("cln");
		cln.setConstant(true);
		this.addParameter(cln);
	}

	/**
	 * Returns the name of this template.
	 * 
	 * @return template name
	 */
	public String getName() {
		return name;
	}

	/**
	 * Changes the name of this template
	 * 
	 * @param name
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Changes the function of this template
	 * 
	 * @param function
	 */
	public void setFunction(SCFunction function) {
		this.function = function;
	}

	/**
	 * Returns the function of this template, returns null if this template does
	 * not implement a SystemC function.
	 * 
	 * @return function this template implements
	 */
	public SCFunction getFunction() {
		return function;
	}

	/**
	 * Returns true if this is a test template.
	 * 
	 * @return boolean
	 */
	public boolean isTestTemplate() {
		return this.testTemplate;
	}

	/**
	 * Marks this template as Test Template.
	 */
	public void setTestTemplate() {
		this.testTemplate = true;
	}

	/**
	 * Returns true if this is a monitor template.
	 * 
	 * @return boolean
	 */
	public boolean isMonitorTemplate() {
		return this.monitorTemplate;
	}

	/**
	 * Marks this template as montior template.
	 */
	public void setMonitorTemplate() {
		this.monitorTemplate = true;
	}

	/**
	 * Returns the list of parameters to this template.
	 * 
	 * @return List of parameters
	 */
	public Set<TAVariable> getParameters() {
		return parameters;
	}

	/**
	 * Sets a list of parameters to this template. Replaces the old. Creates a
	 * defensive copy.
	 * 
	 * @param parameters
	 */
	public void setParameters(List<TAVariable> parameters) {
		this.parameters = new HashSet<TAVariable>(parameters);
	}

	/**
	 * Returns the parameter variable or null if there is no parameter with this
	 * name
	 * 
	 * @param varName
	 * @return
	 */
	public TAVariable getParameter(String varName) {
		for (TAVariable var : parameters) {
			if (var.getName().equals(varName))
				return var;
		}
		return null;
	}

	/**
	 * Adds a parameter variable to the parameter list.
	 * 
	 * @param param
	 */
	public void addParameter(TAVariable param) {
		this.parameters.add(param);
	}

	/**
	 * Adds a parameter variable to the parameter list.
	 * 
	 * @param param
	 */
	public void addParameters(List<TAVariable> params) {
		this.parameters.addAll(params);
	}

	/**
	 * replaces the first occurrence of the given variable in the list of
	 * parameters with the given replacement
	 * 
	 * @param var
	 *            the variable to be replaced
	 * @param replace
	 *            the replacement
	 * @return true if the var has been replaced
	 */
	public boolean replaceParam(TAVariable var, TAVariable replace) {
		if (parameters.contains(var)) {
			boolean removed = parameters.remove(var);
			boolean added = parameters.add(replace);
			return removed && added;
		} else {
			return false;
		}
	}

	public TAVariable getLocalVar(String name) {
		for (TAVariable v : localVars) {
			if (name.equals(v.getName())) {
				return v;
			}
		}
		return null;
	}

	/**
	 * Returns the list of local variables.
	 * 
	 * @return
	 */
	public List<TAVariable> getLocalVars() {
		return localVars;
	}

	public List<TAVariable> getLocalVarsAsParameters() {
		return localVarsAsParameters;
	}

	/**
	 * Adds a variable to the list of local variables.
	 * 
	 * @param var
	 */
	public void addLocalVar(TAVariable var) {
		if (Engine.USE_DEFRAGMENTATION
				&& var.getType().endsWith(VariableConverter.getAddrPtrType(""))) {
			// if defragmentation is used, we have to handle all address
			// pointers
			// as parameters and create global variables for each of them.
			localVarsAsParameters.add(var);
		} else {
			if (this.localVars.contains(var))
				logger.warn(
						"local variable {} cannot be added more than once to template {}!",
						var.getName(), this.getName());
			else
				this.localVars.add(var);
		}
	}

	/**
	 * replaces the first occurrence of the given variable with the given
	 * replacement
	 * 
	 * @param var
	 *            the variable to be replaced
	 * @param replace
	 *            the replacement
	 */
	public void replaceLocalVar(TAVariable var, TAVariable replace) {
		if (localVars.contains(var))
			localVars.set(localVars.indexOf(var), replace);
	}

	private boolean containsLocalVar(String arg1) {
		for (TAVariable var : this.localVars)
			if (var.getName().equals(arg1))
				return true;
		return false;
	}
	
	private boolean containsLocalVarAsParameter(String arg1) {
		for (TAVariable var : this.localVarsAsParameters)
			if (var.getName().equals(arg1))
				return true;
		return false;
	}

	private boolean containsCbvParAsLocVar(String name) {
		for (TAVariable var : this.cbvParsAsLocalVars)
			if (var.getName().equals(name))
				return true;
		return false;
	}

	public boolean containsLocalVarOrCbvParAsLocVar(String name) {
		return (containsLocalVar(name) || containsCbvParAsLocVar(name) || containsLocalVarAsParameter(name));
	}

	public TAFunction getLocalFun(String name) {
		for (TAFunction f : localFuns) {
			if (name.equals(f.getName())) {
				return f;
			}
		}
		return null;
	}

	/**
	 * Returns the list of local functions.
	 * 
	 * @return
	 */
	public List<TAFunction> getLocalFuns() {
		return localFuns;
	}

	/**
	 * Adds a function to the list of local functions.
	 * 
	 * @param var
	 */
	public void addLocalFun(TAFunction fun) {
		this.localFuns.add(fun);
	}

	/**
	 * @return the callByValueParamsAsLocalVars
	 */
	public List<TAVariable> getcbvParsAsLocalVars() {
		return cbvParsAsLocalVars;
	}

	public void addCbvParsAsLocalVar(TAVariable var) {
		if (this.cbvParsAsLocalVars.contains(var)) {
			logger.warn(
					"callbyref param {} cannot be added more than once to template {}!",
					var.getName(), this.getName());
		} else {
			this.cbvParsAsLocalVars.add(var);
		}
	}

	/**
	 * Returns the list of locations of this template.
	 * 
	 * @return
	 */
	public List<TALocation> getLocations() {
		return locations;
	}

	/**
	 * Adds a new location to this template.
	 * 
	 * @param location
	 */
	public void addLocation(TALocation location) {
		if (!this.locations.contains(location)) {
			this.locations.add(location);
		}
	}

	/**
	 * Creates a new named location, adds it to the template and returns it.
	 * 
	 * @param name
	 *            Name of the new location
	 * @return The newly created TA location
	 */
	public TALocation createLocation(String name) {
		TALocation newLoc = new TALocation(name);
		this.locations.add(newLoc);
		return newLoc;
	}

	/**
	 * Creates a new anonymous location, adds it to this template and returns
	 * it.
	 * 
	 * @return The newly created TA location
	 */
	public TALocation createLocation() {
		TALocation newLoc = new TALocation();
		this.locations.add(newLoc);
		return newLoc;
	}

	/**
	 * Creates a new anonymous location, sets the urgent flag, adds it to this
	 * template and returns it.
	 * 
	 * @return The newly created TA location
	 */
	public TALocation createUrgentLocation() {
		TALocation newLoc = new TALocation();
		newLoc.setUrgent(true);
		this.locations.add(newLoc);
		return newLoc;
	}

	/**
	 * Creates a new anonymous location, sets the committed flag, adds it to
	 * this template and returns it.
	 * 
	 * @return The newly created TA location
	 */
	public TALocation createCommittedLocation() {
		TALocation newLoc = new TALocation();
		newLoc.setCommitted(true);
		this.locations.add(newLoc);
		return newLoc;
	}

	/**
	 * Creates a new anonymous location, sets the standard flag, adds it to this
	 * template and returns it.
	 * 
	 * @return The newly created TA location
	 */
	public TALocation createStandardLocation() {
		TALocation newLoc = new TALocation();
		newLoc.setUrgent(true);
		this.locations.add(newLoc);
		return newLoc;
	}

	/**
	 * Sets a new list of locations to this template. Replaces the old one.
	 * 
	 * @param locations
	 */
	public void setLocations(List<TALocation> locations) {
		this.locations = locations;
	}

	/**
	 * Returns the list of transitions of this templates.
	 * 
	 * @return List of transitions
	 */
	public List<TATransition> getTransitions() {
		return transitions;
	}

	/**
	 * Adds a new transition to the template.
	 * 
	 * @param t
	 *            The new transition
	 */
	public void addTransition(TATransition t) {
		this.transitions.add(t);
	}

	public void removeTransition(TATransition t) {
		this.transitions.remove(t);
	}

	/**
	 * Sets a new list of transitions to this template. Replaces the old one.
	 * 
	 * @param transitions
	 */
	public void setTransitions(List<TATransition> transitions) {
		this.transitions = transitions;
	}

	/**
	 * Creates a new instance of this template.
	 * 
	 * @param name
	 *            Instance name
	 * @return An instance of this template
	 */
	public TATemplateInstance getInstance(String name) {
		return new TATemplateInstance(this, name);
	}

	public TALocation createInitLocation() {
		initLocation = createLocation(Constants.INIT_LOCATION_NAME);
		addLocation(initLocation);
		return initLocation;
	}

	/**
	 * Returns the initial location of this template or null if not set.
	 * 
	 * @return
	 */
	public TALocation getInitLocation() {
		return initLocation;
	}

	/**
	 * Sets the initial location of this template. If the given location is not
	 * yet part of this template, it is added to the list of locations.
	 * 
	 * @param initLocation
	 */
	public void setInitLocation(TALocation initLocation) {
		this.initLocation = initLocation;

		if (!this.locations.contains(initLocation)) {
			this.locations.add(initLocation);
		}
	}

	/**
	 * Computes a list of transitions whose start location is the given
	 * location.
	 * 
	 * @param loc
	 *            Starting location
	 * @return List of outgoing transitions
	 */
	public List<TATransition> getOutgoingTransitions(TALocation loc) {
		LinkedList<TATransition> list = new LinkedList<TATransition>();
		for (TATransition t : this.transitions) {
			if (t.getStart() == loc) {
				list.add(t);
			}
		}
		return list;
	}

	/**
	 * Computes a list of transitions whose end location is the given location.
	 * 
	 * @param loc
	 *            End location
	 * @return List of incoming transitions
	 */
	public List<TATransition> getIncomingTransitions(TALocation loc) {
		LinkedList<TATransition> list = new LinkedList<TATransition>();
		for (TATransition t : this.transitions) {
			if (t.getEnd() == loc) {
				list.add(t);
			}
		}
		return list;
	}

	// TODO
	// /**
	// * Creates a copy of this template with the given name by adding all its
	// members to a new template.
	// * Does not create copies of the locations and transitions. Only variables
	// and parameters are copied.
	// *
	// * @param name
	// * @return Copy of this template
	// */
	// public TATemplate createCopy(String name) {
	//
	// TATemplate newT = new TATemplate(name);
	// for(TAVariable var: parameters) {
	// newT.addParameter(var.createCopy(""));
	// }
	// for(TAVariable var: localVarsAsParameter) {
	// newT.addLocalVarAsParameter(var.createCopy(""));
	// }
	// for(TAVariable var: localVars) {
	// newT.addLocalVar(var.createCopy(""));
	// }
	// for(TAFunction fun: localFuns) {
	// newT.addLocalFun(fun.createCopy());
	// }
	// for(TALocation loc: locations) {
	// TALocation newloc = loc.createCopy();
	// newT.addLocation(newloc);
	// if(loc.getId() == initLocation.getId())
	// newT.setInitLocation(newloc);
	// }
	// for(TATransition trans: transitions) {
	// newT.addTransition(trans);
	// }
	//
	// return newT;
	// }

	/**
	 * Prints the template into the given stream. Format is XML conforming to
	 * the Uppaal modeling language.
	 * 
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void print(XMLWriter writer) throws IOException {
		// start of tag
		writer.startTag("template");

		// name of template
		writer.startTag("name").writeText(name).endTag("name");

		// parameters to template
		writer.startTag("parameter");
		Iterator<TAVariable> paramIterator = parameters.iterator();
		while (paramIterator.hasNext()) {
			paramIterator.next().printAsParam(writer);
			if (paramIterator.hasNext())
				writer.writeText(", ");
		}

		// if we use defragmentation, all address pointers are handled as
		// parameters
		if (Engine.USE_DEFRAGMENTATION) {
			for (int i = 0; i < localVarsAsParameters.size(); i++) {
				if (i == 0 && parameters.size() != 0) {
					writer.writeText(", ");
				}
				if (i != 0) {
					writer.writeText(", ");
				}
				localVarsAsParameters.get(i).printAsParam(writer);
			}
		}
		writer.endTag("parameter");

		writer.startTag("declaration");
		for (TAVariable var : cbvParsAsLocalVars) {
			writer.writeText(var.createDeclarationString());
		}
		for (TAVariable var : localVars) {
			writer.writeText(var.createDeclarationString());
		}
		for (TAFunction fun : localFuns) {
			writer.writeText(fun.createDeclarationString());
		}
		writer.endTag("declaration");

		// location tags
		for (int i = 0; i < locations.size(); i++) {
			locations.get(i).print(writer);
		}

		// initial location [required]
		if (initLocation != null) {
			writer.writeEmptyTag("init ref=\"id" + initLocation.getId() + "\"");
		}

		// transition tags
		for (int i = 0; i < transitions.size(); i++) {
			transitions.get(i).print(writer);
		}

		// end of tag
		writer.endTag("template");
	}

	@Override
	public String toString() {
		return name;
	}

	public void removeLocalVars(LinkedList<TAVariable> toRemove) {
		this.localVars.removeAll(toRemove);
	}

	public void removeLocalVar(TAVariable r) {
		this.localVars.remove(r);
	}

	public void addLocations(TALocation... locations) {
		for (TALocation loc : locations) {
			this.addLocation(loc);
		}
	}

	public void addLocalVars(TAVariable... variables) {
		for (TAVariable var : variables) {
			this.addLocalVar(var);
		}
	}

	public void addTransitions(TATransition... transitions) {
		for (TATransition t : transitions) {
			this.addTransition(t);
		}
	}

	public boolean isVarUsedInTemplate(TAVariable var) {
		for (TALocation loc : this.getLocations()) {
			Expression expr = loc.getInvariant();

			if (exprContainsVar(expr, var)) {
				return true;
			}
		}

		for (TATransition trans : this.getTransitions()) {
			for (Expression exp : trans.getSelect()) {
				if (exprContainsVar(exp, var)) {
					return true;
				}
			}
			for (Expression exp : trans.getUpdate()) {
				if (exprContainsVar(exp, var)) {
					return true;
				}
			}
			if (trans.getGuard() != null
					&& exprContainsVar(trans.getGuard(), var)) {
				return true;
			}

			if (trans.getSync() != null
					&& trans.getSync().isChanName(var.getName())) {
				return true;
			}
		}
		// check if var is used in function
		for (TAFunction fun : this.localFuns) {
			if (fun.toString().contains(var.getName()))
				return true;
		}
		return false;
	}

	private boolean exprContainsVar(Expression expr, TAVariable var) {
		if (expr != null) {
			String varName = var.getName();
			// FIXME: this is obviously dangerous
			if (expr.toStringNoSem().replaceAll("\\[.*\\]", "").equals(varName))
				return true;
			for (Expression iexpr : expr.crawlDeeper()) {
				if (iexpr instanceof SCVariableExpression) {
					SCVariableExpression svexpr = (SCVariableExpression) iexpr;
					if (svexpr.getVar().getName().equals(varName)) {
						return true;
					}
					if (iexpr instanceof ArrayAccessExpression) {
						if (exprContainsVar(iexpr, var)) {
							return true;
						}
					}

				} else if (iexpr instanceof TAVariableExpression) {
					TAVariableExpression tvexpr = (TAVariableExpression) iexpr;
					if (tvexpr.getVar().getName().equals(varName)) {
						return true;
					}
				} else if (iexpr instanceof ConstantExpression) {
					ConstantExpression cexpr = (ConstantExpression) iexpr;
					if (cexpr.getValue().equals(varName)) {
						return true;
					}
				}	else if (iexpr instanceof AccessExpression) {
					// check left if we talk about tlm-sockets as methods
					if (exprContainsVar(((AccessExpression) iexpr).getLeft(),
							var)) {
						return true;
					}
					// TODO: we inserted this case for a local sc_signal (TAAcessExpression) - check whether this is correct in all cases
					else if(iexpr.toStringNoSem().equals(varName))
					{
						return true;
					}
					// check the whole thing in case of ports
					if (((AccessExpression) iexpr).getLeft() instanceof SCPortSCSocketExpression) {
						if (iexpr.toStringNoSem().equals(varName)) {
							return true;
						}
					}
				} else if (exprContainsVar(iexpr, var)) {
					return true;
				}
			}
		}
		return false;
	}

	// TODO: dont recompute this all the time
	public List<TAVariable> getCalledCtrlChannelsFromParameters() {
		LinkedList<TAVariable> calledChannels = new LinkedList<TAVariable>();
		if (this.getInitLocation() != null) {
			List<TATransition> initTransitions = this
					.getOutgoingTransitions(this.getInitLocation());
			if (initTransitions == null || initTransitions.isEmpty()) {
				logger.debug("Template {} has no init transitions.",
						this.getName());
			} else {
				// This is copied from earlier version, although I am pretty
				// sure it should only happen if there is exactly one initial
				// transition...
				TATransition initTransition = initTransitions.get(0);
				for (TAVariable var : this.getParameters()) {
					if ((var instanceof TAChannel || var instanceof TAArray)
							&& !initTransition.getSync().contains(var)
							&& var.getName()
									.matches(
											".*"
													+ Constants.LOCAL_FUNCTION_CTRL_KEYWORD)) {
						if (isVarUsedInTemplate(var)) {
							calledChannels.add(var);
						}
					}
				}
			}
		}
		return calledChannels;
	}

	public List<String> getCalledChannelNames() {
		LinkedList<String> calledChannels = new LinkedList<String>();
		for (TATransition trans : this.transitions) {
			TASendRecvExpression sendRecvExpr = trans.getSync();
			if (sendRecvExpr != null
					&& sendRecvExpr instanceof TASendExpression) {
				TASendExpression sendExpr = (TASendExpression) sendRecvExpr;
				calledChannels.add(sendExpr.getChanName());
			}
		}
		return calledChannels;
	}

	public TAVariable getCtrlParameter() {
		List<TATransition> out = this.getOutgoingTransitions(this
				.getInitLocation());
		if (out.size() == 1) {
			TATransition t = out.get(0);
			if (t.getSync() != null) {
				String s = t.getSync().toString();
				int index = s.indexOf(Constants.LOCAL_FUNCTION_CTRL_KEYWORD
						+ "?");
				if (index >= 0
						&& index >= s.length()
								- (Constants.LOCAL_FUNCTION_CTRL_KEYWORD + "? ")
										.length()) {
					return this.getParameter(s.replaceAll("\\?", ""));
				}
			}
		}
		return null;
	}

	public boolean removeParameter(TAVariable param) {
		return this.parameters.remove(param);
	}

	public boolean removeParameters(List<TAVariable> params) {
		return this.parameters.removeAll(params);
	}

	public POR getPor() {
		return por;
	}

	public void setPor(POR por) {
		this.por = por;
		if (this.por != null) {
			this.porActive = true;
			this.waitOffset = 0;
		}
	}

	public boolean isPorActive() {
		return porActive;
	}

	public int getWaitOffset() {
		return waitOffset;
	}

	public void setWaitOffset(int waitOffset) {
		this.waitOffset = waitOffset;
	}
}