/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import de.tub.pes.sc2uppaal.engine.Engine;
import de.tub.pes.sc2uppaal.optimization.POR;
import de.tub.pes.sc2uppaal.optimization.UpdateRequestsDeterminism;
import de.tub.pes.sc2uppaal.util.XMLWriter;

/**
 * This is the base class of the TA model, the internal representation of the
 * Uppaal model that is the output of this transformation. Like a usual Uppaal
 * model it consists of three parts, a number of TA templates, the system
 * declaration (a number of TA template instances) and the declaration (a list
 * of global variable declarations). Other than these containers this class only
 * contains methods to print its contents out.
 * <p>
 * A TA model is intended to store all final information created during a
 * transformation process. For that reason, there exists only one instance of
 * the TA model in the system.
 * 
 * @author Joachim Fellmuth, Paula Herber, Marcel Pockrandt, Björn Beckmann
 * 
 */
public class TAModel {

	/**
	 * TA templates defined for the current model. Stored in a map with the
	 * template names as keys.
	 */
	private Map<String, TATemplate> templates;

	/**
	 * A MemoryType is a SySC/C++ type represented in the memory model.
	 */
	private final LinkedList<TAMemoryType> memoryTypes;

	/**
	 * TA template instances created from the templates in this system. Stored
	 * in a map with the instance names as keys.
	 */
	private Map<String, TATemplateInstance> systemDeclaration;

	/**
	 * List of typedefs.
	 */
	private final LinkedList<TATypedef> typedefs;

	/**
	 * List of global variables.
	 */
	private LinkedList<TAVariable> declaration;

	/**
	 * List of global functions
	 */
	private final LinkedList<TAFunction> functions;
	
	/**
	 *  Partial Order Reduction Object
	 *  
	 *  contains necessary informations to support partial order reduction
	 *  
	 *  e.g. results from data race analysis
	 *  
	 */
	private POR por;
	
	/**
	 * 
	 */
	private boolean porActive=false;
	
	/**
	 *  Flag Update Request Determinism
	 *  
	 */
	private boolean updateRequestDeterminism=false;

	/**
	 * 
	 */
	private UpdateRequestsDeterminism urD;

	/**
	 * Standard constructor, initializes the containers.
	 */
	public TAModel() {
		templates = new HashMap<String, TATemplate>();
		systemDeclaration = new HashMap<String, TATemplateInstance>();
		typedefs = new LinkedList<TATypedef>();
		declaration = new LinkedList<TAVariable>();
		functions = new LinkedList<TAFunction>();
		memoryTypes = new LinkedList<TAMemoryType>();
	}

	/**
	 * Returns the map of templates defined in this model. Keys are the template
	 * names.
	 * 
	 * @return
	 */
	public Map<String, TATemplate> getTemplates() {
		return templates;
	}

	/**
	 * Returns the template with the given name
	 * 
	 * @param name
	 * @return template
	 */
	public TATemplate getTemplate(String templName) {
		TATemplate template = templates.get(templName);
		return template;
	}

	/**
	 * Sets a new map of templates.
	 * 
	 * @param templates
	 */
	public void setTemplates(Map<String, TATemplate> templates) {
		this.templates = templates;
	}

	/**
	 * Returns the system declaration, which is a map of template instances.
	 * Keys are their names.
	 * 
	 * @return System declaration
	 */
	public Map<String, TATemplateInstance> getSystemDeclaration() {
		return systemDeclaration;
	}

	/**
	 * Sets a new system declaration
	 * 
	 * @param systemDeclaration
	 */
	public void setSystemDeclaration(
			Map<String, TATemplateInstance> systemDeclaration) {
		this.systemDeclaration = systemDeclaration;
	}

	/**
	 * Finds a global variable with the given name and returns it. Returns null
	 * if not found.
	 * 
	 * @param name
	 * @return Variable with the given name or null if not found
	 */
	public TAVariable getGlobalVariable(String name) {
		if (name.contains("[")) {
			int i = name.indexOf("[");
			name = name.substring(0, i);
		}
		for (TAVariable var : declaration) {
			if (var.getName().equals(name))
				return var;
		}
		return null;
	}

	/**
	 * Returns the list of global variables
	 * 
	 * @return
	 */
	public List<TAVariable> getDeclaration() {
		return declaration;
	}

	/**
	 * Sets a new list of global variables to the model
	 * 
	 * @param declaration
	 */
	public void setDeclaration(LinkedList<TAVariable> declaration) {
		this.declaration = declaration;
	}

	/**
	 * Adds a typedef to the model.
	 * 
	 * @param typedef
	 */
	public void addTypedef(TATypedef typedef) {
		this.typedefs.add(typedef);
	}

	/**
	 * Adds a collection of typedefs to the model.
	 * 
	 * @param typedefs
	 */
	public void addTypedefs(Collection<TATypedef> typedef) {
		this.typedefs.addAll(typedef);
	}

	/**
	 * Adds a template to the model.
	 * 
	 * @param template
	 */
	public void addTemplate(TATemplate template) {
		templates.put(template.getName(), template);
	}
	
	/**
	 * Returns a list of all types defined in this ta model.
	 * @return
	 */
	public List<TATypedef> getTypedefs() {
		return this.typedefs;
	}

	/**
	 * Adds a new global variable to the model.
	 * 
	 * @param variable
	 */
	public void addVariable(TAVariable variable) {
		for (TAVariable var : declaration) {
			if (var.getName().equals(variable.getName())) {
				// System.out.println("Variable " + variable.getName() +
				// " already declared.");
				return;
			}
		}
		declaration.addLast(variable);
	}

	/**
	 * replaces a global variable in the model. This is used for the replacement
	 * of referenced parameters (especially arrays) occurring in templates.
	 * 
	 * @param variable
	 *            the variable that should be replaced (with the correct prefix
	 *            including the module instance)
	 * @param replace
	 *            the replacement (with the correct prefix including the module
	 *            instance)
	 * @return true if the variable was found and replaced
	 */
	public boolean replaceVar(TAVariable variable, TAVariable replace) {
		TAVariable var = this.getGlobalVariable(variable.getName());
		if (var != null) {
			declaration.set(declaration.indexOf(var), replace);
			return true;
		}
		return false;
	}

	/**
	 * finds a global variable in the model.
	 * 
	 * @param variable
	 *            the variable that should be found (without prefix)
	 * @param initiator
	 * @param target
	 * @param prefix
	 *            the prefix "#functionName#param#"
	 * @return the variable (with prefix) or null
	 */
	public TAVariable getVarTLM(TAVariable variable, String initiator,
			String target, String prefix) {
		TAVariable v = null;
		for (TAVariable var : declaration) {
			if (initiator.equals("")) {
				if (var.getName()
						.contains(target + prefix + variable.getName())
						&& var.getName()
								.substring(
										var.getName().lastIndexOf(
												Constants.PREFIX_DELIMITER) + 1)
								.equals(variable.getName())) {
					v = var;
					break;
				}
			} else {
				String[] arr = var.getName().split(
						target + prefix + variable.getName());
				if (arr.length == 1
						&& var.getName().contains(
								target + prefix + variable.getName())
						&& var.getName()
								.substring(
										var.getName().lastIndexOf(
												Constants.PREFIX_DELIMITER) + 1)
								.equals(variable.getName())) {
					if (arr[0].contains(initiator)) {
						v = var;
						break;
					}
				}
			}
		}
		return v;
	}

	/**
	 * Adds a new template instance to the model.
	 * 
	 * @param instance
	 */
	public void addTemplateInstance(TATemplateInstance instance) {
		systemDeclaration.put(instance.getName(), instance);
	}

	/**
	 * Prints the model into a stream writer, which can be directed to a file or
	 * to any stream. The format is XML, strictly conforming with the Uppaal
	 * modeling language.
	 * 
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void print(XMLWriter writer) throws IOException {
		writer.writeStdUppaalHeader();
		writer.startTag("nta");

		writer.startTag("declaration");
		// part 0: typedefs
		for (TATypedef typedef : typedefs) {
			typedef.print(writer);
		}

		// part 1: global variables
		writer.writeTextLine("clock global;");

		writer.writeTextLine(TAMemoryType.getGlobalMemVariables());
		// sort memory types to get dependencies correctly
		Comparator<TAMemoryType> depsCompare = new Comparator<TAMemoryType>() {
			private String getInnerCurlyBraces(String s) {
				int start = s.indexOf("{");
				int end = s.indexOf("}");
				if (start != -1) {
					return s.substring(start, end);
				}
				return s;
			}

			@Override
			public int compare(TAMemoryType o1, TAMemoryType o2) {
				// check for dependency
				// ex: o1 uses int_addr_ptr and o2 is "int" --> o1 > o2 -->
				// define o1 later
				if (!o1.getTypedef().equals("") && o2.getTypedef().equals("")) {
					return 1;
				} else if (o1.getTypedef().equals("")
						&& !o2.getTypedef().equals("")) {
					return -1;
				} else if (getInnerCurlyBraces(o1.getTypedef()).contains(
						o2.getType())
						|| getInnerCurlyBraces(o1.getTypedef()).contains(
								VariableConverter.getAddrPtrType(o2.getType()))) {
					return 1;
				} else if (getInnerCurlyBraces(o2.getTypedef()).contains(
						o1.getType())
						|| getInnerCurlyBraces(o2.getTypedef()).contains(
								VariableConverter.getAddrPtrType(o1.getType()))) {
					return -1;
				}
				return o1.getType().compareTo(o2.getType());
			}

		};
		Collections.sort(memoryTypes, depsCompare);
		for (TAMemoryType mt : memoryTypes) {
			writer.writeTextLine(mt.getVariableDefinitions());
		}

		writer.writeHeader("Global Variable Definition");
		
		// declaration of all non-mem variables
		for (TAVariable var : declaration) {
			var.print(writer);
		}
		
		// if we use defragmentation, we have to create all global variables for all local address pointers
		if (Engine.USE_DEFRAGMENTATION) {
			writer.writeHeader("Address Pointer Definition for Defragmentation");
			for (String tiName : systemDeclaration.keySet()) {
				TATemplateInstance ti = systemDeclaration.get(tiName);
				TATemplate temp = ti.getTemplate();
				for (TAVariable var : temp.getLocalVarsAsParameters()) {
					TAVariable paramVar = var.createCopy(ti.getName() + Constants.PREFIX_DELIMITER);
					paramVar.print(writer);
				}
			}
		}

		for (TAMemoryType mt : memoryTypes) {
			writer.writeTextLine(mt.getHelperFunctionDefinitions());
			if (mt.getMemSize() > 0) {
				writer.writeTextLine(mt.getFunctionDefinitions());
			}
		}
	
		// declaration of all functions
		for (TAFunction taf : functions) {
			taf.print(writer);
		}
		
		//Request Update Determinism
		if(isUpdateRequestDeterminism()){
			writer.writeTextLine(urD.getGlobalVariables());
			writer.writeTextLine(urD.getGlobalFunctionDefinitions());
		}
		
		//POR
		if(this.isPorActive()){
			writer.writeTextLine(por.getGlobalPORVariables());
			
			writer.writeTextLine(por.getGlobalPORFunctionDefinitions());
		}
		writer.endTag("declaration");

		// part 2: templates
		TreeMap<String, TATemplate> sortedTemplates = new TreeMap<String, TATemplate>(
				templates);
		for (String key : sortedTemplates.keySet()) {
			// templates.get(key).print(writer);
			sortedTemplates.get(key).print(writer);
		}

		// part3: system declaration
		writer.startTag("system");
		Set<String> keys = systemDeclaration.keySet();
		String[] keyArray = keys.toArray(new String[keys.size()]);
		Arrays.sort(keyArray);
		String finalDec = null;
			
		for (int i = 0; i < keyArray.length; i++) {
			String key = keyArray[i];
			finalDec = (finalDec == null) ? key : finalDec + ","
					+ Constants.END_LINE + key;
			systemDeclaration.get(key).print(writer);
		}
		writer.writeText(Constants.END_LINE + Constants.END_LINE + "system "
				+ finalDec + ";");

		writer.endTag("system");

		writer.endTag("nta");
	}

	public void labelLocations() {
		for (String templName : templates.keySet()) {
			TATemplate template = templates.get(templName);
			if (template.isTestTemplate()) {
				int i = 0;
				for (TALocation l : template.getLocations()) {
					l.setName("tc" + i);
					i++;
				}
				TALocation init = template.getInitLocation();
				init.setName("tc_start");
				for (TATransition t : template.getIncomingTransitions(init)) {
					t.getStart().setName("tc_end");
				}

			}
			if (template.isMonitorTemplate()) {
				int i = 0;
				for (TALocation l : template.getLocations()) {
					l.setName("mon" + i);
					i++;
				}
				TALocation init = template.getInitLocation();
				init.setName("mon_start");
				for (TATransition t : template.getIncomingTransitions(init)) {
					t.getStart().setName("mon_end");
				}

			}

		}

	}

	public void addVariables(LinkedList<TAVariable> vars) {
		for (TAVariable var : vars) {
			this.addVariable(var);
		}

	}

	public HashMap<TATemplate, List<TAVariable>> createMethodCallerMap() {
		HashMap<TATemplate, List<TAVariable>> calledMethods = new HashMap<TATemplate, List<TAVariable>>();
		for (String templateName : this.getTemplates().keySet()) {
			TATemplate template = this.getTemplates().get(templateName);
			List<TAVariable> calledCtrlChannels = template
					.getCalledCtrlChannelsFromParameters();
			List<TAVariable> globalCrtlChannels = getCalledGlobalChannelsByName(template
					.getCalledChannelNames());
			calledCtrlChannels.addAll(globalCrtlChannels);
			calledMethods.put(template, calledCtrlChannels);
		}
		return calledMethods;
	}

	private List<TAVariable> getCalledGlobalChannelsByName(
			List<String> calledChannelNames) {
		List<TAVariable> globalChannels = new ArrayList<TAVariable>();
		for (String chanName : calledChannelNames) {
			TAVariable globalChannel = this.getGlobalVariable(chanName);
			if (globalChannel != null) {
				globalChannels.add(globalChannel);
			}
		}
		return globalChannels;
	}

	public static String getTransportVariableName(String templInstName,
			String varName) {
		return templInstName + Constants.PREFIX_DELIMITER
				+ Constants.LOCAL_VARIABLE_KEYWORD + Constants.PREFIX_DELIMITER
				+ varName;
	}

	public void addFunction(TAFunction taf) {
		functions.add(taf);
	}

	public void addFunctions(List<TAFunction> tafs) {
		for (TAFunction taf : tafs) {
			this.addFunction(taf);
		}
	}

	public List<TAFunction> getFunctions() {
		return functions;
	}

	public TAFunction getFunctionByName(String name) {
		for (TAFunction func : functions) {
			if (func.name.equals(name)) {
				return func;
			}
		}
		return null;
	}

	public void addMemoryType(TAMemoryType mt) {
		memoryTypes.add(mt);
	}
	
	public List<TAMemoryType> getMemoryTypes() {
		return memoryTypes;
	}

	public void setPor(POR por){
		this.por = por;
		if(por!=null){
			this.porActive=true;
		}
	}
	
	public POR getPor(){
		return this.por;
	}

	public boolean isPorActive() {
		return porActive;
	}

	public UpdateRequestsDeterminism getUrD() {
		return urD;
	}

	public void setUrD(UpdateRequestsDeterminism urD) {
		this.urD = urD;
	}

	public boolean isUpdateRequestDeterminism() {
		return updateRequestDeterminism;
	}

	public void setUpdateRequestDeterminism(boolean updateRequestDeterminism) {
		this.updateRequestDeterminism = updateRequestDeterminism;
		if(updateRequestDeterminism){
			setUrD(new UpdateRequestsDeterminism());
		}
	}

}
