/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.engine.Engine;
import de.tub.pes.syscir.sc_model.SCClass;
import de.tub.pes.syscir.sc_model.SCEnumType;
import de.tub.pes.syscir.sc_model.SCSystem;
import de.tub.pes.syscir.sc_model.SCVariable;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.FunctionCallExpression;
import de.tub.pes.syscir.sc_model.variables.SCArray;
import de.tub.pes.syscir.sc_model.variables.SCPointer;

/**
 * This class encapsulates the information needed to generate the memory related
 * data for _one_ specific type.
 * 
 * @author rschroeder, Timm Liebrenz
 * 
 */
public class TAMemoryType {
	private static Logger logger = LogManager.getLogger(TAMemoryType.class
			.getName());
	private static final String BR = "\n";

	private final String type;
	private final int memSize;
	private final String defaultValue;
	private String typedef;
	private TAModel ta;
	private SCSystem sc;

	public static enum MEM_STATES {
		MEM_FREE, MEM_STATIC_OBJ, MEM_STATIC_ARR_HEAD, MEM_STATIC_ARR_TAIL, MEM_DYN_OBJ, MEM_DYN_ARR_HEAD, MEM_DYN_ARR_TAIL
	};

	private static final List<String> forbiddenTypes = new ArrayList<String>(7);
	static {
		forbiddenTypes.add("void");
		forbiddenTypes.add("sc_module_name");
		forbiddenTypes.add("sc_signal");
		forbiddenTypes.add("sc_fifo");
		forbiddenTypes.add("peq_with_cb_and_phase");
		forbiddenTypes.add(".*\\*+"); // use regex to catch ptrs
		forbiddenTypes.add("char");
	}

	public TAMemoryType(String type, int memSize, String defaultValue,
			SCSystem sc, TAModel ta) {
		super();
		this.type = type;
		this.memSize = memSize;
		this.defaultValue = defaultValue;
		this.typedef = "";
		this.sc = sc;
		this.ta = ta;
	}

	public TAMemoryType(String type, String typedef, int memSize,
			String defaultValue, SCSystem sc, TAModel ta) {
		this(type, memSize, defaultValue, sc, ta);
		this.typedef = typedef;

	}

	public String getType() {
		return type;
	}

	public int getMemSize() {
		return memSize;
	}

	public String getDefaultValue() {
		return defaultValue;
	}

	public String getTypedef() {
		return typedef;
	}

	/**
	 * Some types are not represented in the mem model. This includes for
	 * example sc_modules or variables of type char.
	 * 
	 * @param scvar
	 * @return
	 */
	public static boolean isForbiddenType(String type) {
		for (String ft : forbiddenTypes) {
			if (type.matches(ft + ".*"))
				return true;
		}
		return false;
	}

	/**
	 * Return the default value for this type. Either taken from a predefined
	 * HashMap (see Constants) or recursively build up (for structs).
	 * 
	 * @param scvar
	 * @param scs
	 * @return
	 */
	public static String getDefaultValueString(SCVariable scvar, SCSystem scs) {
		SCClass scc = scs.getClassByName(scvar.getTypeWithoutSize());
		if (scc != null) {
			StringBuffer defaultValue = new StringBuffer();
			defaultValue.append("{");
			for (Iterator<SCVariable> iter = scc.getMembers().iterator(); iter
					.hasNext();) {
				SCVariable v = iter.next();
				if (v instanceof SCPointer) {
					defaultValue.append(Constants.NULL);
				} else {
					defaultValue.append(getDefaultValueString(v, scs));
				}
				if (iter.hasNext()) {
					defaultValue.append(", ");
				}
			}
			defaultValue.append("}");
			return defaultValue.toString();
		} else {
			String type = scvar.getTypeWithoutSize();
			for (SCEnumType et : scs.getEnumTypes()) {
				if (et.getName().equals(type)) {
					return Constants.defaultValues.get("int");
				}
			}
			return Constants.defaultValues.get(type);
		}
	}

	/**
	 * Returns something like "typedef struct {int data;} myStruct".
	 * 
	 * @param scvar
	 * @param scs
	 * @return
	 */
	public static String getTypedef(SCVariable scvar, SCSystem scs) {
		SCClass scc = scs.getClassByName(scvar.getTypeWithoutSize());
		if (scc != null) {
			StringBuffer typedef = new StringBuffer();
			typedef.append("typedef struct {");
			for (SCVariable v : scc.getMembers()) {
				String type = v.getTypeWithoutSize();
				if (v instanceof SCPointer) {
					type = VariableConverter.getAddrPtrType(type);
				}
				typedef.append(type);
				typedef.append(" ");
				typedef.append(v.getName());
				typedef.append("; ");
			}
			typedef.setLength(typedef.length() - 1);
			typedef.append("} ");
			typedef.append(scc.getName());
			return typedef.toString();
		} else {
			return "";
		}
	}

	public static String getFuncName(String funcName, String type) {
		return funcName + "_" + type;
	}

	// ////////////// ALLOC STATIC ///////////////////////

	public static String getAllocStaticFuncName(String value) {
		return "allocate_static"
				+ (value != null && !value.equals("") ? "_and_init" : "");
	}

	// just for convenience: if there is an init value expr, call the
	// appropriate func
	public static String getAllocStaticFuncName(Expression value) {
		String v = "";
		if (value != null) {
			v = value.toString();
		}
		return getAllocStaticFuncName(v);
	}

	public static String getAllocStaticFuncName() {
		return getAllocStaticFuncName("");
	}

	public static String getAllocStaticArrFuncName() {
		return "allocate_static_arr";
	}

	// ////////////// ALLOC DYNAMIC ///////////////////////

	public static String getAllocDynFuncName(String value) {
		return "allocate_dynamic"
				+ (value != null && !value.equals("") ? "_and_init" : "");
	}

	public static String getAllocDynFuncName() {
		return getAllocDynFuncName("");
	}

	public static String getAllocDynArrFuncName() {
		return "allocate_dynamic_arr";
	}

	// ////////////// DEALLOC STATIC ///////////////////////

	public static String getDeleteStaticFuncName(boolean isArray) {
		return "delete" + (isArray ? "_arr" : "") + "_static";
	}

	public static String getDeleteStaticFuncName() {
		return getDeleteStaticFuncName(false);
	}

	public static String getDeleteStaticArrFuncName() {
		return getDeleteStaticFuncName(true);
	}

	// ////////////// DEALLOC DYNAMIC ///////////////////////

	public static String getDeleteDynFuncName(boolean isArray) {
		return "delete" + (isArray ? "_arr" : "") + "_dynamic";
	}

	public static String getDeleteDynFuncName() {
		return getDeleteDynFuncName(false);
	}

	public static String getDeleteDynArrFuncName() {
		return getDeleteDynFuncName(true);
	}

	// ////////////// MISC ///////////////////////

	public static String getRemoveFromStackFuncName(boolean isArray) {
		return "empty_stack" + (isArray ? "_arr" : "");
	}

	public static String getRemoveFromStackFuncName() {
		return getRemoveFromStackFuncName(false);
	}

	public static String getRemoveArrFromStackFuncName() {
		return getRemoveFromStackFuncName(true);
	}

	public static String getAddrChkFuncName() {
		return "addr_is_valid";
	}

	public static String getInitArrayElemFuncName() {
		return "init_arr_elem";
	}

	public static String getIsArrElemFuncName() {
		return "is_arr_elem";
	}

	public static String getIsArrHeadElemFuncName() {
		return "is_arr_head_elem";
	}

	public static String getIsArrTailElemFuncName() {
		return "is_arr_tail_elem";
	}

	public static String getAddrPtrIsInRangeFuncName() {
		return "addr_is_in_range";
	}

	/**
	 * Delete a local variable from the stack. This is either a parameter which
	 * got created locally (callbyvalue) or a local variable. It might be, that
	 * the addrPtr is invalid. Ex: if (cond) { return 3; } .... int a = 24;
	 * return 4;
	 * 
	 * In this case, there is an addrPtr for a, but if cond is true, we can't
	 * call delete_int(addrPtr). It is still invalid. So we use a special delete
	 * which doesn't cause a MEM_ERR_* to be set.
	 * 
	 * @param addrPtr
	 * @param originalSCVar
	 * @return
	 */
	public static Expression getDeleteExpressionForLocVar(TAVariable addrPtr,
			SCVariable originalSCVar) {
		String name;
		if (originalSCVar instanceof SCArray) {
			name = getRemoveArrFromStackFuncName();
		} else {
			name = getRemoveFromStackFuncName();
		}
		String funcName = getFuncName(name, originalSCVar.getTypeWithoutSize());
		FunctionCallExpression fce = ExpressionConverter.getFCE(funcName,
				addrPtr.getName());
		return fce;
	}

	public static String getMemArrName() {
		return getMemArrName("");
	}

	public static String getMemSizeName(String type) {
		return type.toUpperCase() + "_MEMSIZE";
	}

	public static String getMemArrName(String type) {
		return type + "Mem";
	}

	private String replace(StringBuilder in) {
		String content = in.toString();
		String upper = type.toUpperCase();
		return content.replace("$type$", type).replace("$TYPE$", upper)
				.replace("$SIZE$", Integer.toString(memSize))
				.replace("$DEFAULT$", defaultValue);
	}

	public static String getGlobalMemVariables() {
		StringBuffer tmp = new StringBuffer();
		for (MEM_STATES s : MEM_STATES.values()) {
			tmp.append("const int ").append(s.toString()).append(" = ")
					.append(s.ordinal()).append(";" + BR);
		}
		StringBuilder ret = new StringBuilder();
		ret.append(BR);
		ret.append("///////////////////////////" + BR);
		ret.append("// MEMORY ALLOCATION STATES" + BR);
		ret.append("///////////////////////////" + BR);
		ret.append(BR);
		ret.append(tmp.toString());
		ret.append("const int " + Constants.NULL + " = " + Constants.NULL_VALUE
				+ ";" + BR);
		ret.append(BR);
		ret.append("////////////////////////" + BR);
		ret.append("// MEMORY RELATED ERRORS" + BR);
		ret.append("////////////////////////" + BR);
		ret.append(BR);
		ret.append("bool MEM_ERR_ACCESS_FAIL = false;" + BR);
		ret.append("bool MEM_ERR_DOUBLE_FREE = false;" + BR);
		ret.append("bool MEM_ERR_INVALID_ARR_DELETE = false;" + BR);
		ret.append("bool MEM_ERR_INVALID_OBJ_DELETE = false;" + BR);
		ret.append("bool MEM_ERR_DELETE_ON_STATIC_VAR = false;" + BR);
		ret.append("bool MEM_ERR_NOT_ENOUGH_MEM = false;" + BR);
		ret.append("bool MEM_ERR_CORRUPTED_MEM = false;" + BR);
		if (!Engine.USE_DEFRAGMENTATION) {
			// this helps us to recognize we should use the defrag flag
			ret.append("bool MEM_ERR_FRAGMENTED_MEM = false;" + BR);
		}
		return ret.toString();
	}

	public String getVariableDefinitions() {
		StringBuilder ret = new StringBuilder();
		ret.append(BR);
		ret.append("///////////////////" + BR);
		ret.append("// $TYPE$ - globals" + BR);
		ret.append("///////////////////" + BR);
		ret.append(BR);
		ret.append((typedef.equals("") ? "" : typedef + ";" + BR));
		ret.append("const $type$ " + Constants.getDefaultValueName(type)
				+ " = $DEFAULT$;" + BR);
		ret.append("const int $TYPE$_MEMSIZE = $SIZE$;" + BR);
		ret.append("int $TYPE$_FREE_MEM = $TYPE$_MEMSIZE;" + BR);
		ret.append("typedef int[NULL, $TYPE$_MEMSIZE-1] "
				+ VariableConverter.getAddrPtrType(type) + ";" + BR);
		if (memSize > 0) {
			ret.append("int[0, " + (MEM_STATES.values().length - 1)
					+ "] $type$MemState[$TYPE$_MEMSIZE];" + BR);
			ret.append("$type$ " + getMemArrName(type) + "[$TYPE$_MEMSIZE];"
					+ BR);
			if (!Engine.USE_DEFRAGMENTATION) {
				ret.append("bool MEM_ERR_FRAGMENTED_MEM_$TYPE$ = false;" + BR);
			}
			ret.append("bool MEM_ERR_NOT_ENOUGH_MEM_$TYPE$ = false;" + BR);
		}

		return replace(ret);
	}

	public String getHelperFunctionDefinitions() {
		StringBuilder ret = new StringBuilder();
		// TODO mp: this might cause problems when a pointer is declared but
		// never used in the code (int *ptr;) and its the only usage of a
		// pointer or reference of the corresponding data type, as we need
		// automatically generated checks and do not generate the
		// checkfunctions.
		if (memSize > 0) {
			ret.append(BR);
			ret.append("///////////////////" + BR);
			ret.append("// $TYPE$ - helper functions" + BR);
			ret.append("///////////////////" + BR);
			ret.append(BR);
			ret.append("bool is_free_elem_$type$("
					+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
			ret.append("	return $type$MemState[pos] == MEM_FREE;" + BR);
			ret.append("}" + BR);
			ret.append(BR);
			ret.append("bool is_obj_elem_$type$("
					+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
			ret.append("	return $type$MemState[pos] == MEM_STATIC_OBJ || $type$MemState[pos] == MEM_DYN_OBJ;"
					+ BR);
			ret.append("}" + BR);
			ret.append(BR);
			ret.append("bool is_static_elem_$type$("
					+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
			ret.append("	return $type$MemState[pos] == MEM_STATIC_OBJ || $type$MemState[pos] == MEM_STATIC_ARR_HEAD || $type$MemState[pos] == MEM_STATIC_ARR_TAIL;"
					+ BR);
			ret.append("}" + BR);
			ret.append(BR);
			ret.append("bool is_dynamic_elem_$type$("
					+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
			ret.append("	return $type$MemState[pos] == MEM_DYN_OBJ || $type$MemState[pos] == MEM_DYN_ARR_HEAD || $type$MemState[pos] == MEM_DYN_ARR_TAIL;"
					+ BR);
			ret.append("}" + BR);
			ret.append(BR);
			ret.append("bool " + getIsArrHeadElemFuncName() + "_$type$("
					+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
			ret.append("	return $type$MemState[pos] == MEM_STATIC_ARR_HEAD || $type$MemState[pos] == MEM_DYN_ARR_HEAD;"
					+ BR);
			ret.append("}" + BR);
			ret.append(BR);
			ret.append("bool " + getIsArrTailElemFuncName() + "_$type$("
					+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
			ret.append("	return $type$MemState[pos] == MEM_STATIC_ARR_TAIL || $type$MemState[pos] == MEM_DYN_ARR_TAIL;"
					+ BR);
			ret.append("}" + BR);
			ret.append(BR);
			if (Engine.USE_DEFRAGMENTATION) {
				// we need a stack and a heap defragmentation handling the
				// corresponding memory array from different sides. These
				// functions should never be called outside of their wrapper
				// functions (following below).
				ret.append("void defrag_$type$_stack_internal("
						+ VariableConverter.getAddrPtrType(type)
						+ " &reorg[$TYPE$_MEMSIZE+1]) {" + BR);
				ret.append("    int free_pos = 0;" + BR);
				ret.append("    int it_pos = 0;" + BR);
				ret.append("    bool stop = false;" + BR);
				ret.append("    while (!stop && free_pos < $TYPE$_MEMSIZE && !is_dynamic_elem_$type$(free_pos)) {"
						+ BR);
				ret.append("        if ($type$MemState[free_pos] == MEM_FREE) {"
						+ BR);
				ret.append("            it_pos = free_pos;" + BR);
				ret.append("            stop = true;" + BR);
				ret.append("        } else {" + BR);
				ret.append("            free_pos++;" + BR);
				ret.append("        }" + BR);
				ret.append("    }" + BR);
				ret.append("    stop = false;" + BR);
				ret.append("    while (!stop) {" + BR);
				ret.append("        it_pos++;" + BR);
				ret.append("        if (it_pos >= $TYPE$_MEMSIZE || is_dynamic_elem_$type$(it_pos)) {"
						+ BR);
				ret.append("            stop = true;" + BR);
				ret.append("        } else if (is_static_elem_$type$(it_pos)) {"
						+ BR);
				ret.append("            $type$Mem[free_pos] = $type$Mem[it_pos];"
						+ BR);
				ret.append("            $type$Mem[it_pos] = DEFAULT_$TYPE$_VALUE;"
						+ BR);
				ret.append("            reorg[it_pos+1] = free_pos;" + BR);
				ret.append("            $type$MemState[free_pos] = $type$MemState[it_pos];"
						+ BR);
				ret.append("            $type$MemState[it_pos] = MEM_FREE;"
						+ BR);
				ret.append("            free_pos++;" + BR);
				ret.append("        }" + BR);
				ret.append("    }" + BR);
				ret.append("}" + BR);
				ret.append(BR);
				ret.append("void defrag_$type$_heap_internal("
						+ VariableConverter.getAddrPtrType(type)
						+ " &reorg[$TYPE$_MEMSIZE+1]) {" + BR);
				ret.append("    int free_pos = $TYPE$_MEMSIZE - 1;" + BR);
				ret.append("    int it_pos = $TYPE$_MEMSIZE - 1;" + BR);
				ret.append("    bool stop = false;" + BR);
				ret.append("    while (!stop && free_pos >= 0 && !is_static_elem_$type$(free_pos)) {"
						+ BR);
				ret.append("        if ($type$MemState[free_pos] == MEM_FREE) {"
						+ BR);
				ret.append("            it_pos = free_pos;" + BR);
				ret.append("            stop = true;" + BR);
				ret.append("        } else {" + BR);
				ret.append("            free_pos--;" + BR);
				ret.append("        }" + BR);
				ret.append("    }" + BR);
				ret.append("    stop = false;" + BR);
				ret.append("    while (!stop) {" + BR);
				ret.append("        it_pos--;" + BR);
				ret.append("        if (it_pos < 0 || is_static_elem_$type$(it_pos)) {"
						+ BR);
				ret.append("            stop = true;" + BR);
				ret.append("        } else if (is_dynamic_elem_$type$(it_pos)) {"
						+ BR);
				ret.append("            $type$Mem[free_pos] = $type$Mem[it_pos];"
						+ BR);
				ret.append("            $type$Mem[it_pos] = DEFAULT_$TYPE$_VALUE;"
						+ BR);
				ret.append("            reorg[it_pos+1] = it_pos;" + BR);
				ret.append("            $type$MemState[free_pos] = $type$MemState[it_pos];"
						+ BR);
				ret.append("            $type$MemState[it_pos] = MEM_FREE;"
						+ BR);
				ret.append("            free_pos--;" + BR);
				ret.append("        }" + BR);
				ret.append("    }" + BR);
				ret.append("}" + BR);
				ret.append(BR);
				// method for reorganizing all address pointers
				ret.append("void reorganize_"
						+ VariableConverter.getAddrPtrType(type) + "("
						+ VariableConverter.getAddrPtrType(type)
						+ " &reorg[$TYPE$_MEMSIZE+1]) {" + BR);

				// In principal, we have to iterate through the memories of all
				// structs containing at least one address pointer to the
				// current memory and reorganize all members
				// FIXME: this is quiet dirty and non-generic due to the lack of
				// a type system.
				Map<String, List<SCPointer>> memberMap = new HashMap<String, List<SCPointer>>();
				for (TAMemoryType memType : ta.getMemoryTypes()) {
					SCClass cl = sc.getClassByName(memType.getType());
					if (cl != null) {
						for (SCVariable var : cl.getMembers()) {
							if (var instanceof SCPointer) {
								SCPointer ptr = (SCPointer) var;
								if (ptr.getType().equals(this.type)) {
									if (memberMap.get(memType.type) == null) {
										memberMap.put(memType.type,
												new LinkedList<SCPointer>());
									}
									memberMap.get(memType.type).add(ptr);
								}
							}
						}
					}
				}
				if (memberMap.size() > 0) {
					ret.append("    int i;" + BR);
				}

				for (TATemplateInstance ti : ta.getSystemDeclaration().values()) {
					TATemplate temp = ti.getTemplate();
					for (TAVariable var : temp.getLocalVarsAsParameters()) {
						if (var instanceof TAAddressPointer) {
							TAAddressPointer tap = (TAAddressPointer) var;
							if (tap.getMemType().equals(this.type)) {
								String prefixedName = Constants.createDelimitedString(
										ti.getName(), var.getName());
								ret.append("    " + prefixedName + " = reorg["
										+ prefixedName + "+1];" + BR);
							}
						}
					}
				}
				
				for (TAVariable var : ta.getDeclaration()) {
					if (var instanceof TAAddressPointer) {
						TAAddressPointer tap = (TAAddressPointer) var;
						if (tap.getMemType().equals(this.type)) {
							ret.append("    " + var.getName() + " = reorg["
									+ var.getName() + "+1];" + BR);
						}
					}
				}

				for (String structType : memberMap.keySet()) {
					ret.append(BR);
					ret.append("    //reorganize all address pointers inside datatype "
							+ structType + BR);
					ret.append("    for (i = 0; i < "
							+ getMemSizeName(structType) + "; i++) {" + BR);
					for (SCPointer ptr : memberMap.get(structType)) {
						ret.append("        " + getMemArrName(structType)
								+ "[i]." + ptr.getName() + " = reorg["
								+ getMemArrName(structType) + "[i]."
								+ ptr.getName() + "+1];" + BR);
					}
					ret.append("    }" + BR);
				}

				ret.append("}" + BR);
				ret.append(BR);
				// we have three wrapper methods for defragmentation: one that
				// defragments only the stack, one that defragments only the
				// heap
				// and one that handle both.
				ret.append("void defrag_$type$_stack() {" + BR);
				ret.append("	" + VariableConverter.getAddrPtrType(type)
						+ " reorg[$TYPE$_MEMSIZE+1];" + BR);
				ret.append("    int i;" + BR);
				ret.append("    for (i = -1; i < $TYPE$_MEMSIZE; i++) {" + BR);
				ret.append("      reorg[i+1] = i;" + BR);
				ret.append("    }" + BR);
				ret.append("	defrag_$type$_stack_internal(reorg);" + BR);
				ret.append("    reorganize_"
						+ VariableConverter.getAddrPtrType(type) + "(reorg);"
						+ BR);
				ret.append("}" + BR);
				ret.append(BR);
				ret.append("void defrag_$type$_heap() {" + BR);
				ret.append("	" + VariableConverter.getAddrPtrType(type)
						+ " reorg[$TYPE$_MEMSIZE+1];" + BR);
				ret.append("    int i;" + BR);
				ret.append("    for (i = -1; i < $TYPE$_MEMSIZE; i++) {" + BR);
				ret.append("      reorg[i+1] = i;" + BR);
				ret.append("    }" + BR);
				ret.append("	defrag_$type$_heap_internal(reorg);" + BR);
				ret.append("    reorganize_"
						+ VariableConverter.getAddrPtrType(type) + "(reorg);"
						+ BR);
				ret.append("}" + BR);
				ret.append(BR);
				ret.append("void defrag_$type$() {" + BR);
				ret.append("	" + VariableConverter.getAddrPtrType(type)
						+ " reorg[$TYPE$_MEMSIZE+1];" + BR);
				ret.append("    int i;" + BR);
				ret.append("    for (i = -1; i < $TYPE$_MEMSIZE; i++) {" + BR);
				ret.append("      reorg[i+1] = i;" + BR);
				ret.append("    }" + BR);
				ret.append("	defrag_$type$_stack_internal(reorg);" + BR);
				ret.append("	defrag_$type$_heap_internal(reorg);" + BR);
				ret.append("    reorganize_"
						+ VariableConverter.getAddrPtrType(type) + "(reorg);"
						+ BR);
				ret.append("}" + BR);
			}
			ret.append("void check_mem_errors_for_delete_$type$("
					+ VariableConverter.getAddrPtrType(type)
					+ " pos, bool isStatic, bool shouldBeAnArray) {" + BR);
			ret.append("	if (pos == NULL) {" + BR);
			ret.append("		MEM_ERR_CORRUPTED_MEM = true;" + BR);
			ret.append("	} else if (is_free_elem_$type$(pos)) {" + BR);
			ret.append("		MEM_ERR_DOUBLE_FREE = true;" + BR);
			ret.append("	} else if (not is_obj_elem_$type$(pos) && not is_arr_head_elem_$type$(pos)) {"
					+ BR);
			ret.append("		// something went completely wrong :(" + BR);
			ret.append("		// ex: delete somewhere in the middle of an array"
					+ BR);
			ret.append("		MEM_ERR_CORRUPTED_MEM = true;" + BR);
			ret.append("	} else {" + BR);
			ret.append("		if (!isStatic && is_static_elem_$type$(pos)) {" + BR);
			ret.append("			MEM_ERR_DELETE_ON_STATIC_VAR = true;" + BR);
			ret.append("		}" + BR);
			ret.append("		if (shouldBeAnArray) {" + BR);
			ret.append("			if (not is_arr_head_elem_$type$(pos)) {" + BR);
			ret.append("				MEM_ERR_INVALID_ARR_DELETE = true; // shouldn't have called 'delete []' ?"
					+ BR);
			ret.append("			}" + BR);
			ret.append("		} else { // delete normal obj" + BR);
			ret.append("			if (not is_obj_elem_$type$(pos)) {" + BR);
			ret.append("				MEM_ERR_INVALID_OBJ_DELETE = true;" + BR);
			ret.append("			}" + BR);
			ret.append("		}" + BR);
			ret.append("	}" + BR);
			ret.append("}" + BR);
			ret.append(BR);
			ret.append("bool " + getAddrPtrIsInRangeFuncName()
					+ "_$type$(int pos) {" + BR);
			ret.append("	return pos >= 0 && pos < $TYPE$_MEMSIZE;" + BR);
			ret.append("}" + BR);
			ret.append(BR);
			ret.append("bool " + getAddrChkFuncName() + "_$type$(int pos) {"
					+ BR);
			ret.append("	return " + getAddrPtrIsInRangeFuncName()
					+ "_$type$(pos) && $type$MemState[pos] != MEM_FREE;" + BR);
			ret.append("}" + BR);

			ret.append(BR);
			ret.append("bool " + getIsArrElemFuncName()
					+ "_$type$(int head, int index) {" + BR);
			ret.append("	int i;" + BR);
			ret.append("	if (!" + getAddrPtrIsInRangeFuncName()
					+ "_$type$(head + index)) {" + BR);
			ret.append("		return false;" + BR);
			ret.append("	}" + BR);
			ret.append("	if (!" + getIsArrHeadElemFuncName()
					+ "_$type$(head)) {" + BR);
			ret.append("		return false;" + BR);
			ret.append("	}" + BR);
			ret.append("	if (index < 0) { // assuming heap/stack arrays grow leftward"
					+ BR);
			ret.append("		return false;" + BR);
			ret.append("	}" + BR);
			ret.append("	for (i = head + 1; i <= head + index; i++) {" + BR);
			ret.append("		if (!" + getIsArrTailElemFuncName() + "_$type$(i)) {"
					+ BR);
			ret.append("			return false;" + BR);
			ret.append("		}" + BR);
			ret.append("	}" + BR);
			ret.append("	return true;" + BR);
			ret.append("}" + BR);
		}
		return replace(ret);
	}

	/**
	 * Obj = non-array variable/'normal' variable Var = variable (array or not)
	 * 
	 * @return
	 */
	public String getFunctionDefinitions() {
		StringBuilder ret = new StringBuilder();
		ret.append(BR);
		ret.append("///////////////////" + BR);
		ret.append("// $TYPE$ - private" + BR);
		ret.append("///////////////////" + BR);
		ret.append(BR);
		ret.append("void allocate_and_init_$type$(int memState, "
				+ VariableConverter.getAddrPtrType(type)
				+ " pos, $type$ value) {" + BR);
		ret.append("	if (pos == NULL) {" + BR);
		ret.append("		MEM_ERR_CORRUPTED_MEM = true;" + BR);
		ret.append("	} else {" + BR);
		ret.append("		$type$MemState[pos] = memState;" + BR);
		ret.append("		$type$Mem[pos] = value;" + BR);
		ret.append("        if (memState == MEM_FREE) {" + BR);
		ret.append("            $TYPE$_FREE_MEM++;" + BR);
		ret.append("        } else {" + BR);
		ret.append("            $TYPE$_FREE_MEM--;" + BR);
		ret.append("        }" + BR);
		ret.append("	}" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append("void allocate_$type$(int memState, "
				+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
		ret.append("	"
				+ "allocate_and_init_$type$(memState, pos, DEFAULT_$TYPE$_VALUE);"
				+ BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append(VariableConverter.getAddrPtrType(type)
				+ " find_free_pos_$type$(int memState) {" + BR);
		ret.append("	" + VariableConverter.getAddrPtrType(type)
				+ " pos = NULL;" + BR);
		if (Engine.USE_DEFRAGMENTATION) {
			ret.append("	if (memState == MEM_STATIC_OBJ) { // stack" + BR);
			ret.append("		for (pos = 0; pos <= $TYPE$_MEMSIZE - 1; pos++) {"
					+ BR);
			ret.append("			if ($type$MemState[pos] == MEM_FREE) {" + BR);
			ret.append("				return pos;" + BR);
			ret.append("			} else if (is_dynamic_elem_$type$($type$MemState[pos])) {"
					+ BR);
			// if we encounter a dynamic memory element before we encounter a
			// free element we have a fragmented heap
			ret.append("                defrag_$type$_heap();" + BR);
			// afterwards, allocation should always work, still we do some error
			// handling
			ret.append("		        for (pos = 0; pos <= $TYPE$_MEMSIZE - 1; pos++) {"
					+ BR);
			ret.append("                    if ($type$MemState[pos] == MEM_FREE) {"
					+ BR);
			ret.append("				        return pos;" + BR);
			ret.append("			        }" + BR);
			ret.append("                }" + BR);
			ret.append("                MEM_ERR_CORRUPTED_MEM = true;" + BR);
			ret.append("				return NULL;" + BR);
			ret.append("            }" + BR);
			ret.append("		}" + BR);
			ret.append("	} else if (memState == MEM_DYN_OBJ) { // heap" + BR);
			ret.append("		for (pos = $TYPE$_MEMSIZE - 1; pos >= 0; pos--) {"
					+ BR);
			ret.append("			if ($type$MemState[pos] == MEM_FREE) {" + BR);
			ret.append("				return pos;" + BR);
			ret.append("			} else if (is_static_elem_$type$($type$MemState[pos])) {"
					+ BR);
			// if we encounter a static memory element before we encounter a
			// free element we have a fragmented stack
			ret.append("                defrag_$type$_stack();" + BR);
			// afterwards, allocation should always work, still we do some error
			// handling
			ret.append("		        for (pos = $TYPE$_MEMSIZE - 1; pos >= 0; pos--) {"
					+ BR);
			ret.append("                    if ($type$MemState[pos] == MEM_FREE) {"
					+ BR);
			ret.append("				        return pos;" + BR);
			ret.append("			        }" + BR);
			ret.append("                }" + BR);
			ret.append("                MEM_ERR_CORRUPTED_MEM = true;" + BR);
			ret.append("				return NULL;" + BR);
			ret.append("            }" + BR);
			ret.append("		}" + BR);
			ret.append("	}" + BR);
			ret.append("	return NULL;" + BR);
		} else {
			ret.append("	if (memState == MEM_STATIC_OBJ) { // stack" + BR);
			ret.append("		for(pos = 0; pos <= $TYPE$_MEMSIZE - 1; pos++) {"
					+ BR);
			ret.append("			if ($type$MemState[pos] == MEM_FREE) {" + BR);
			ret.append("				return pos;" + BR);
			ret.append("			}" + BR);
			ret.append("		}" + BR);
			ret.append("	} else if (memState == MEM_DYN_OBJ) { // heap" + BR);
			ret.append("		for(pos = $TYPE$_MEMSIZE - 1; pos >= 0; pos--) {"
					+ BR);
			ret.append("			if ($type$MemState[pos] == MEM_FREE) {" + BR);
			ret.append("				return pos;" + BR);
			ret.append("			}" + BR);
			ret.append("		}" + BR);
			ret.append("	}" + BR);
			ret.append("	return NULL;" + BR);
		}
		ret.append("}" + BR);
		ret.append(BR);
		ret.append(VariableConverter.getAddrPtrType(type)
				+ " find_free_pos_and_allocate_and_init_$type$(int memState, $type$ value) {"
				+ BR);
		ret.append("	" + VariableConverter.getAddrPtrType(type) + " pos;" + BR);
		ret.append("	if ($TYPE$_FREE_MEM == 0) {" + BR);
		ret.append("		MEM_ERR_NOT_ENOUGH_MEM = true;" + BR);
		ret.append("		MEM_ERR_NOT_ENOUGH_MEM_$TYPE$ = true;" + BR);
		ret.append("		return NULL;" + BR);
		ret.append("	}" + BR);
		ret.append("	pos = find_free_pos_$type$(memState);" + BR);
		// the allocation can only fail if something went totally wrong
		ret.append("	if (pos == NULL) {" + BR);
		ret.append("		MEM_ERR_CORRUPTED_MEM = true;" + BR);
		ret.append("	} else {" + BR);
		ret.append("		" + "allocate_and_init_$type$(memState, pos, value);"
				+ BR);
		ret.append("	}" + BR);
		ret.append("	return pos;" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append(VariableConverter.getAddrPtrType(type)
				+ " find_free_pos_and_allocate_$type$(int memState) {" + BR);
		ret.append("	return find_free_pos_and_allocate_and_init_$type$(memState, "
				+ Constants.getDefaultValueName(type) + ");" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append("void delete_$type$("
				+ VariableConverter.getAddrPtrType(type)
				+ " pos, bool isStatic) {" + BR);
		ret.append("	check_mem_errors_for_delete_$type$(pos, isStatic, false);"
				+ BR);
		ret.append("	allocate_and_init_$type$(MEM_FREE, pos, "
				+ Constants.getDefaultValueName(type) + ");" + BR);
		ret.append("}" + BR);
		ret.append("//////////////////" + BR);
		ret.append("// $TYPE$ - public" + BR);
		ret.append("//////////////////" + BR);
		ret.append(BR);
		ret.append(VariableConverter.getAddrPtrType(type) + " "
				+ getAllocStaticFuncName() + "_$type$() {" + BR);
		ret.append("	return find_free_pos_and_allocate_and_init_$type$(MEM_STATIC_OBJ, "
				+ Constants.getDefaultValueName(type) + ");" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append(VariableConverter.getAddrPtrType(type) + " "
				+ getAllocDynFuncName() + "_$type$() {" + BR);
		ret.append("	return find_free_pos_and_allocate_and_init_$type$(MEM_DYN_OBJ, "
				+ Constants.getDefaultValueName(type) + ");" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append(VariableConverter.getAddrPtrType(type) + " "
				+ getAllocStaticFuncName("dummy") + "_$type$($type$ value) {"
				+ BR);
		ret.append("	return find_free_pos_and_allocate_and_init_$type$(MEM_STATIC_OBJ, value);"
				+ BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append(VariableConverter.getAddrPtrType(type) + " "
				+ getAllocDynFuncName("dummy") + "_$type$($type$ value) {" + BR);
		ret.append("	return find_free_pos_and_allocate_and_init_$type$(MEM_DYN_OBJ, value);"
				+ BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append("void " + getDeleteStaticFuncName() + "_$type$("
				+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
		ret.append("	delete_$type$(pos, true);" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append("void " + getDeleteDynFuncName() + "_$type$("
				+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
		ret.append("	delete_$type$(pos, false);" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		if (!Engine.USE_DEFRAGMENTATION) {
			ret.append("void " + getRemoveFromStackFuncName() + "_$type$("
					+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
			ret.append("	if (pos != NULL) {" + BR);
			ret.append("	    allocate_and_init_$type$(MEM_FREE, pos, "
					+ Constants.getDefaultValueName(type) + ");" + BR);
			ret.append("	}" + BR);
			ret.append("}" + BR);
		} else {
			ret.append("void " + getRemoveFromStackFuncName() + "_$type$("
					+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
			ret.append("	if (pos != NULL) {" + BR);
			ret.append("		allocate_and_init_$type$(MEM_FREE, pos, "
					+ Constants.getDefaultValueName(type) + ");" + BR);
			ret.append("	}" + BR);
			ret.append("}" + BR);
		}
		ret.append(BR);
		ret.append(BR);
		ret.append("/////////////////////////" + BR);
		ret.append("// $TYPE$ ARRAY - private" + BR);
		ret.append("/////////////////////////" + BR);
		ret.append(BR);
		ret.append(VariableConverter.getAddrPtrType(type)
				+ " find_free_pos_arr_$type$(int memState, int length) {" + BR);
		ret.append("	int count = 0;" + BR);
		ret.append("	" + VariableConverter.getAddrPtrType(type) + " pos = -1;"
				+ BR);
		ret.append("	int i = 0;" + BR);
		ret.append("	if (memState == MEM_STATIC_ARR_HEAD) {" + BR);
		ret.append("		for (i = 0; i <= $TYPE$_MEMSIZE - 1; i++) {" + BR);
		ret.append("			if ($type$MemState[i] == MEM_FREE) {" + BR);
		ret.append("				if (count == 0) {" + BR);
		ret.append("					pos = i;" + BR);
		ret.append("				}" + BR);
		ret.append("				count++;" + BR);
		ret.append("				if (count == length) {" + BR);
		ret.append("					return pos;" + BR);
		ret.append("				}" + BR);
		ret.append("			} else if (is_static_elem_$type$(i)) {" + BR);
		ret.append("				count = 0;" + BR);
		ret.append("				pos = NULL;" + BR);
		ret.append("			} else {" + BR);
		ret.append("				return NULL;" + BR);
		ret.append("			}" + BR);
		ret.append("		}" + BR);
		ret.append("	} else if (memState == MEM_DYN_ARR_HEAD) {" + BR);
		ret.append("		for(i = $TYPE$_MEMSIZE - 1; i >= 0; i--) {" + BR);
		ret.append("			if ($type$MemState[i] == MEM_FREE) {" + BR);
		ret.append("				if (count == 0) {" + BR);
		ret.append("					pos = i;" + BR);
		ret.append("				}" + BR);
		ret.append("				count++;" + BR);
		ret.append("				if (count == length) {" + BR);
		ret.append("					return pos - length + 1; // return the 'lower' index w.r.t. the memArray; + 1 because count starts at 0"
				+ BR);
		ret.append("				}" + BR);
		ret.append("			} else if (is_dynamic_elem_$type$(i)) {" + BR);
		ret.append("				count = 0;" + BR);
		ret.append("				pos = NULL;" + BR);
		ret.append("			} else {" + BR);
		ret.append("				return NULL;" + BR);
		ret.append("			}" + BR);
		ret.append("		}" + BR);
		ret.append("	}" + BR);
		ret.append("	return NULL;" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		if (!Engine.USE_DEFRAGMENTATION) {
			ret.append(VariableConverter.getAddrPtrType(type) + " "
					+ "allocate_arr_$type$(int memState, int length) {" + BR);
			ret.append("	int i;" + BR);
			ret.append("    " + VariableConverter.getAddrPtrType(type)
					+ " pos;" + BR);
			ret.append("    int tailElemStateType;" + BR);
			ret.append("    if ($TYPE$_FREE_MEM < length) {" + BR);
			ret.append("		MEM_ERR_NOT_ENOUGH_MEM = true;" + BR);
			ret.append("		MEM_ERR_NOT_ENOUGH_MEM_$TYPE$ = true;" + BR);
			ret.append("		return NULL;" + BR);
			ret.append("	}" + BR);
			ret.append("	pos = find_free_pos_arr_$type$(memState, length);"
					+ BR);
			ret.append("	if (pos == NULL) {" + BR);
			ret.append("		MEM_ERR_FRAGMENTED_MEM = true;" + BR);
			ret.append("		MEM_ERR_FRAGMENTED_MEM_$TYPE$ = true;" + BR);
			ret.append("		return NULL;" + BR);
			ret.append("	}" + BR);
			ret.append("	if (memState == MEM_STATIC_ARR_HEAD) {" + BR);
			ret.append("		tailElemStateType = MEM_STATIC_ARR_TAIL;" + BR);
			ret.append("	} else {" + BR);
			ret.append("		tailElemStateType = MEM_DYN_ARR_TAIL;" + BR);
			ret.append("	}" + BR);
			ret.append("	allocate_$type$(memState, pos); // allocate head" + BR);
			ret.append("	for(i = pos + 1; i < (pos + length); i++) {" + BR);
			ret.append("		allocate_$type$(tailElemStateType, i);" + BR);
			ret.append("	}" + BR);
			ret.append("	return pos;" + BR);
			ret.append("}" + BR);
		} else {
			ret.append(VariableConverter.getAddrPtrType(type) + " "
					+ "allocate_arr_$type$(int memState, int length) {" + BR);
			ret.append("	int i;" + BR);
			ret.append("    " + VariableConverter.getAddrPtrType(type)
					+ " pos;" + BR);
			ret.append("	int tailElemStateType;" + BR);
			ret.append("    if ($TYPE$_FREE_MEM < length) {" + BR);
			ret.append("		MEM_ERR_NOT_ENOUGH_MEM = true;" + BR);
			ret.append("		MEM_ERR_NOT_ENOUGH_MEM_$TYPE$ = true;" + BR);
			ret.append("		return NULL;" + BR);
			ret.append("	}" + BR);
			ret.append("	pos = find_free_pos_arr_$type$(memState, length);"
					+ BR);
			ret.append("	if (pos == NULL) {" + BR);
			ret.append("		defrag_$type$();" + BR);
			ret.append("	    pos = find_free_pos_arr_$type$(memState, length);"
					+ BR);
			ret.append("	}" + BR);
			ret.append("	if (memState == MEM_STATIC_ARR_HEAD) {" + BR);
			ret.append("		tailElemStateType = MEM_STATIC_ARR_TAIL;" + BR);
			ret.append("	} else {" + BR);
			ret.append("		tailElemStateType = MEM_DYN_ARR_TAIL;" + BR);
			ret.append("	}" + BR);
			ret.append("	allocate_$type$(memState, pos); // allocate head" + BR);
			ret.append("	for(i = pos + 1; i < (pos + length); i++) {" + BR);
			ret.append("		allocate_$type$(tailElemStateType, i);" + BR);
			ret.append("	}" + BR);
			ret.append("	return pos;" + BR);
			ret.append("}" + BR);
		}
		ret.append(BR);
		ret.append("void delete_arr_$type$("
				+ VariableConverter.getAddrPtrType(type)
				+ " pos, bool isStatic) {" + BR);
		ret.append("	int i = pos + 1;" + BR);
		ret.append("	check_mem_errors_for_delete_$type$(pos, isStatic, true);"
				+ BR);
		ret.append("	allocate_and_init_$type$(MEM_FREE, pos, "
				+ Constants.getDefaultValueName(type) + ");" + BR);
		ret.append("	while(i < $TYPE$_MEMSIZE && is_arr_tail_elem_$type$(i)) {"
				+ BR);
		ret.append("		allocate_and_init_$type$(MEM_FREE, i, "
				+ Constants.getDefaultValueName(type) + ");" + BR);
		ret.append("		i++;" + BR);
		ret.append("	}" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append("////////////////////////" + BR);
		ret.append("// $TYPE$ ARRAY - public" + BR);
		ret.append("////////////////////////" + BR);
		ret.append(BR);
		ret.append(VariableConverter.getAddrPtrType(type) + " "
				+ getAllocStaticArrFuncName() + "_$type$(int length) {" + BR);
		ret.append("	return "
				+ "allocate_arr_$type$(MEM_STATIC_ARR_HEAD, length);" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append(VariableConverter.getAddrPtrType(type) + " "
				+ getAllocDynArrFuncName() + "_$type$(int length) {" + BR);
		ret.append("	return allocate_arr_$type$(MEM_DYN_ARR_HEAD, length);"
				+ BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append("void " + getDeleteStaticArrFuncName() + "_$type$("
				+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
		ret.append("	" + "delete_arr_$type$(pos, true);" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append("void " + getDeleteDynArrFuncName() + "_$type$("
				+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
		ret.append("	delete_arr_$type$(pos, false);" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append("void " + getRemoveArrFromStackFuncName() + "_$type$("
				+ VariableConverter.getAddrPtrType(type) + " pos) {" + BR);
		ret.append("	int i = pos + 1;" + BR);
		ret.append("	if (pos != NULL) {" + BR);
		ret.append("		allocate_and_init_$type$(MEM_FREE, pos, "
				+ Constants.getDefaultValueName(type) + ");" + BR);
		ret.append("		while(i < $TYPE$_MEMSIZE && is_arr_tail_elem_$type$(i)) {"
				+ BR);
		ret.append("			allocate_and_init_$type$(MEM_FREE, i, "
				+ Constants.getDefaultValueName(type) + ");" + BR);
		ret.append("			i++;" + BR);
		ret.append("		}" + BR);
		ret.append("	}" + BR);
		ret.append("}" + BR);
		ret.append(BR);
		ret.append("void " + getInitArrayElemFuncName() + "_$type$("
				+ VariableConverter.getAddrPtrType(type)
				+ " pos, $type$ value) {" + BR);
		ret.append("	if (pos == NULL) {" + BR);
		ret.append("		MEM_ERR_CORRUPTED_MEM = true;" + BR);
		ret.append("	} else {" + BR);
		ret.append("		" + getMemArrName(type) + "[pos] = value;" + BR);
		ret.append("	}" + BR);
		ret.append("}" + BR);
		return replace(ret);
	}
}
