/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.io.IOException;

import de.tub.pes.sc2uppaal.util.XMLWriter;
import de.tub.pes.syscir.sc_model.expressions.Expression;

/**
 * Represents a location in an Uppaal template. See Uppaal documentation for
 * details on locations.
 * 
 * @author Joachim Fellmuth, Marcel Pockrandt
 * 
 */
public class TALocation {

	/**
	 * Unique location id count.
	 */
	private static int idCnt = 0;

	/**
	 * Creates a new unique id number.
	 * 
	 * @return Unique location id
	 */
	public static int getUniqueID() {
		return idCnt++;
	}

	/**
	 * Optional display name of the location.
	 */
	private String name = null;

	/**
	 * ID of this location.
	 */
	private int id;

	/**
	 * Urgent flag
	 */
	private boolean urgent;

	/**
	 * Committed flag
	 */
	private boolean committed;

	/**
	 * Display position of the location within the template. Optional.
	 */
	private int posx = 0, posy = 0;

	/**
	 * Flag to determine if location position was set.
	 */
	private boolean setPos = false;

	/**
	 * Invariant expression for this location. As in Uppaal, setting the
	 * invariant is optional and the default value is "true" in Uppaal.
	 */
	private Expression invariant = null;

	/**
	 * Creates a regular anonymous location.
	 */
	public TALocation() {
		this.id = getUniqueID();
	}

	/**
	 * Creates a regular named location.
	 * 
	 * @param name
	 */
	public TALocation(String name) {
		this();
		if (!("".equals(name))) {
			this.name = name;
		}
	}

	/**
	 * Set the display position of the location within the template. See Uppaal
	 * for rlative values.
	 * 
	 * @param x
	 * @param y
	 */
	public void setPosition(int x, int y) {
		this.posx = x;
		this.posy = y;
	}

	/**
	 * @return Display name of location
	 */
	public String getName() {
		return name;
	}

	/**
	 * Sets the optional display name of the location.
	 * 
	 * @param name
	 *            Optional location name
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Returns the unique id of the location.
	 * 
	 * @return ID
	 */
	public int getId() {
		return id;
	}

	/**
	 * To determine if location is urgent.
	 * 
	 * @return true if urgent else false
	 */
	public boolean isUrgent() {
		return urgent;
	}

	/**
	 * Sets the urgent flag of the location.
	 * 
	 * @param urgent
	 */
	public void setUrgent(boolean urgent) {
		this.urgent = urgent;
	}

	/**
	 * To determine if the location is committed.
	 * 
	 * @return true if committed, else false
	 */
	public boolean isCommitted() {
		return committed;
	}

	/**
	 * Sets the committed flag of the location.
	 * 
	 * @param committed
	 */
	public void setCommitted(boolean committed) {
		this.committed = committed;
	}

	/**
	 * Returns the invariant expression. Null if not set.
	 * 
	 * @return invariant expression or null if not set
	 */
	public Expression getInvariant() {
		return invariant;
	}

	/**
	 * Sets a new invariant expression for this location. See Uppaal
	 * documentation on invariants for restrictions.
	 * 
	 * @param invariant
	 */
	public void setInvariant(Expression invariant) {
		this.invariant = invariant;
	}

	/**
	 * Returns horizontal display position or 0 if not set.
	 * 
	 * @return
	 */
	public int getPositionX() {
		return this.posx;
	}

	/**
	 * Returns vertical display position or 0 if not set.
	 * 
	 * @return
	 */
	public int getPositionY() {
		return this.posy;
	}

	public String toString() {
		return "(" + id + (urgent ? ", u" : (committed ? ", c" : ""))
				+ (name != null ? (", " + name) : "") + ")";
	}

	/**
	 * Prints the XML representation of this location to the given stream writer
	 * as required in the Uppaal model language.
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void print(XMLWriter writer) throws IOException {
		String pos = setPos ? " x=\"" + posx + "\" y=\"" + posy + "\"" : "";

		writer.startTag("location id=\"id" + id + "\"" + pos);
		if (name != null && name.length() > 0)
			writer.startTag("name").writeText(name).endTag("name");

		if (invariant != null) {
			writer.startTag("label kind=\"invariant\"");
			writer.writeTextLine(invariant, true);
			writer.endTag("label");
		}

		if (urgent)
			writer.writeEmptyTag("urgent");

		if (committed)
			writer.writeEmptyTag("committed");

		writer.endTag("location");
	}

	// TODO
	// public TALocation createCopy() {
	// TALocation newL = new TALocation();
	// newL.id = id;
	// newL.name = name;
	// newL.urgent = urgent;
	// newL.committed = committed;
	// newL.posx = posx;
	// newL.posy = posy;
	// newL.setPos = setPos;
	// if (invariant != null) {
	// newL.invariant = invariant.createCopy();
	// }
	// return newL;
	// }
}
