/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.io.IOException;

import de.tub.pes.sc2uppaal.util.XMLWriter;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;

/**
 * Represents the Uppaal integer type, as well as the constant version of it.
 * Also holds the integer bit size and the initialization value.
 * 
 * @author Joachim Fellmuth, Paula Herber, Marcel Pockrandt
 * 
 */
public class TAInteger extends TAVariable {
	/**
	 * The bit size of this integer. Initial value is 32, like in SystemC.
	 */
	private int bits = 16;

	/**
	 * Maximal value, could be a variable. Overwrites bit width!
	 */
	private String max;

	/**
	 * Minimal value, could be a variable.
	 */
	private String min = "0";

	/**
	 * Simple constructor creates a TAVariable without initialization, not
	 * constant, no reference.
	 * 
	 * @param name
	 *            Name of the variable
	 */
	public TAInteger(String name) {
		super(name, "int");
	}

	public TAInteger(String name, Expression initValue) {
		this(name);
		setInitValue(initValue);
	}

	public TAInteger(String name, int initValue) {
		this(name);
		setInitValue(new ConstantExpression(null, Integer.toString(initValue)));
	}

	public String getMax() {
		return this.max;
	}

	public void setMax(String max) {
		this.max = max;
	}

	public String getMin() {
		return this.min;
	}

	public void setMin(String min) {
		this.min = min;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.tub.pes.state.model.TAVariable#createCopy(java.lang.String)
	 */
	@Override
	public TAVariable createCopy(String prefix) {
		TAInteger copy = new TAInteger(prefix + this.name);
		copyAttributesExceptName(copy);
		copy.setBits(bits);
		copy.setMax(max);
		copy.setMin(min);
		return copy;
	}

	@Override
	public String createDeclarationString() {
		String out = isConst2Str();
		if (bits == 16 && max == null) {
			out += "int " + name
					+ (hasInitValue() ? " = " + initValue.toStringNoSem() : "")
					+ ";" + Constants.END_LINE;
		} else {
			if (max == null) {
				long left = -((long) 1 << ((long) bits));
				long right = -left;
				out += "int["
						+ left
						+ ", "
						+ right
						+ "] "
						+ name
						+ (hasInitValue()
								? " = " + initValue.toStringNoSem()
								: "") + ";" + Constants.END_LINE;
			} else {
				/*
				 * would be more appropriate int left = -(1 << (bits - 1)); int
				 * right = -left -1;
				 */
				out += "int["
						+ min
						+ ", "
						+ max
						+ "] "
						+ name
						+ (hasInitValue()
								? " = " + initValue.toStringNoSem()
								: "") + ";" + Constants.END_LINE;
			}
		}
		return out;
	}

	@Override
	public void printAsParam(XMLWriter writer) throws IOException {
		String out = isConst2Str();
		if (bits == 16 && max == null) {
			out += "int " + getRefParamPrefix() + name;
		} else {
			if (max == null) {
				long left = -((long) 1 << ((long) bits));
				long right = -left;

				/*
				 * would be more appropriate int left = -(1 << (bits - 1)); int
				 * right = -left -1;
				 */
				out += "int[" + left + ", " + right + "] "
						+ getRefParamPrefix() + name;
			} else
				out += "int[" + min + ", " + max + "] " + getRefParamPrefix()
						+ name;
		}
		writer.writeText(out);
	}

	/**
	 * Returns the bit size of this variable. If it was not set, 32 is returned.
	 * 
	 * @return variable bit size
	 */
	public int getBits() {
		return bits;
	}

	/**
	 * Changes the bit size of this variable.
	 * 
	 * @param bits
	 */
	public void setBits(int bits) {
		this.bits = bits;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.tub.pes.state.model.TAVariable#getDummyName(de.tub.pes.state.model
	 * .TAModel)
	 */
	@Override
	public String getDummyName(TAModel ta) {
		String name = "dummyinteger" + ((isConstant()) ? "c" : "") + bits;
		if (ta.getGlobalVariable(name) == null) {
			TAInteger newVar = new TAInteger(name);
			newVar.setBits(bits);
			newVar.setConstant(isConstant());
			ta.addVariable(newVar);
		}
		return name;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + bits;
		result = prime * result + ((max == null) ? 0 : max.hashCode());
		result = prime * result + ((min == null) ? 0 : min.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (getClass() != obj.getClass())
			return false;
		TAInteger other = (TAInteger) obj;
		if (bits != other.bits)
			return false;
		if (max == null) {
			if (other.max != null)
				return false;
		} else if (!max.equals(other.max))
			return false;
		if (min == null) {
			if (other.min != null)
				return false;
		} else if (!min.equals(other.min))
			return false;
		return true;
	}

}
