/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.expressions.TAFunctionCallExpression;
import de.tub.pes.sc2uppaal.util.XMLWriter;
import de.tub.pes.syscir.engine.util.Pair;
import de.tub.pes.syscir.sc_model.SCFunction;
import de.tub.pes.syscir.sc_model.SCParameter;
import de.tub.pes.syscir.sc_model.SCREFERENCETYPE;
import de.tub.pes.syscir.sc_model.SCVariable;
import de.tub.pes.syscir.sc_model.expressions.BinaryExpression;
import de.tub.pes.syscir.sc_model.expressions.BreakExpression;
import de.tub.pes.syscir.sc_model.expressions.CaseExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.EmptyExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.ExpressionBlock;
import de.tub.pes.syscir.sc_model.expressions.FunctionCallExpression;
import de.tub.pes.syscir.sc_model.expressions.IfElseExpression;
import de.tub.pes.syscir.sc_model.expressions.OutputExpression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableDeclarationExpression;
import de.tub.pes.syscir.sc_model.expressions.SwitchExpression;

/**
 * 
 * @author Pfeffer, Timm Liebrenz
 * 
 */
public class TAFunction implements Serializable {

	private static final long serialVersionUID = -3958495317708875177L;

	private static transient final Logger logger = LogManager
			.getLogger(TAFunction.class.getName());

	/**
	 * name of the function
	 */
	protected String name;
	/**
	 * list of SCVariable which represent the parameters
	 */
	protected List<SCParameter> parameters;
	/**
	 * the list of expressions, which represent the body
	 */
	protected List<Expression> body;
	/**
	 * the list of the local variables, this contains also the parameters
	 */
	protected List<TAVariable> localVariables;
	/**
	 * the list of the local variables which have to be passed as parameters
	 * (usually due to the memory model)
	 */
	protected List<TAVariable> localVariablesAsParameters;
	/**
	 * the sc_function this ta function is based on. this usually only exists
	 * for methods generated using the optimized method transformer
	 */
	protected SCFunction originalFunction;

	/**
	 * the return type
	 */
	protected String returnType;
	/**
	 * A list of other functions which are called by this function.
	 */
	protected List<FunctionCallExpression> functionCalls;
	/**
	 * A list of other functions name which are called by this functions.
	 */
	protected List<String> calledNames;

	static public class Builder {

		private final String name;
		private SCFunction function;
		private List<SCParameter> parameters;
		private List<Expression> body;
		private List<TAVariable> localVariables;
		private List<TAVariable> localVariablesAsParameters;
		private final String returnType;

		public Builder(String name, String returnType) {
			this.name = name;
			this.returnType = returnType;
			this.function = null;
			this.parameters = new ArrayList<SCParameter>();
			this.body = new ArrayList<Expression>();
			this.localVariables = new ArrayList<TAVariable>();
			this.localVariablesAsParameters = new ArrayList<TAVariable>();
		}

		public Builder parameters(List<SCParameter> params) {
			this.parameters = params;
			return this;
		}

		public Builder body(List<Expression> body) {
			this.body = body;
			return this;
		}

		public Builder function(SCFunction function) {
			this.function = function;
			return this;
		}

		public Builder body(Expression singleExpr) {
			List<Expression> ls = new LinkedList<Expression>();
			ls.add(singleExpr);
			return body(ls);
		}

		public Builder localVariables(List<TAVariable> localVariables) {
			this.localVariables = localVariables;
			return this;
		}

		public Builder localVariable(TAVariable localVariables) {
			this.localVariables = new LinkedList<TAVariable>();
			this.localVariables.add(localVariables);
			return this;
		}

		// for your convenience ;)
		public Builder localVariablesFromSCFunction(
				List<SCVariable> localVariables) {
			List<TAVariable> localVarsTA = new LinkedList<TAVariable>();
			for (SCVariable scv : localVariables) {
				// do not add class instances to the local variables
				// they are handled as global variables in the TA
				if (!scv.isSCClassInstance()) {
					localVarsTA.add(VariableConverter.convertVariable(scv));
				}
			}
			this.localVariables = localVarsTA;
			return this;
		}

		public Builder localVariablesFromTATemplate(List<TAVariable> localVars) {
			this.localVariables = localVars;
			return this;
		}

		public Builder localVariablesAsParametersFromTATemplate(
				List<TAVariable> localVarsAsParameters) {
			this.localVariablesAsParameters = localVarsAsParameters;
			return this;
		}

		public TAFunction build() {
			return new TAFunction(this);
		}

	}

	private TAFunction(Builder builder) {
		this.name = builder.name;
		this.originalFunction = builder.function;
		this.returnType = builder.returnType;
		this.parameters = builder.parameters;
		this.localVariables = builder.localVariables;
		this.localVariablesAsParameters = builder.localVariablesAsParameters;
		this.body = builder.body;
	}

	/**
	 * Factory constructor to create TAFunction from SCFunctions
	 * 
	 * @param scf
	 * @return
	 */
	public static TAFunction fromSCFunction(SCFunction scf) {
		if (scf == null)
			throw new NullPointerException();

		TAFunction taf = new TAFunction.Builder(scf.getName(),
				scf.getReturnType()).function(scf)
				.parameters(scf.getParameters())
				.localVariablesFromSCFunction(scf.getLocalVariables()).build();
		taf.setFunctionCalls(scf.getFunctionCalls());
		taf.setBody(scf.getBody());
		return taf;
	}

	public static TAFunction fromTATemplate(TATemplate tmp) {
		if (tmp == null)
			throw new NullPointerException();
		SCFunction scf = tmp.getFunction();
		TAFunction taf = new TAFunction.Builder(tmp.getName(),
				scf.getReturnType())
				.function(scf)
				.parameters(scf.getParameters())
				.localVariablesFromTATemplate(tmp.getLocalVars())
				.localVariablesAsParametersFromTATemplate(
						tmp.getLocalVarsAsParameters()).build();
		taf.setFunctionCalls(scf.getFunctionCalls());
		taf.setBody(scf.getBody());
		return taf;
	}

	public void setBody(List<Expression> body) {
		this.body = body;
		translateBody();
	}

	public void setParameters(List<SCParameter> params) {
		this.parameters = params;
	}

	public void setFunctionCalls(List<FunctionCallExpression> fcs) {
		this.functionCalls = fcs;
		this.calledNames = new LinkedList<String>();
		for (FunctionCallExpression fce : functionCalls) {
			calledNames.add(fce.getFunction().getName());
		}
	}

	public String getName() {
		return name;
	}

	public List<Expression> getBody() {
		return body;
	}

	public List<FunctionCallExpression> getFunctionCalls() {
		return functionCalls;
	}

	public List<String> getCalledNames() {
		return calledNames;
	}

	public List<SCParameter> getParameters() {
		return parameters;
	}

	/**
	 * This method is used to find all transformed function calls in an
	 * Expression. Hence, all InnerExpressions are checked too.
	 * 
	 * @param e
	 * @return Pairs of expressions, consisting of the transformed function call
	 *         and their original expression
	 */
	private List<Pair<Expression, Expression>> findFunctionCalls(Expression e) {
		ArrayList<Pair<Expression, Expression>> fcs = new ArrayList<Pair<Expression, Expression>>();
		if (e instanceof TAFunctionCallExpression) {
			// add pair to replacements
			fcs.add(new Pair<Expression, Expression>(e,
					((TAFunctionCallExpression) e).getOriginal()));
		} else {
			// check inner expressions for further transformed function calls
			if (!e.getInnerExpressions().isEmpty()) {
				for (Expression exp : e.getInnerExpressions()) {
					fcs.addAll(findFunctionCalls(exp));
				}
			}
		}
		return fcs;
	}

	/**
	 * This method translates all transformed function calls
	 * (TAFunctionCallExpressions) to their original form
	 * (FunctionCallExpression). Since the original form might have been
	 * transformed too, replacement loops till no further transformed function
	 * calls can be identified.
	 * 
	 * @param taf
	 */
	private void translateFunctionCalls() {
		assert this.getFunctionCalls() != null;
		if (!this.getFunctionCalls().isEmpty()) {
			ArrayList<Pair<Expression, Expression>> replacements = new ArrayList<Pair<Expression, Expression>>();
			for (Expression e : this.getBody()) {
				replacements.addAll(findFunctionCalls(e));
			}
			if (!replacements.isEmpty()) {
				this.replaceExpressions(replacements);
				// continue to replace newly introduced function calls
				translateFunctionCalls();
			}
		}
	}

	/**
	 * This method translates all switch case expressions to if-else
	 * equivalents, since UPPAAL does not natively support switch statements.
	 */
	private void translateSwitchExpressions() {
		List<Pair<Expression, Expression>> replacements = new LinkedList<Pair<Expression, Expression>>();
		for (Expression exp : this.body) {
			// find switch expressions
			if (exp instanceof SwitchExpression) {
				SwitchExpression swe = (SwitchExpression) exp;
				Expression switchExpression = swe.getSwitchExpression();
				IfElseExpression parent = null;
				IfElseExpression previous = null;
				List<List<Expression>> unbrokenBodies = new LinkedList<List<Expression>>();
				// translate cases
				for (Expression ce : swe.getCases()) {
					// build condition
					if (ce instanceof CaseExpression) {
						CaseExpression cas = (CaseExpression) ce;
						Expression condition;
						if (cas.isDefaultCase()) {
							condition = new ConstantExpression(null, "true");
						} else {
							condition = new BinaryExpression(null,
									switchExpression, "==", cas.getCondition());
						}
						// look for break statements
						boolean hasBreak = false;
						List<Expression> caseBody = cas.getBody();
						if (!caseBody.isEmpty()
								&& caseBody.get(caseBody.size() - 1) instanceof BreakExpression) {
							// remove it
							caseBody.remove(caseBody.size() - 1);
							hasBreak = true;
						}
						// append body to all previous unbroken cases
						for (List<Expression> unbrokenBody : unbrokenBodies) {
							unbrokenBody.addAll(caseBody);
						}
						// set unbroken cases
						if (!hasBreak) {
							unbrokenBodies.add(caseBody);
						} else {
							unbrokenBodies.clear();
						}
						// build if-else expression
						IfElseExpression ieExpr = new IfElseExpression(null,
								condition, caseBody,
								new LinkedList<Expression>());
						// stitch together
						if (previous != null) {
							previous.addElseExpression(ieExpr);
						} else {
							parent = ieExpr;
						}
						previous = ieExpr;
					} else {
						logger.error(
								"Switch expression contains a case which is not a case expression: {}",
								ce.toString());
					}
				}
				if (parent == null) {
					logger.warn("Switch expression with no cases found: {}",
							swe);
				}
				replacements.add(new Pair<Expression, Expression>(swe, parent));
			}
		}
		// replace
		this.replaceExpressions(replacements);
	}

	/**
	 * Translates the body, replacing Expressions not natively supported by
	 * UPPAAL.
	 * 
	 * @param scf
	 * @return
	 */
	private void translateBody() {
		translateFunctionCalls();
		translateSwitchExpressions();
		// XXX any more expressions to translate?
	}

	/**
	 * Replaces all expressions in the method body equal to a first element of a
	 * pair of the list with the second element of the pair.
	 * 
	 * @param replacements
	 */
	public void replaceExpressions(
			List<Pair<Expression, Expression>> replacements) {
		if (replacements == null)
			throw new NullPointerException();

		for (int i = 0; i < body.size(); i++) {
			boolean replaced = false;
			for (Pair<Expression, Expression> pair : replacements) {
				// yes, we really mean ==
				if (body.get(i) == pair.getFirst()) {
					body.set(i, pair.getSecond());
					replaced = true;
					break;
				}
			}
			if (!replaced) {
				body.get(i).replaceInnerExpressions(replacements);
			}
		}

	}

	public String createDeclarationString() {
		StringBuffer output = new StringBuffer();
		output.append(returnType + " ");
		output.append(name + "(");
		// create parameter list
		LinkedList<String> paramVarNames = new LinkedList<String>();
		for (Iterator<SCParameter> param = parameters.iterator(); param
				.hasNext();) {
			SCParameter par = param.next();
			// TODO currently, we only support this transformation if all
			// variables use the memory model
			// therefore if a TAFunction has an original function, all its
			// parameters have to be converted to address pointers.
			if (this.originalFunction != null) {
				output.append(TAAddressPointer.typeAsAddrPtr(par.getVar()
						.getType())
						+ " "
						+ (par.getRefType().equals(SCREFERENCETYPE.BYREFERENCE) ? "&"
								: "") + par.getVar().getName());
			} else {
				output.append(par.toString());
			}
			paramVarNames.add(par.getVar().getName());
			if (param.hasNext()) {
				output.append(", ");
			}
		}
		if (parameters.size() > 0 && localVariablesAsParameters.size() > 0) {
			output.append(", ");
		}
		for (Iterator<TAVariable> param = localVariablesAsParameters.iterator(); param
				.hasNext();) {
			TAVariable par = param.next();
			output.append(par.getType() + " &" + par.toString());
			paramVarNames.add(par.getName());
			if (param.hasNext())
				output.append(", ");
		}
//		//TODO mp: again, this only applies for methods created by the OMT Optimizer
//		if (originalFunction != null) {
//			ArrayList<SCVariable> additionalParameters = new ArrayList<SCVariable>();
//			for (Expression exp : this.originalFunction.getAllExpressions()) {
//				if (exp instanceof SCVariableExpression) {
//					SCVariableExpression scvar = (SCVariableExpression) exp;
//					if (!localVariables.contains(scvar) && !localVariablesAsParameters.contains(scvar) && originalFunction.getParameter(scvar.getVar().getName()) != null) {
//						//if we get here we found a 
//					}
//				}
//			}
//		}


		output.append("){" + Constants.END_LINE);
		// initialize local variables
		for (TAVariable tav : localVariables) {
			// prevent duplicate declaration
			if (!paramVarNames.contains(tav.getName())) {
				output.append(tav.createDeclarationString());
			}
		}

		// // get all the function calls back
		List<Pair<Expression, Expression>> replacements = new LinkedList<Pair<Expression, Expression>>();
		ExpressionBlock block = new ExpressionBlock(null, body);
		// if (this.functionCalls != null) {
		// for (Expression call : this.functionCalls) {
		// replacements.add(new Pair<Expression, Expression>(call.getParent(),
		// call));
		// }
		// block.replaceInnerExpressions(replacements);
		// replacements.clear();
		// }

		// convert and output function body
		List<Expression> cleanLocalStackExpr = new LinkedList<Expression>();
		List<Expression> innerExprs = block.getInnerExpressions();
		for (Expression innerExpr : innerExprs) {
			Expression newExpr = ExpressionConverter.convert(innerExpr,
					this.originalFunction);
			if (newExpr == null
					&& innerExpr instanceof SCVariableDeclarationExpression) {
				if (((SCVariableDeclarationExpression) innerExpr)
						.getFirstInitialValue() == null) {
					// ExpresionConverter returns null in this case, but here,
					// we actually want to replace!
					newExpr = ((SCVariableDeclarationExpression) innerExpr)
							.getVariable();
				}
			} else if (newExpr == null && innerExpr instanceof OutputExpression) {
				newExpr = new EmptyExpression(innerExpr.getNode());
			}
			if (newExpr != null) {
				replacements.add(new Pair<Expression, Expression>(innerExpr,
						newExpr));
			}

		}
		block.replaceInnerExpressions(replacements);
		block.addAll(cleanLocalStackExpr);
		for (Expression expr : block.getBlock()) {
			output.append(expr.toString() + Constants.END_LINE);
		}
		output.append("}");
		output.append(Constants.END_LINE);
		// reset original expressions
		for (Pair<Expression, Expression> pair : replacements) {
			pair.swap();
		}
		block.replaceInnerExpressions(replacements);

		return output.toString();
	}

	/**
	 * Prints the function. This method is used when creating the declaration of
	 * the TAFunction.
	 * 
	 * @param writer
	 * @throws IOException
	 */
	public void print(XMLWriter writer) throws IOException {
		if (writer == null)
			throw new NullPointerException();

		writer.writeText(createDeclarationString());
	}

	@Override
	public String toString() {
		return createDeclarationString();
	}
}
