/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.io.IOException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.util.XMLWriter;
import de.tub.pes.syscir.sc_model.expressions.ArrayInitializerExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableExpression;

/**
 * Represents an array in UPPAAL.
 * Only one dimensional arrays are allowed. Examples:
 * int a[1];
 * [const] int a[3] = {1,2,3};
 * @author rschroeder 
 */

public class TAArray extends TAVariable {

	private static Logger logger = LogManager
			.getLogger(TAArray.class.getName());

	/**
	 * Expression array initializer or size Expression
	 */
	private Expression sizeOrInitializer = null;

	/**
	 * The only constructor. Array name, type, dimension and constant flag are
	 * mandatory. Sizes of each dimension initially equal one.
	 * 
	 * @param name
	 * @param type
	 * @param dim
	 * @param constant
	 */
	public TAArray(String name, String type, boolean constant, Expression sizeOrInitializer) {
		super(name, type);
		this.isConstant = constant;
		this.sizeOrInitializer = sizeOrInitializer;
		if (constant && !(sizeOrInitializer instanceof ArrayInitializerExpression)) {
			logger.error("Constants must have an initializer");
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.tub.pes.sc2uppaal.model.TAVariable#createCopy(java.lang.String)
	 */
	public TAVariable createCopy(String prefix) {
		return new TAArray(prefix + name, type,	isConstant, sizeOrInitializer);
	}

	@Override
	public String createDeclarationString() {
		boolean withAmpersand = false;
		boolean withInitialization = true;
		String output = getAll(withAmpersand, withInitialization);
		output += ";" + Constants.END_LINE;
		return output;
	}

	@Override
	public void printAsParam(XMLWriter writer) throws IOException {
		boolean withAmpersand = true;
		boolean withInitialization = false;
		String output = getAll(withAmpersand, withInitialization);
		writer.writeText(output);
	}

	private String getAll(boolean withAmpersand, boolean withInitialization) {
		String output = (isConstant ? "const " : "") + type;
		output += " " + (withAmpersand ? "&" : "") + name;
		ArrayInitializerExpression aie = null;
		String size = null;
		if (sizeOrInitializer instanceof ArrayInitializerExpression) {
			aie = (ArrayInitializerExpression) sizeOrInitializer;
			if (aie.getInnerExpressions() != null) {
				size = Integer.toString(aie.getInnerExpressions().size());
			}
		} else if (sizeOrInitializer instanceof ConstantExpression) {
			size = ((ConstantExpression) sizeOrInitializer).retValue();
		} else if (sizeOrInitializer instanceof SCVariableExpression) {
			size = ((SCVariableExpression) sizeOrInitializer).getVar().getName();
		} else {
			logger.error("Can't handle this expression");
			return null;
		}
		if (size != null) {
			output += "[" + size + "]";
		}
		if (withInitialization && aie != null) {
			output += " = " + aie.toStringNoSem();
		}
		return output;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.tub.pes.state.model.TAVariable#getDummyName(de.tub.pes.state.model
	 * .TAModel)
	 */
	public String getDummyName(TAModel ta) {
		logger.error("unimplemented method!");
		return null;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime
				* result
				+ ((sizeOrInitializer == null) ? 0 : sizeOrInitializer
						.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (getClass() != obj.getClass())
			return false;
		TAArray other = (TAArray) obj;
		if (sizeOrInitializer == null) {
			if (other.sizeOrInitializer != null)
				return false;
		} else if (!sizeOrInitializer.equals(other.sizeOrInitializer))
			return false;
		return true;
	}

}
