/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import de.tub.pes.sc2uppaal.tamodel.expressions.TASendExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TAVariableAssignmentExpression;
import de.tub.pes.syscir.sc_model.SCClass;
import de.tub.pes.syscir.sc_model.SCFunction;
import de.tub.pes.syscir.sc_model.SCParameter;
import de.tub.pes.syscir.sc_model.SCREFERENCETYPE;
import de.tub.pes.syscir.sc_model.SCSystem;
import de.tub.pes.syscir.sc_model.SCVariable;
import de.tub.pes.syscir.sc_model.expressions.AccessExpression;
import de.tub.pes.syscir.sc_model.expressions.BinaryExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.FunctionCallExpression;
import de.tub.pes.syscir.sc_model.expressions.OutputExpression;
import de.tub.pes.syscir.sc_model.expressions.ReturnExpression;
import de.tub.pes.syscir.sc_model.expressions.SCClassInstanceExpression;
import de.tub.pes.syscir.sc_model.expressions.SCPortSCSocketExpression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableDeclarationExpression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableExpression;
import de.tub.pes.syscir.sc_model.expressions.TimeUnitExpression;
import de.tub.pes.syscir.sc_model.variables.SCArray;
import de.tub.pes.syscir.sc_model.variables.SCClassInstance;
import de.tub.pes.syscir.sc_model.variables.SCPointer;
import de.tub.pes.syscir.sc_model.variables.SCSimpleType;

/**
 * This class encapsualtes the functionality to allocate memory for all
 * scmodules and execute their constructor. The SCMAIN template shall be
 * executed before the scheduler starts.
 * 
 * @author rschroeder, Timm Liebrenz
 * 
 */
public class SCMainConverter {
	private final List<TAFunction> ctorList;

	private final TAModel model;
	private final SCSystem scs;

	public SCMainConverter(TAModel model, SCSystem scs) {
		// use linked hash map to remember insertion order
		ctorList = new LinkedList<TAFunction>();

		this.model = model;
		this.scs = scs;
	}

	/**
	 * Create the scmain template. Alloc all members, run constructors.
	 * 
	 * @param model
	 */
	public void createSCMainTemplate() {
		model.addVariable(new TAChannel(Constants.SCMAIN_FINISHED_CHANNEL,
				false));
		TATemplate template = new TATemplate(Constants.SCMAIN_TEMPLATE_NAME);
		model.addTemplate(template);

		TALocation start = template.createInitLocation();
		start.setUrgent(true);

		model.addTemplateInstance(template
				.getInstance(Constants.SCMAIN_TEMPLATE_INSTANCE_NAME));

		TALocation currentLoc = start;
		SCFunction scmain = scs.getGlobalFunction("sc_main");

		// first, alloc all members for all modules
		// generate per-instance-ctor on the fly
		for (Expression expr : scmain.getBody()) {
			if (expr instanceof SCVariableDeclarationExpression) {
				SCVariableDeclarationExpression scvarDeclExpr = (SCVariableDeclarationExpression) expr;
				if (scvarDeclExpr.getVariable() instanceof SCClassInstanceExpression) {
					SCClassInstance scci = ((SCClassInstanceExpression) scvarDeclExpr
							.getVariable()).getInstance();
					currentLoc = allocMembers(scci, template, currentLoc,
							scci.getName());
				} else if (scvarDeclExpr.getVariable() instanceof SCVariableExpression) {
					// TODO: alloc local var for scmain()
				}
			}
		}
		// don't forget to add all ctors to the model
		for (TAFunction ctor : ctorList) {
			model.addFunction(ctor);
		}

		// now create and run scmain
		TAFunction scmainNative = new TAFunction.Builder(
				Constants.SCMAIN_NATIVE_METHOD_NAME, "void")
				.body(convertBody(scmain.getBody(), ""))
				.localVariablesFromSCFunction(
						cleanLocVars(scmain.getLocalVariables())).build();
		model.addFunction(scmainNative);
		TALocation callSCMain = template.createUrgentLocation();
		TATransition scmainTrans = new TATransition(currentLoc, callSCMain);
		scmainTrans.setGuard(new ConstantExpression(null,
				Constants.MEM_ERR_NOT_ENOUGH_MEM + " == 0"));
		scmainTrans.addUpdateExpression(new ConstantExpression(null,
				Constants.SCMAIN_NATIVE_METHOD_NAME + "()"));
		template.addTransition(scmainTrans);

		// end location mustn't be urgent! otherwise the schedule deadlocks
		TALocation end = template.createLocation();
		end.setName(Constants.SCMAIN_TEMPLATE_END_LOCATION);
		template.addTransition(new TATransition(callSCMain, end,
				new TASendExpression(null, Constants.SCMAIN_FINISHED_CHANNEL)));
	}

	private TALocation allocMembers(SCClassInstance scci,
			TATemplate scmainTmpl, TALocation currentLoc, String prefix) {
		SCClass scc = scci.getSCClass();
		for (Iterator<SCVariable> iter = scc.getMembers().iterator(); iter
				.hasNext();) {
			SCVariable member = iter.next();
			if (member instanceof SCClassInstance) {
				SCClassInstance inner = (SCClassInstance) member;
				if (inner.isSCModule() || inner.isChannel()) {
					currentLoc = allocMembers(inner, scmainTmpl, currentLoc,
							TAVariable.addPrefix(inner.getName(), prefix));
				}
			} else if (member.isArrayOfSCClassInstances()) {
			}
			// no "else if" here because we need to alloc structs (which are
			// scci), too
			if (VariableConverter.useMemModel(member)) {
				Expression right = null;
				if (member instanceof SCArray) {
					// member arr size is either an int or another const member:
					Expression size = ((SCArray) member).getSize();
					String sizeAsString = size.toStringNoSem();
					if (size instanceof SCVariableExpression) {
						sizeAsString = TAVariable.addPrefix(sizeAsString,
								prefix);
					}
					String funcName = TAMemoryType.getFuncName(
							TAMemoryType.getAllocStaticArrFuncName(),
							member.getTypeWithoutSize());
					right = ExpressionConverter.getFCE(funcName, sizeAsString);
				} else {
					String value = "";
					if (member.hasInitialValue()) {
						// only take first value, might crash in case of more
						// complicated stuff
						Expression initValue = member.getDeclaration()
								.getFirstInitialValue();
						value = initValue.toStringNoSem();
					}
					String funcName = TAMemoryType.getFuncName(
							TAMemoryType.getAllocStaticFuncName(value),
							member.getTypeWithoutSize());
					right = ExpressionConverter.getFCE(funcName, value);

				}
				TAVariable tavar = VariableConverter.convertVariable(member);
				tavar.addPrefix(prefix);

				TALocation nextLoc = scmainTmpl.createUrgentLocation();
				TATransition trans = new TATransition(currentLoc, nextLoc);
				scmainTmpl.addTransition(trans);
				Expression update = new TAVariableAssignmentExpression(null,
						tavar, right);
				trans.addUpdateExpression(update);
				currentLoc = nextLoc;
			}
		}
		// handle class instances in the constructor as member variables
		for (SCVariable member : scc.getConstructor().getLocalVariables()) {
			if (member instanceof SCClassInstance) {
				SCClassInstance inner = (SCClassInstance) member;
				if (inner.isSCModule() || inner.isChannel()) {
					currentLoc = allocMembers(inner, scmainTmpl, currentLoc,
							TAVariable.addPrefix(inner.getName(), prefix));
				}
			}
		}

		// oh, btw ... dont forget to create the ctor :D
		TAFunction ctor = getNativeContstructor(scci, prefix);
		ctorList.add(ctor);
		return currentLoc;
	}

	private String getCtorName(String prefix) {
		return prefix + Constants.PREFIX_DELIMITER + "ctor";
	}

	private TAFunction getNativeContstructor(SCClassInstance scci, String prefix) {
		SCClass scc = scci.getSCClass();
		String ctorName = getCtorName(prefix);
		TAFunction ctor = new TAFunction.Builder(ctorName, "void")
				.parameters(getNativeParameters(scc))
				.body(getNativeBody(scci, prefix))
				.localVariablesFromSCFunction(
						scc.getConstructor().getLocalVariables()).build();
		return ctor;
	}

	private List<SCParameter> getNativeParameters(SCClass scc) {
		List<SCParameter> finalParams = new ArrayList<SCParameter>();
		SCFunction scCtor = scc.getConstructor();
		// first, add real ctor params (except name)
		for (int i = 1; i < scCtor.getParameters().size(); i++) {
			SCParameter scParam = scCtor.getParameters().get(i);
			/**
			 * Set params callbyvalue: This allows for constants to be used as
			 * arguements.
			 * */
			SCParameter newParam = new SCParameter(scParam.getVar(),
					SCREFERENCETYPE.BYVALUE);
			/**
			 * Set params const: If they are not const, the get may get coverted
			 * to access the memory, but we dont want this here. Note: this
			 * causes trouble if the ctor is called with a var that is actually
			 * in the mem.
			 */
			SCVariable tmp = newParam.getVar();
			tmp.setConst(true);
			tmp.setType(tmp.getTypeWithoutSize());
			
			finalParams.add(newParam);
		}
		// snd, add member varis as params
		for (Iterator<SCVariable> iter = scc.getMembers().iterator(); iter
				.hasNext();) {
			SCVariable member = iter.next();
			String type = null;
			SCREFERENCETYPE refType = null;
			if (member.isConst()) {
				// const members
				type = member.getTypeWithoutSize();
				refType = SCREFERENCETYPE.BYVALUE;
			} else if (VariableConverter.useMemModel(member)) {
				// member is in mem model (eg. referenced at least once and no
				// scmodule)
				type = TAAddressPointer.typeAsAddrPtr(member
						.getTypeWithoutSize());
				refType = SCREFERENCETYPE.BYREFERENCE;
			} else if (member.isNotSCModule()
					&& member.isNotChannel()
					&& !TAMemoryType.isForbiddenType(member
							.getTypeWithoutSize())) {
				// member is not in memmodel ...
				if (member instanceof SCPointer) {
					// member is ptr
					type = TAAddressPointer.typeAsAddrPtr(member
							.getTypeWithoutSize());
				} else {
					// member is var which is never referenced
					type = member.getTypeWithoutSize();
				}
				refType = SCREFERENCETYPE.BYREFERENCE;
			}
			if (type != null && refType != null) {
				SCVariable dummy = new SCSimpleType(member.getName(), type);
				SCParameter par = new SCParameter(dummy, refType);
				finalParams.add(par);
			}
		}
		return finalParams;
	}

	private List<Expression> getNativeBody(SCClassInstance scci, String prefix) {
		SCClass scc = scci.getSCClass();
		SCFunction scCtor = scc.getConstructor();
		List<Expression> newBody = convertBody(scCtor.getBody(), prefix);
		return newBody;
	}

	/**
	 * Convert body of sc_main() or ctors. When scmodules are declared, we call
	 * the appropriate ctor. Some expressions have to be omitted (signal
	 * declaration, return expression, ...).
	 * 
	 * @param body
	 * @param prefix
	 * @return
	 */
	private List<Expression> convertBody(List<Expression> body, String prefix) {
		List<Expression> newBody = new LinkedList<Expression>();
		if (body == null || body.size() == 0) {
			return newBody;
		}
		for (Expression expr : body) {
			if (expr instanceof FunctionCallExpression) {
				if (((FunctionCallExpression) expr).getFunction().getName()
						.equals("sc_module")
						|| (((FunctionCallExpression) expr).getFunction()
								.getName().equals("sc_start"))) {
					continue;
				}
			} else if (expr instanceof ReturnExpression
					|| expr instanceof OutputExpression) {
				continue; // in scmain
			} else if (expr instanceof AccessExpression) {
				AccessExpression ae = (AccessExpression) expr;
				if (ae.getRight() instanceof AccessExpression) {
					if (((AccessExpression) ae.getRight()).getLeft() instanceof SCPortSCSocketExpression) {
						continue; // "myModuleInst.modulePort(mySignalInst);" in
									// scmain
					}
				}
			} else if (isSCCIDeclaration(expr)) {
				newBody.add(scciDeclaration2CtorCall(expr, prefix));
			} else {
				newBody.add(expr);
			}
		}
		return newBody;
	}

	private Expression scciDeclaration2CtorCall(Expression expr, String prefix) {
		SCClassInstance scci = getScciFromDecl(expr);
		SCClass scc = scci.getSCClass();
		if (!prefix.equals("")) {
			prefix += Constants.PREFIX_DELIMITER;
		}
		prefix += scci.getName(); // we access stuff *inside* the scci we just
									// got
		FunctionCallExpression fce = getInnerCtorCall(expr);
		List<Expression> nativeArgs = new LinkedList<Expression>();
		// the first n params of the native ctor are real params, the rest are
		// local members.
		// skip module name as the first arg of the fce
		for (int i = 1; i < fce.getParameters().size(); i++) {
			Expression realArg = ExpressionConverter.convert(fce
					.getParameters().get(i), null);
			if (realArg instanceof TimeUnitExpression) {
				continue; // dont care about time units
			}
			nativeArgs.add(realArg);
		}
		// now, add the members of the declared module
		// either member
		// -is in mem model
		// -is constant
		// -is notscmodul and not forbidden type
		for (SCVariable member : scc.getMembers()) {
			if (VariableConverter.useMemModel(member)
					|| member.isConst()
					|| (member.isNotSCModule() && member.isNotChannel() && !TAMemoryType
							.isForbiddenType(member.getTypeWithoutSize()))) {
				String tmpName = prefix + Constants.PREFIX_DELIMITER
						+ member.getName();
				SCVariable dummy = new SCSimpleType(tmpName,
						member.getTypeWithoutSize());
				Expression memberAsArg = new SCVariableExpression(null, dummy);
				nativeArgs.add(memberAsArg);
			}
		}

		// put everything together
		String ctorName = getCtorName(prefix);
		StringBuilder ctorCall = new StringBuilder();
		ctorCall.append(ctorName);
		ctorCall.append("(");
		for (Iterator<Expression> iter = nativeArgs.iterator(); iter.hasNext();) {
			Expression arg = iter.next();
			ctorCall.append(arg.toStringNoSem());
			if (iter.hasNext()) {
				ctorCall.append(", ");
			}
		}
		ctorCall.append(");");
		ConstantExpression ret = new ConstantExpression(null,
				ctorCall.toString());
		return ret;
	}

	private SCClassInstance getScciFromDecl(Expression expr) {
		SCClassInstance scci = null;
		SCVariableExpression vd = null;
		if (expr instanceof SCVariableDeclarationExpression) {
			SCVariableDeclarationExpression vde = (SCVariableDeclarationExpression) expr;
			if (vde.getVariable() instanceof SCVariableExpression) {
				vd = (SCVariableExpression) vde.getVariable();
				scci = (SCClassInstance) vd.getVar();
			} else {
				scci = ((SCClassInstanceExpression) vde.getVariable())
						.getInstance();
			}
		} else if (expr instanceof BinaryExpression) {
			BinaryExpression be = (BinaryExpression) expr;
			vd = (SCVariableExpression) be.getLeft();
			scci = (SCClassInstance) vd.getVar();
		}
		return scci;
	}

	private FunctionCallExpression getInnerCtorCall(Expression expr) {
		// 'instanceOfInner = inner("my_inner_inst");'
		if (expr instanceof BinaryExpression) {
			BinaryExpression binExpr = (BinaryExpression) expr;
			if (binExpr.getLeft() instanceof SCVariableExpression
					&& binExpr.getRight() instanceof FunctionCallExpression) {
				SCVariableExpression scvarExpr = (SCVariableExpression) binExpr
						.getLeft();
				if (scvarExpr.getVar() instanceof SCClassInstance) {
					return (FunctionCallExpression) binExpr.getRight();
				}
			}
			// 'MyMdoule mm("my_inst");'
		} else if (expr instanceof SCVariableDeclarationExpression) {
			SCVariableDeclarationExpression vde = (SCVariableDeclarationExpression) expr;
			List<Expression> args = new ArrayList<Expression>(
					vde.getInitialValues());
			FunctionCallExpression fce = new FunctionCallExpression(null, null,
					args);
			return fce;
		}
		return null;
	}

	private boolean isSCCIDeclaration(Expression expr) {
		// 'instanceOfInner = inner("my_inner_inst");'
		if (expr instanceof BinaryExpression) {
			BinaryExpression binExpr = (BinaryExpression) expr;
			if (binExpr.getLeft() instanceof SCVariableExpression) {
				SCVariableExpression scvarExpr = (SCVariableExpression) binExpr
						.getLeft();
				if (scvarExpr.getVar() instanceof SCClassInstance) {
					return true;
				}
			}
			// 'MyMdoule mm("my_inst");'
		} else if (expr instanceof SCVariableDeclarationExpression) {
			SCVariableDeclarationExpression vde = (SCVariableDeclarationExpression) expr;
			if (vde.getVariable() instanceof SCClassInstanceExpression) {
				return true;
			}
		}
		return false;
	}

	private List<SCVariable> cleanLocVars(List<SCVariable> localVariables) {
		List<SCVariable> clean = new ArrayList<SCVariable>(
				localVariables.size());
		for (SCVariable scvar : localVariables) {
			if (!scvar.isSCClassInstance()
					&& !TAMemoryType
							.isForbiddenType(scvar.getTypeWithoutSize())) {
				clean.add(scvar);
			}
		}
		return clean;
	}
}
