/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.util.LinkedList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.expressions.TAAccessExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TAVariableAssignmentExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TAVariableExpression;
import de.tub.pes.syscir.sc_model.SCFunction;
import de.tub.pes.syscir.sc_model.SCParameter;
import de.tub.pes.syscir.sc_model.SCVariable;
import de.tub.pes.syscir.sc_model.expressions.AccessExpression;
import de.tub.pes.syscir.sc_model.expressions.ArrayAccessExpression;
import de.tub.pes.syscir.sc_model.expressions.ArrayInitializerExpression;
import de.tub.pes.syscir.sc_model.expressions.AssertionExpression;
import de.tub.pes.syscir.sc_model.expressions.BinaryExpression;
import de.tub.pes.syscir.sc_model.expressions.BracketExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.DeleteArrayExpression;
import de.tub.pes.syscir.sc_model.expressions.DeleteExpression;
import de.tub.pes.syscir.sc_model.expressions.DoWhileLoopExpression;
import de.tub.pes.syscir.sc_model.expressions.EnumElementExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.ExpressionBlock;
import de.tub.pes.syscir.sc_model.expressions.ForLoopExpression;
import de.tub.pes.syscir.sc_model.expressions.FunctionCallExpression;
import de.tub.pes.syscir.sc_model.expressions.NewArrayExpression;
import de.tub.pes.syscir.sc_model.expressions.NewExpression;
import de.tub.pes.syscir.sc_model.expressions.QuestionmarkExpression;
import de.tub.pes.syscir.sc_model.expressions.RefDerefExpression;
import de.tub.pes.syscir.sc_model.expressions.ReturnExpression;
import de.tub.pes.syscir.sc_model.expressions.SCClassInstanceExpression;
import de.tub.pes.syscir.sc_model.expressions.SCDeltaCountExpression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableDeclarationExpression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableExpression;
import de.tub.pes.syscir.sc_model.expressions.TimeUnitExpression;
import de.tub.pes.syscir.sc_model.expressions.UnaryExpression;
import de.tub.pes.syscir.sc_model.expressions.WhileLoopExpression;
import de.tub.pes.syscir.sc_model.variables.SCArray;
import de.tub.pes.syscir.sc_model.variables.SCClassInstance;
import de.tub.pes.syscir.sc_model.variables.SCPointer;

/**
 * Ptr = pointer, Arr = array, Var = everything else
 * 
 * @author rschroeder, Timm Liebrenz
 * 
 */
public class ExpressionConverter {

	private static Logger logger = LogManager
			.getLogger(ExpressionConverter.class.getName());

	// call-by-value parameters represented in memory need allocation call.
	public static Expression getAllocExprForParAsLocVar(SCParameter par,
			SCFunction func) {
		if (VariableConverter.useMemModel(par.getVar())) {
			ConstantExpression left = new ConstantExpression(null,
					VariableConverter.getAddrPtrName(par.getVar()));
			String value = VariableConverter.getTemplParamName(func.getName(),
					par.getVar().getName());
			String funcName = TAMemoryType.getFuncName(TAMemoryType
					.getAllocStaticFuncName(value), par.getVar()
					.getTypeWithoutSize());
			FunctionCallExpression right = getFCE(funcName, value);

			Expression ret = new BinaryExpression(null, left, "=", right);
			return ret;
		}
		return null;
	}

	public static FunctionCallExpression getFCE(String funcName,
			List<Expression> params) {
		SCFunction dummyFunc = new SCFunction(funcName);
		FunctionCallExpression fce = new FunctionCallExpression(null,
				dummyFunc, params);
		return fce;
	}

	public static FunctionCallExpression getFCE(String funcName) {
		return getFCE(funcName, new LinkedList<Expression>());
	}

	public static FunctionCallExpression getFCE(String funcName,
			String paramsAsString) {
		List<Expression> params = new LinkedList<Expression>();
		params.add(new ConstantExpression(null, paramsAsString));
		return getFCE(funcName, params);
	}

	public static FunctionCallExpression getFCE(String funcName,
			Expression singleParam) {
		List<Expression> params = new LinkedList<Expression>();
		params.add(singleParam);
		return getFCE(funcName, params);
	}

	/*
	 * Convert a given expression to access our mem model correctly. Note:
	 * Unhandled expression return null. If you get a nullptr exception and your
	 * stack trace contains this class, make sure, the expression is handled.
	 * 
	 * ^This should be changed to throwing a reasonable exception.
	 */
	public static Expression convert(Expression expr, SCFunction func) {
		if (expr instanceof ConstantExpression
				|| expr instanceof EnumElementExpression
				|| expr instanceof FunctionCallExpression
				|| expr instanceof TAAccessExpression
				|| expr instanceof TimeUnitExpression) {
			return expr;
		} else if (expr instanceof AssertionExpression) {
			return convertAssertionExpr((AssertionExpression) expr, func);
		} else if (expr instanceof SCVariableExpression) {
			SCVariableExpression scvarExpr = (SCVariableExpression) expr;
			SCVariable scvar = scvarExpr.getVar();
			if (expr instanceof ArrayAccessExpression) {
				return convertArrAccess((ArrayAccessExpression) expr, func);
			} else if (scvar instanceof SCPointer) {
				return convertPtrExpression(scvarExpr, func);
			} else if (scvar instanceof SCArray) {
				return convertArrExpression(scvarExpr, func);
			} else {
				return convertVarExpression(scvarExpr, func);
			}
		} else if (expr instanceof SCVariableDeclarationExpression) {
			SCVariableDeclarationExpression scvarDeclExpr = (SCVariableDeclarationExpression) expr;
			Expression varExpr = scvarDeclExpr.getVariable();
			SCVariable scvar = null;
			if (varExpr instanceof SCVariableExpression) {
				scvar = ((SCVariableExpression) scvarDeclExpr.getVariable())
						.getVar();
			} else { // class instance
				scvar = ((SCClassInstanceExpression) scvarDeclExpr
						.getVariable()).getInstance();
			}
			if (scvarDeclExpr.getFirstInitialValue() == null) {
				if (scvar instanceof SCPointer) {
					return convertPtrDeclarationNoInit(scvarDeclExpr, func);
				} else if (scvar instanceof SCArray) {
					return convertArrDeclarationNoInit(scvarDeclExpr, func);
				} else if (scvar instanceof SCClassInstance) {
					return convertSCCIDeclarationNoInit(scvarDeclExpr, func);
				} else {
					return convertVarDeclarationNoInit(scvarDeclExpr, func);
				}
			} else { // with initialization
				if (scvar instanceof SCPointer) {
					return convertPtrDeclaration(scvarDeclExpr, func);
				} else if (scvar instanceof SCArray) {
					return convertArrDeclaration(scvarDeclExpr, func);
				} else {
					return convertVarDeclaration(scvarDeclExpr, func);
				}
			}
		} else if (expr instanceof BinaryExpression) {
			BinaryExpression binExpr = (BinaryExpression) expr;
			// if new(), else if ??
			return convertBinaryExpr(binExpr, func);
		} else if (expr instanceof RefDerefExpression) {
			RefDerefExpression rde = (RefDerefExpression) expr;
			return convertRefDeref(rde, func);
		} else if (expr instanceof ReturnExpression) {
			return convertReturn((ReturnExpression) expr, func);
		} else if (expr instanceof BracketExpression) {
			return convertBracket((BracketExpression) expr, func);
		} else if (expr instanceof NewExpression) {
			return convertNew((NewExpression) expr, func);
		} else if (expr instanceof NewArrayExpression) {
			return convertNewArr((NewArrayExpression) expr, func);
		} else if (expr instanceof DeleteExpression) {
			return convertDelete((DeleteExpression) expr, func);
		} else if (expr instanceof UnaryExpression) {
			return convertUnary((UnaryExpression) expr, func);
		} else if (expr instanceof QuestionmarkExpression) {
			return convertQuestionmark((QuestionmarkExpression) expr, func);
		} else if (expr instanceof AccessExpression) {
			return convertAccess((AccessExpression) expr, func);
		} else if (expr instanceof SCDeltaCountExpression) {
			return convertDeltaCount((SCDeltaCountExpression) expr, func);
		} else if (expr instanceof TAVariableExpression) {
			// noting to do here...
			return expr;
		} else if (expr instanceof ForLoopExpression) {
			return convertForLoop((ForLoopExpression) expr, func);
		} else if (expr instanceof DoWhileLoopExpression) {
			return convertDoWhileLoop((DoWhileLoopExpression) expr, func);
		} else if (expr instanceof WhileLoopExpression) {
			return convertWhileLoop((WhileLoopExpression) expr, func);
		}
		logger.error("Can't convert expression {} of type {}", expr,
				expr.getClass());
		return null;
	}

	// Struct s; -> int s = alloc_static_Struct();
	// OR: nothing
	private static Expression convertSCCIDeclarationNoInit(
			SCVariableDeclarationExpression vdexpr, SCFunction func) {
		SCClassInstance scvar = ((SCClassInstanceExpression) vdexpr
				.getVariable()).getInstance();
		SCVariableExpression dummy1 = new SCVariableExpression(
				vdexpr.getNode(), scvar);
		SCVariableDeclarationExpression dummy2 = new SCVariableDeclarationExpression(
				vdexpr.getNode(), dummy1);
		return convertVarDeclarationNoInit(dummy2, func);
	}

	// T v; -> int v = alloc_static_T();
	// OR: nothing
	private static Expression convertVarDeclarationNoInit(
			SCVariableDeclarationExpression vdexpr, SCFunction func) {
		SCVariable scvar = ((SCVariableExpression) vdexpr.getVariable())
				.getVar();
		if (VariableConverter.useMemModel(scvar)) {
			TAVariable dummy = VariableConverter.getAddrPtr(scvar);
			String funcName = TAMemoryType.getFuncName(
					TAMemoryType.getAllocStaticFuncName(),
					scvar.getTypeWithoutSize());
			FunctionCallExpression right = getFCE(funcName);
			Expression ret = new TAVariableAssignmentExpression(
					vdexpr.getNode(), dummy, right);
			return ret;
		}
		return null;
	}

	// T* p; -> int p = NULL;
	// OR: nothing
	private static Expression convertPtrDeclarationNoInit(
			SCVariableDeclarationExpression vdexpr, SCFunction func) {
		SCVariable scvar = ((SCVariableExpression) vdexpr.getVariable())
				.getVar();
		TAVariable dummy = VariableConverter.getAddrPtr(scvar);
		Expression right = new ConstantExpression(vdexpr.getNode(),
				Constants.NULL);
		Expression ret = new TAVariableAssignmentExpression(vdexpr.getNode(),
				dummy, right);
		return ret;
	}

	// T a[E]; -> int a = alloc_static_arr_T(E);
	// OR: nothing
	private static Expression convertArrDeclarationNoInit(
			SCVariableDeclarationExpression vdexpr, SCFunction func) {
		SCArray scvar = (SCArray) ((SCVariableExpression) vdexpr.getVariable())
				.getVar();
		TAVariable dummy = VariableConverter.getAddrPtr(scvar);
		String funcName = TAMemoryType.getFuncName(
				TAMemoryType.getAllocStaticArrFuncName(),
				scvar.getTypeWithoutSize());
		FunctionCallExpression right = getFCE(funcName,
				convert(scvar.getDerivedSize(), func).toStringNoSem());
		Expression ret = new TAVariableAssignmentExpression(vdexpr.getNode(),
				dummy, right);
		return ret;
	}

	// v; -> TMem[v];
	// OR: v
	private static Expression convertVarExpression(
			SCVariableExpression scvarExpr, SCFunction func) {
		SCVariable scvar = scvarExpr.getVar();
		if (VariableConverter.useMemModel(scvar)) {
			Expression ret = VariableConverter.derefAddrPtr(scvar);
			return ret;
		} else {
			return scvarExpr;
		}
	}

	// p; -> p;
	private static Expression convertPtrExpression(
			SCVariableExpression scvarExpr, SCFunction func) {
		SCVariable scvar = scvarExpr.getVar();
		TAVariable dummy = VariableConverter.getAddrPtr(scvar);
		Expression ret = new TAVariableExpression(scvarExpr.getNode(), dummy);
		return ret;
	}

	// a; -> a;
	private static Expression convertArrExpression(
			SCVariableExpression scvarExpr, SCFunction func) {
		SCVariable scvar = scvarExpr.getVar();
		TAVariable dummy = VariableConverter.getAddrPtr(scvar);
		return new TAVariableExpression(scvarExpr.getNode(), dummy);
	}

	// T v = E; -> int v = alloc_static_T(E);
	// OR: v = E;
	private static Expression convertVarDeclaration(
			SCVariableDeclarationExpression vdexpr, SCFunction func) {
		SCVariable scvar;
		if (vdexpr.getVariable() instanceof SCVariableExpression) {
			scvar = ((SCVariableExpression) vdexpr.getVariable()).getVar();
		} else if (vdexpr.getVariable() instanceof SCClassInstanceExpression) {
			scvar = ((SCClassInstanceExpression) vdexpr.getVariable())
					.getInstance();
		} else {
			logger.error("unkown SCVariableDeclarationExpression variable");
			return null;
		}
		Expression initValue = convert(vdexpr.getFirstInitialValue(), func);
		assert (initValue != null);
		Expression right = initValue;
		TAVariable dummy = new TAGenericTypeVariable(vdexpr.getVariable()
				.toStringNoSem(), "");
		if (VariableConverter.useMemModel(scvar)) {
			dummy = VariableConverter.getAddrPtr(scvar);
			String funcName = TAMemoryType.getFuncName(
					TAMemoryType.getAllocStaticFuncName(initValue),
					scvar.getTypeWithoutSize());
			right = getFCE(funcName, initValue);
		}
		Expression ret = new TAVariableAssignmentExpression(vdexpr.getNode(),
				dummy, right);
		return ret;
	}

	// T* p = E; -> int p = E;
	private static Expression convertPtrDeclaration(
			SCVariableDeclarationExpression vdexpr, SCFunction func) {
		SCVariable scvar = ((SCVariableExpression) vdexpr.getVariable())
				.getVar();
		TAVariable dummy = VariableConverter.getAddrPtr(scvar);
		Expression right = convert(vdexpr.getFirstInitialValue(), func);
		Expression ret = new TAVariableAssignmentExpression(vdexpr.getNode(),
				dummy, right);
		return ret;
	}

	// T a[] = {v0 ,...,vn−1 }; -> int a = alloc_static_arr_T(n); TMem[a+0] =
	// v0
	// ; ...; TMem[a+(n-1)] = vn−1 ;
	private static Expression convertArrDeclaration(
			SCVariableDeclarationExpression vdexpr, SCFunction func) {
		SCArray scvar = (SCArray) ((SCVariableExpression) vdexpr.getVariable())
				.getVar();
		ExpressionBlock ret = new ExpressionBlock(vdexpr.getNode());
		ret.addExpression(convertArrDeclarationNoInit(vdexpr, func));

		Expression initValue = null;
		Expression call = null;
		ArrayInitializerExpression aie = (ArrayInitializerExpression) vdexpr
				.getFirstInitialValue();
		for (int i = 0; i < aie.getInnerExpressions().size(); i++) {
			initValue = convert(aie.getInnerExpressions().get(i), func);
			String args = VariableConverter.getAddrPtrName(scvar) + " + "
					+ Integer.toString(i) + ", " + initValue.toStringNoSem();
			String funcName = TAMemoryType.getFuncName(
					TAMemoryType.getInitArrayElemFuncName(),
					scvar.getTypeWithoutSize());
			call = getFCE(funcName, args);
			ret.addExpression(call);
		}
		return ret;
	}

	// v = E; -> TMem[v] = E; ...
	private static Expression convertBinaryExpr(BinaryExpression binExpr,
			SCFunction func) {
		Expression left = convert(binExpr.getLeft(), func);
		Expression right = convert(binExpr.getRight(), func);
		Expression ret = new BinaryExpression(binExpr.getNode(), left,
				binExpr.getOp(), right);
		return ret;
	}

	private static Expression convertAssertionExpr(AssertionExpression assExpr,
			SCFunction func) {
		Expression condition = convert(assExpr.getCondition(), func);
		Expression ret = new AssertionExpression(assExpr.getNode(), condition);
		return ret;
	}

	// &v -> v; *p -> TMem[p];
	private static Expression convertRefDeref(RefDerefExpression rde,
			SCFunction func) {
		// TODO: can we do other tricks?
		// we suppose here, that in '&v' v is a var and in '*p' p is ptr/arr
		SCVariableExpression scvarExpr = (SCVariableExpression) rde
				.getExpression();
		SCVariable scvar = scvarExpr.getVar();
		// currently &p is not possible in our model as ptrs aren't represented
		// in the memory.
		if (rde.isReferencing()) {
			TAVariable dummy = VariableConverter.getAddrPtr(scvar);
			Expression varExp = new TAVariableExpression(rde.getNode(), dummy);
			return varExp;
		} else {
			return VariableConverter.derefAddrPtr(scvar);
		}
	}

	// return v; -> $f$return = Tmem[v];
	// OR $f$return = v;
	private static Expression convertReturn(ReturnExpression retExpr,
			SCFunction func) {
		Expression ret = null;
		Expression retStat = retExpr.getReturnStatement();
		if (retStat != null) {
			ret = new TAVariableAssignmentExpression(retExpr.getNode(),
					VariableConverter.getTrnsprtVarForFuncReturn(func),
					convert(retStat, func));
		}
		return ret;
	}

	// a[E]; -> TMem[a+E]
	private static Expression convertArrAccess(ArrayAccessExpression aae,
			SCFunction func) {
		SCVariable scvar = aae.getVar();
		// note: we can use getAccess().get(0) because we only support one-dim
		// arrays
		Expression offset = convert(aae.getAccess().get(0), func);
		Expression ret = VariableConverter.derefAddrPtr(scvar, offset);
		return ret;
	}

	// (E); -> (converted(E))
	private static Expression convertBracket(BracketExpression be,
			SCFunction func) {
		Expression inside = convert(be.getInBrackets(), func);
		Expression newBracket = new BracketExpression(be.getNode(), inside);
		return newBracket;
	}

	private static Expression convertDeltaCount(SCDeltaCountExpression be,
			SCFunction func) {
		Expression ret = new ConstantExpression(be.getNode(),
				Constants.GLOBAL_DELTA_COUNT_INT);
		return ret;
	}

	// new T; -> alloc_dynamic_T(); new T(E) -> alloc_dynamic_T(E)
	private static Expression convertNew(NewExpression newExpr, SCFunction func) {
		// TODO:
		// note: we access newExpr.getArgs().get(0)
		// this won't work with multiple arguments
		// (only necessary if mem model supports member acces, right?)
		String value = "";
		if (newExpr.getArguments().size() > 0) {
			value = convert(newExpr.getArguments().get(0), func)
					.toStringNoSem();
		}
		String funcName = TAMemoryType.getFuncName(
				TAMemoryType.getAllocDynFuncName(value), newExpr.getObjType());
		Expression ret = getFCE(funcName, value);
		return ret;
	}

	// new T[E]; -> alloca_dynamic_arr_T(E)
	private static Expression convertNewArr(NewArrayExpression newArr,
			SCFunction func) {
		String size = convert(newArr.getSize(), func).toStringNoSem();
		String funcName = TAMemoryType.getFuncName(
				TAMemoryType.getAllocDynArrFuncName(), newArr.getObjType());
		Expression ret = getFCE(funcName, size);
		return ret;
	}

	// delete p; -> delete_T(p);
	private static Expression convertDelete(DeleteExpression del,
			SCFunction func) {
		// deletearrexpr inherits from delete
		if (del instanceof DeleteArrayExpression) {
			return convertDeleteArr((DeleteArrayExpression) del, func);
		} else {
			SCVariable scvar = null;
			Expression delExpr = del.getVarToDeleteExpr();
			if (delExpr instanceof SCVariableExpression) {
				scvar = ((SCVariableExpression) delExpr).getVar();
			} else if (delExpr instanceof RefDerefExpression) {
				Expression inner = ((RefDerefExpression) delExpr).getExpression();
				if (inner instanceof SCVariableExpression) {
					scvar = ((SCVariableExpression) inner).getVar();
				} else {
					logger.error("Could not determine variable to delete in expression '{}'.", delExpr.toString());
				}
			} else {
				logger.error("Encountered delete statement with parameter different from simple variable or dereferenced variable in expression '{}'.", delExpr.toString());
			}
			String funcName = TAMemoryType.getFuncName(
					TAMemoryType.getDeleteDynFuncName(),
					scvar.getTypeWithoutSize());
			Expression ret = getFCE(funcName,
					VariableConverter.getAddrPtrName(scvar));
			return ret;
		}
	}

	// delete [] a; -> delete_arr_T(a);
	private static Expression convertDeleteArr(DeleteArrayExpression del,
			SCFunction func) {
		SCVariable scvar = ((SCVariableExpression) del.getVarToDeleteExpr())
				.getVar();
		String funcName = TAMemoryType.getFuncName(
				TAMemoryType.getDeleteDynArrFuncName(),
				scvar.getTypeWithoutSize());
		Expression ret = getFCE(funcName,
				VariableConverter.getAddrPtrName(scvar));
		return ret;
	}

	// a++; -> (convert(a))++;
	private static Expression convertUnary(UnaryExpression expr, SCFunction func) {
		Expression newInner = convert(expr.getExpression(), func);
		Expression ret = new UnaryExpression(expr.getNode(), expr.isPrepost(),
				expr.getOperator(), newInner);
		return ret;
	}

	// (condition ? left : right) -> (convert(condition) ? convert(left) :
	// convert(right))
	private static Expression convertQuestionmark(QuestionmarkExpression expr,
			SCFunction func) {
		Expression cond = convert(expr.getCondition(), func);
		Expression left = convert(expr.getThen(), func);
		Expression right = convert(expr.getElse(), func);
		ConstantExpression ret = new ConstantExpression(expr.getNode(), "("
				+ cond.toStringNoSem() + " ? " + left.toStringNoSem() + " : "
				+ right.toStringNoSem() + ")");
		return ret;
	}

	// var.field -> TMEM[var].field
	// var.p->field -> VMEM[TMEM[var].p].field
	// arr[E].field -> UMEM[arr+E].field
	// arr[E].p->field -> VMEM[UMEM[arr+E].p].field
	// p->field -> VMEM[p].field
	// p->q->field -> QMEM[PMEM[p].q].field
	// OR:
	// var.field -> var.field
	// var.p->field -> VMEM[var.p].field
	// arr[E].field -> UMEM[arr+E].field
	// arr[E].p->field -> VMEM[UMEM[arr+E].p].field
	// p->field -> VMEM[p].field
	// p->q->field -> QMEM[PMEM[p].q].field
	// AccessExpressions concern only structs/class. Modules are never accessed
	// from outside.
	private static Expression convertAccess(AccessExpression expr,
			SCFunction func) {
		Expression newLeft = expr.getLeft();
		Expression newRight = expr.getRight();
		String op = expr.getOp();
		if (op.equals(".")) { // access inner field in the struct mem
			newLeft = convert(newLeft, func);
		} else if (op.equals("->")) { // access inner field via ptr
			SCVariable ptr;
			if (newLeft instanceof AccessExpression) {
				// to get this here right:
				// p->q->field -> QMEM[PMEM[p].q].field
				// The workaround is somewhat strange. Basically,
				// convert "p->q" to a dummy SCPointer called "PMEM[p].q".
				// It's type is the type of q. We need this for the outer deref
				// (see below)
				String type = ((SCVariableExpression) (((AccessExpression) newLeft)
						.getRight())).getVar().getTypeWithoutSize();
				Expression derefedPtrConstExpr = convert(newLeft, func);
				ptr = new SCPointer(derefedPtrConstExpr.toStringNoSem(), type);
			} else {
				ptr = ((SCVariableExpression) expr.getLeft()).getVar();
			}
			// we normally don't do anything with pointers
			// but in this case, we need do access the struct mem as if the
			// variable wasn't a pointer. So we need to deref the outer pointer
			newLeft = VariableConverter.derefAddrPtr(ptr);
		}
		Expression ret = new AccessExpression(expr.getNode(), newLeft, ".",
				newRight);
		return ret;
	}

	private static Expression convertForLoop(ForLoopExpression expr,
			SCFunction func) {
		Expression init = convert(expr.getInitializer(), func);
		Expression cond = convert(expr.getCondition(), func);
		Expression iter = convert(expr.getIterator(), func);

		List<Expression> body = new LinkedList<Expression>();

		for (Expression bodyExpr : expr.getLoopBody()) {
			body.add(convert(bodyExpr, func));
		}

		Expression ret = new ForLoopExpression(expr.getNode(), "", init, cond,
				iter, body);
		return ret;
	}

	private static Expression convertDoWhileLoop(DoWhileLoopExpression expr,
			SCFunction func) {
		Expression cond = convert(expr.getCondition(), func);

		List<Expression> body = new LinkedList<Expression>();

		for (Expression bodyExpr : expr.getLoopBody()) {
			body.add(convert(bodyExpr, func));
		}

		Expression ret = new DoWhileLoopExpression(expr.getNode(), "", cond,
				body);
		return ret;
	}

	private static Expression convertWhileLoop(WhileLoopExpression expr,
			SCFunction func) {
		Expression cond = convert(expr.getCondition(), func);

		List<Expression> body = new LinkedList<Expression>();

		for (Expression bodyExpr : expr.getLoopBody()) {
			body.add(convert(bodyExpr, func));
		}

		Expression ret = new WhileLoopExpression(expr.getNode(), "", cond, body);
		return ret;
	}
}
