/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.tamodel;

import java.util.HashMap;
import java.util.LinkedList;

import de.tub.pes.sc2uppaal.tamodel.expressions.TARecvExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TASendExpression;
import de.tub.pes.syscir.sc_model.expressions.ArrayAccessExpression;
import de.tub.pes.syscir.sc_model.expressions.BinaryExpression;
import de.tub.pes.syscir.sc_model.expressions.BracketExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableExpression;
import de.tub.pes.syscir.sc_model.expressions.UnaryExpression;
import de.tub.pes.syscir.sc_model.variables.SCArray;
import de.tub.pes.syscir.sc_model.variables.SCSimpleType;

/**
 * @author Joachim Fellmuth, Paula Herber, Marcel Pockrandt, Björn Beckmann,
 *         Timm Liebrenz
 * 
 */

public class Constants {

	public static final String END_LINE = System.getProperty("line.separator");

	/**
	 * Most of the identifiers in our model need to be prefixed in some way,
	 * either depending on scope, or thread, or to create some keywords. This
	 * String is used to separated the pre.fixes internally.
	 */
	public static final String PREFIX_DELIMITER = "$";

	/**
	 * Used for comments in config files.
	 */
	public static final String COMMENT = "#";

	public static final String DOT = ".";

	public static String CFG_ROOT = "";

	/**
	 * NULL for pointers.
	 */
	public static final String NULL = "NULL";
	public static final String NULL_VALUE = "-1";
	public static final String NULL_LOWERCASE = "null"; // uppaal is case
														// sensitive

	/**
	 * Time Resolution in nano seconds. Clock values are calculated from time in
	 * ns / SC_TIME_RESOLUTION
	 */
	public static int SC_TIME_RESOLUTION = 1; // 1 ns 10000; // 10 us

	/**
	 * An optional prefix for keywords. This can be used to make keywords
	 * unique. If this String is empty, certain keywords are reserved and can
	 * not be used as identifiers.
	 */
	public static final String KEYWORD_PREFIX = "";

	/**
	 * Scheduler-Channel - activates a process<BR>
	 * Scope: global
	 * 
	 */
	public static final String GLOBAL_ACTIVATE_CHAN = KEYWORD_PREFIX
			+ "activate";

	/**
	 * Scheduler-Channel - a process deactivates itself<BR>
	 * Scope: global
	 * 
	 */
	public static final String GLOBAL_DEACTIVATE_CHAN = KEYWORD_PREFIX
			+ "deactivate";

	/**
	 * Scheduler-Variable - number of processes that are ready to be executed<BR>
	 * Scope: global
	 * 
	 */
	public static final String GLOBAL_READY_PROCS_INT = KEYWORD_PREFIX
			+ "readyprocs";

	/**
	 * Scheduler-Variable - number of requested updates on channels<BR>
	 * Scope: global
	 */
	public static final String GLOBAL_UPDATE_REQUESTS_INT = KEYWORD_PREFIX
			+ "updaterequests";

	/**
	 * Scheduler-Variable - number of requested updates on channels<BR>
	 * Scope: global
	 */
	public static final String GLOBAL_DELTA_COUNT_INT = KEYWORD_PREFIX
			+ "deltacount";

	/**
	 * Scheduler-Channel - starts update phase of a channel<BR>
	 * Scope: global
	 * 
	 */
	public static final String GLOBAL_UPDATE_START_CHAN = KEYWORD_PREFIX
			+ "updatestart";

	/**
	 * Scheduler-Channel - channel indicates end of update phase<BR>
	 * Scope: global
	 * 
	 */
	public static final String GLOBAL_UPDATE_END_CHAN = KEYWORD_PREFIX
			+ "updateend";

	/**
	 * Scheduler-Channel - time is passing<BR>
	 * - only for wait statements with timed wait<BR>
	 * Scope: global
	 * 
	 */
	public static final String GLOBAL_ADVANCE_TIME_CHAN = KEYWORD_PREFIX
			+ "advancetime";

	/**
	 * Scheduler-Channel - starts delta delay notification phase Scope: global
	 * 
	 */
	public static final String GLOBAL_DELTA_DELAY_CHAN = KEYWORD_PREFIX
			+ "deltadelay";

	/**
	 * Scheduler-Channel - marks end of initialization phase Scope: global
	 * 
	 */
	public static final String GLOBAL_INIT_DONE_CHAN = KEYWORD_PREFIX
			+ "initdone";

	// specific template names
	/**
	 * name of the scheduler template
	 */
	public static final String TEMPLATE_NAME_SCHEDULER = "SchedulerTemplate";

	/**
	 * name of the event template
	 */
	public static final String TEMPLATE_NAME_SC_EVENT = "SCEventTemplate";

	/**
	 * name of the thread starter template
	 */
	public static final String TEMPLATE_NAME_THREAD_STARTER = "ThreadStarterTemplate";

	/**
	 * name of the thread starter template without initialization
	 */
	public static final String TEMPLATE_NAME_THREAD_STARTER_NO_INIT = "ThreadStarterNoInitTemplate";

	/**
	 * name of the method starter template
	 */
	public static final String TEMPLATE_NAME_METHOD_STARTER = "MethodStarterTemplate";

	/**
	 * name of the method starter template initialization
	 */
	public static final String TEMPLATE_NAME_METHOD_STARTER_NO_INIT = "MethodStarterNoInitTemplate";

	/**
	 * name of the template to start the update phase
	 */
	public static final String TEMPLATE_NAME_UPDATE_STARTER = "UpdateStarterTemplate";

	/**
	 * name of the template to mark the end of the initialization phase
	 */
	public static final String TEMPLATE_NAME_INITIALIZER = "InitializerTemplate";

	/**
	 * maximal number of concurrent transport requests per PEQ
	 * 
	 */
	public static int GLOBAL_PEQ_SIZE = 4;

	/**
	 * The debuglevel. Ranges from 0 (no debug output) to 5 (all debug output)
	 */
	public static int DEBUG_LEVEL = 5;

	/**
	 * flag that determines whether the name of the class should be used in the
	 * variables for the template bindings
	 */
	public static boolean USE_CLASSNAME_IN_PREFIX = true;

	/**
	 * the reserved keyword in the peq size file to set the global peq size.
	 */
	public static String GLOBAL_PEQ = "global";

	public static String SC_PEQ_TEMPLATE_ELEMENT = "peq_element";

	/**
	 * PEQ-Channel-Keyword - notify on a peq, the channel gets this keyword
	 * suffix
	 * 
	 */
	public static final String SC_PEQ_TEMPLATE_PARAM_ELEMENT = "peq_element"
			+ PREFIX_DELIMITER + "ctrl";

	/**
	 * PEQ-Param-Keyword - transaction parameter for peq notifications
	 */
	public static final String SC_PEQ_TEMPLATE_PARAM_ELEMENT_TRANS = "peq_element"
			+ PREFIX_DELIMITER + "param" + PREFIX_DELIMITER + "trans";

	/**
	 * PEQ-Param-Keyword - phase parameter for peq notifications
	 */
	public static final String SC_PEQ_TEMPLATE_PARAM_ELEMENT_PHASE = "peq_element"
			+ PREFIX_DELIMITER + "param" + PREFIX_DELIMITER + "phase";

	/**
	 * PEQ-Param-Keyword - delay parameter for peq notifications
	 */
	public static final String SC_PEQ_TEMPLATE_PARAM_ELEMENT_DELAY = "peq_element"
			+ PREFIX_DELIMITER + "param" + PREFIX_DELIMITER + "delay";

	/**
	 * Event-Channel-Keyword - the wait channel for events gets this keyword
	 * suffix
	 */
	public static final String SC_EVENT_TEMPLATE_PARAM_WAIT = "wait";

	/**
	 * Event-Channel-Keyword - delta delay notify or timed notify on an event,
	 * the channel gets this keyword suffix
	 * 
	 */
	public static final String SC_EVENT_TEMPLATE_PARAM_NOTIFY = "notify";

	/**
	 * Event-Channel-Keyword - immediate notify, the channel gets this keyword
	 * suffix
	 */
	public static final String SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM = "notify_imm";

	/**
	 * Event-Channel-Keyword - delay value parameter for timed delay
	 */
	public static final String SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME = "notify_t";

	/**
	 * Local variable in any function template that contains a timed wait. Can
	 * not be used as an identifier then.
	 */
	public static final String SC_EVENT_LOCAL_WAIT_CLOCK = "wait_clock";

	/**
	 * The delay value for timed notifications.
	 */
	public static final String SC_EVENT_LOCAL_DELAY_INT = "ndelay";

	/**
	 * The keyword suffix for control channels of functions.
	 */
	public static final String LOCAL_FUNCTION_CTRL_KEYWORD = KEYWORD_PREFIX
			+ "ctrl";

	/**
	 * The keyword suffix for global parameter definitions of functions.
	 */
	public static final String LOCAL_FUNCTION_PARAM_KEYWORD = KEYWORD_PREFIX
			+ "param";

	/**
	 * The keyword suffix for global definition of local variables of functions.
	 */
	public static final String LOCAL_VARIABLE_KEYWORD = KEYWORD_PREFIX
			+ "local";

	/**
	 * The keyword suffix for the global return value definition of functions
	 */
	public static final String LOCAL_FUNCTION_RETURN_KEYWORD = KEYWORD_PREFIX
			+ "return";

	/**
	 * switch expression
	 */
	public static final String LOCAL_SWITCH_EXPR_INTENTIFIER = KEYWORD_PREFIX
			+ "switch";

	/**
	 * The sensitivity keyword for threads/processes.
	 */
	public static final String LOCAL_SENSITIVE_KEYWORD = "sensitive";

	/**
	 * The name of the timeout event of threads/processes.
	 */
	public static final String LOCAL_TIMEOUT_EVENT_KEYWORD = "timeoutevent";

	/**
	 * Global sensitivity channel name(suffix).
	 */
	public static final String LOCAL_SENSITIVE_CHANNEL = "sensitivechan";

	/**
	 * Request update channel keyword, to be used in primitive channel modules
	 */
	public static final String REQUEST_UPDATE_CHANNEL = "requestupdate";

	/**
	 * Local indentifier for update method ctrl channel.
	 */
	public static final String UPDATE_CTRL_CHANNEL = "update"
			+ PREFIX_DELIMITER + "ctrl";

	/**
	 * 
	 */
	public static final String REQUEST_UPDATE_NUMBER = "requestupdatenumber";

	/**
	 * Global dummy broadcast channel.
	 */
	public static final String DUMMY_BROADCAST_CHANNEL = "dummybchannel";

	/**
	 * Global dummy channel.
	 */
	public static final String DUMMY_CHANNEL = "dummychannel";

	/**
	 * Global dummy integer.
	 */
	public static final String DUMMY_INTEGER = "dummyinteger";

	/**
	 * Global dummy clock.
	 */
	public static final String DUMMY_CLOCK = "dummyinteger";

	public static final String INSTANCE_NAME = "this";

	public static final String QUEUEELEMENT_PREFIX = "queueelement_";

	public static final String VERSION = "1.0";

	public static final HashMap<String, String> defaultValues = new HashMap<String, String>();
	static {
		HashMap<String, String> hm = new HashMap<String, String>();
		hm.put("int", "0");
		hm.put("unsigned_int", "0");
		hm.put("short", "0");
		hm.put("unsigned_short", "0");
		hm.put("bool", "false");

		// XXX: default values
		hm.put("sc_time", "0");
		hm.put("uint64", "0");
		hm.put("tlm_dmi", "0");
		hm.put("tlm_sync_enum", "0");
		hm.put("void", "0");
		hm.put("sc_uint", "0");

		hm.put("peq_with_cb_and_phase", "0");

		defaultValues.putAll(hm);
	}

	public static String getDefaultValueName(String type) {
		String upper = type.toUpperCase();
		return "DEFAULT_" + upper + "_VALUE";
	}

	/**
	 * The name used for each initial location in a TATemplate.
	 */
	public static final String INIT_LOCATION_NAME = "init_location";

	/**
	 * This name is used to mark all locations in which assertions should be
	 * evaluated.
	 */
	private static final String ASSERTION_LOCATION_NAME = "assertion_violation";

	/**
	 * Counter for assertions. Ensures each assert location gets a unique
	 * identifier.
	 */
	private static int assertionCounter = 0;

	public static final String SCMAIN_FINISHED_CHANNEL = "scmodules_initialized";
	public static final String SCMAIN_TEMPLATE_NAME = "SCMainTemplate";
	public static final String SCMAIN_TEMPLATE_INSTANCE_NAME = "scmain";
	public static final String SCMAIN_TEMPLATE_END_LOCATION = "scmain_end";
	public static final String SCMAIN_NATIVE_METHOD_NAME = "sc_main";
	public static final String MEM_ERR_NOT_ENOUGH_MEM = "MEM_ERR_NOT_ENOUGH_MEM";
	public static final String ERR_ASSERTION_VIOLATION = "ERR_ASSERTION_VIOLATION";

	/**
	 * Inserts all static predefined templates and variables into the given
	 * TAModel instance.
	 * 
	 * @param model
	 */
	public static void createPredefined(TAModel model) {
		createInitializerTemplate(model);
		createThreadStarterTemplate(model, true, TEMPLATE_NAME_THREAD_STARTER);
		createThreadStarterTemplate(model, false,
				TEMPLATE_NAME_THREAD_STARTER_NO_INIT);
		createMethodStarterTemplate(model, true, TEMPLATE_NAME_METHOD_STARTER);
		createMethodStarterTemplate(model, false,
				TEMPLATE_NAME_METHOD_STARTER_NO_INIT);
		createSCEventTemplate(model);
		createUpdateHandlerTemplate(model);

		model.addVariable(new TAChannel(DUMMY_BROADCAST_CHANNEL, true));
		model.addVariable(new TAChannel(DUMMY_CHANNEL, false));
		model.addVariable(new TAInteger(DUMMY_INTEGER));
		model.addVariable(new TABoolean(Constants.ERR_ASSERTION_VIOLATION, new ConstantExpression(null, "false")));

	}

	/**
	 * Adds a SystemC-scheduler template and instance to a given TAModel
	 * instance. Contains of one template and certain global variables. For the
	 * exact scheduler template see documentation.
	 * 
	 * @param model
	 *            Puts the scheduler in there.
	 * @param numberOfThreads
	 *            The number of processes WITH initialization to initialize the
	 *            scheduler with
	 */
	public static void createScheduler(TAModel model, int numberOfThreads) {
		model.addVariable(new TAChannel(GLOBAL_ACTIVATE_CHAN, false));
		model.addVariable(new TAChannel(GLOBAL_DEACTIVATE_CHAN, false));
		model.addVariable(new TAChannel(GLOBAL_UPDATE_START_CHAN, false));
		model.addVariable(new TAChannel(GLOBAL_UPDATE_END_CHAN, false));
		model.addVariable(new TAChannel(GLOBAL_INIT_DONE_CHAN, true));
		model.addVariable(new TAChannel(GLOBAL_DELTA_DELAY_CHAN, true));
		model.addVariable(new TAChannel(GLOBAL_ADVANCE_TIME_CHAN, true));

		model.addVariable(new TAInteger(GLOBAL_DELTA_COUNT_INT, 0));
		model.addVariable(new TAInteger(GLOBAL_READY_PROCS_INT, numberOfThreads));
		model.addVariable(new TAInteger(GLOBAL_UPDATE_REQUESTS_INT, 0));

		TATemplate scheduler = new TATemplate(TEMPLATE_NAME_SCHEDULER);
		model.addTemplate(scheduler);

		TALocation beforeSCMain = scheduler.createInitLocation();
		beforeSCMain.setUrgent(true);
		TALocation execute = scheduler.createLocation("execute");
		execute.setUrgent(true);
		TALocation evaluate = scheduler.createLocation("evaluate");
		evaluate.setUrgent(true);
		TALocation update = scheduler.createLocation("update");
		update.setUrgent(true);
		TALocation updating = scheduler.createLocation("updating");
		updating.setUrgent(true);
		TALocation nextDelta = scheduler.createLocation("nextdelta");
		nextDelta.setUrgent(true);

		TALocation timeProgress = scheduler.createLocation("time_progress");

		TATransition afterSCMain = new TATransition(beforeSCMain, evaluate,
				new TARecvExpression(null, SCMAIN_FINISHED_CHANNEL));
		scheduler.addTransition(afterSCMain);

		TATransition trans;

		// Partial Order Reduction
		if (model.isPorActive()) {
			TALocation porUpdateEnabled = scheduler.createLocation();
			porUpdateEnabled.setUrgent(true);
			trans = new TATransition(evaluate, porUpdateEnabled);
			trans.setGuard(new BinaryExpression(null, new ConstantExpression(
					null, GLOBAL_READY_PROCS_INT), ">", new ConstantExpression(
					null, "0")));
			trans.addUpdateExpression(new ConstantExpression(null,
					"recompute_enabled()"));
			scheduler.addTransition(trans);

			TATransition updateEnabled = new TATransition(porUpdateEnabled,
					execute, new TASendExpression(null, GLOBAL_ACTIVATE_CHAN));
			scheduler.addTransition(updateEnabled);
		} else {
			trans = new TATransition(evaluate, execute, new TASendExpression(
					null, GLOBAL_ACTIVATE_CHAN));
			trans.setGuard(new BinaryExpression(null, new ConstantExpression(
					null, GLOBAL_READY_PROCS_INT), ">", new ConstantExpression(
					null, "0")));
			scheduler.addTransition(trans);
		}

		trans = new TATransition(execute, evaluate, new TARecvExpression(null,
				GLOBAL_DEACTIVATE_CHAN));
		scheduler.addTransition(trans);

		trans = new TATransition(evaluate, update);
		trans.setGuard(new BinaryExpression(null, new ConstantExpression(null,
				GLOBAL_READY_PROCS_INT), "==",
				new ConstantExpression(null, "0")));
		trans.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, GLOBAL_DELTA_COUNT_INT), "=",
				new BinaryExpression(null, new BracketExpression(null,
						new BinaryExpression(null, new ConstantExpression(null,
								GLOBAL_DELTA_COUNT_INT), "+",
								new ConstantExpression(null, "1"))), "%",
						new ConstantExpression(null, "2"))));
		scheduler.addTransition(trans);

		TALocation enabled_updaterequests = scheduler.createLocation();
		enabled_updaterequests.setUrgent(true);
		trans = new TATransition(update, enabled_updaterequests);
		trans.setGuard(new BinaryExpression(null, new ConstantExpression(null,
				GLOBAL_UPDATE_REQUESTS_INT), ">", new ConstantExpression(null,
				"0")));

		if (model.isUpdateRequestDeterminism()) {
				trans.addUpdateExpression(new ConstantExpression(null,
						"recompute_enabled_ur()"));
		}

		scheduler.addTransition(trans);

		trans = new TATransition(enabled_updaterequests, updating,
				new TASendExpression(null, GLOBAL_UPDATE_START_CHAN));
		scheduler.addTransition(trans);

		trans = new TATransition(updating, update, new TARecvExpression(null,
				GLOBAL_UPDATE_END_CHAN));
		scheduler.addTransition(trans);

		trans = new TATransition(update, nextDelta, new TASendExpression(null,
				GLOBAL_DELTA_DELAY_CHAN));
		trans.setGuard(new BinaryExpression(null, new ConstantExpression(null,
				GLOBAL_UPDATE_REQUESTS_INT), "==", new ConstantExpression(null,
				"0")));
		if (model.isPorActive() && model.getPor().isSleepSet()) {
			trans.addUpdateExpression(new ConstantExpression(null,
					"reset_sleeps()"));
		}

		scheduler.addTransition(trans);

		trans = new TATransition(nextDelta, timeProgress);
		trans.setGuard(new BinaryExpression(null, new ConstantExpression(null,
				GLOBAL_READY_PROCS_INT), "==",
				new ConstantExpression(null, "0")));
		scheduler.addTransition(trans);

		trans = new TATransition(nextDelta, evaluate);
		trans.setGuard(new BinaryExpression(null, new ConstantExpression(null,
				GLOBAL_READY_PROCS_INT), ">", new ConstantExpression(null, "0")));
		scheduler.addTransition(trans);

		trans = new TATransition(timeProgress, evaluate, new TARecvExpression(
				null, GLOBAL_ADVANCE_TIME_CHAN));
		scheduler.addTransition(trans);

		model.addTemplateInstance(new TATemplateInstance(scheduler, "scheduler"));
	}

	/**
	 * Creates a little template that indicates the end of the initialization
	 * phase of the scheduler.
	 * 
	 * @param model
	 */
	public static void createInitializerTemplate(TAModel model) {
		TATemplate template = new TATemplate(TEMPLATE_NAME_INITIALIZER);
		model.addTemplate(template);

		TALocation start = template.createLocation();
		start.setUrgent(true);
		template.setInitLocation(start);
		TALocation middle = template.createLocation();
		middle.setCommitted(true);
		TALocation end = template.createLocation();

		template.addTransition(new TATransition(start, middle,
				new TARecvExpression(null, GLOBAL_DELTA_DELAY_CHAN)));
		template.addTransition(new TATransition(middle, end,
				new TASendExpression(null, GLOBAL_INIT_DONE_CHAN)));

		model.addTemplateInstance(template.getInstance("initialization"));
	}

	/**
	 * Creates a thread starter template. A Thread in SystemC is only started
	 * once and handles suspension and sensitivity in the method logic. So the
	 * logic to start and maintain a thread is really simple. The template only
	 * contains a method call to the thread's method and activation/deactivation
	 * logic (see scheduler). Optionally waits for end of init phase before
	 * starting.
	 * 
	 * @param ta
	 *            The TAModel to put the template in.
	 * @param init
	 *            To determine which type of thread starter template should be
	 *            created - with or without automatic initialization phase.
	 * @param name
	 *            The actual name of the template.
	 */
	public static void createThreadStarterTemplate(TAModel ta, boolean init,
			String name) {
		TATemplate threadStarter = new TATemplate(name);
		threadStarter.addParameter(new TAChannel(LOCAL_FUNCTION_CTRL_KEYWORD,
				false));

		if (ta.isPorActive()) {
			TAInteger cln = new TAInteger("cln");
			cln.setConstant(true);
			threadStarter.addParameter(cln);
		}

		TALocation start = threadStarter.createLocation();
		start.setUrgent(true);

		TALocation beforeFunc = threadStarter.createLocation();
		beforeFunc.setUrgent(true);
		TALocation whileFunc = threadStarter.createLocation();
		TALocation afterFunc = threadStarter.createLocation();
		afterFunc.setUrgent(true);
		TALocation end = threadStarter.createLocation();

		if (init) {
			threadStarter.setInitLocation(start);
		} else {
			TALocation waitForInit = threadStarter.createLocation();
			waitForInit.setUrgent(true);
			threadStarter.setInitLocation(waitForInit);

			TATransition initT = new TATransition(waitForInit, start,
					new TARecvExpression(null, GLOBAL_INIT_DONE_CHAN));
			initT.addUpdateExpression(new UnaryExpression(null,
					UnaryExpression.POST, "++", new ConstantExpression(null,
							GLOBAL_READY_PROCS_INT)));
			if (ta.isPorActive()) {
				LinkedList<Expression> list = new LinkedList<Expression>();
				list.add(new SCVariableExpression(null, new SCSimpleType("cln",
						"int")));
				initT.addUpdateExpression(new BinaryExpression(null,
						new ArrayAccessExpression(null, new SCArray("runnable",
								null), list), " = ", new ConstantExpression(
								null, "true")));

			}
			threadStarter.addTransition(initT);
		}

		TATransition activeTransition;
		if (ta.isPorActive()) {
			LinkedList<Expression> list = new LinkedList<Expression>();
			list.add(new SCVariableExpression(null, new SCSimpleType("cln",
					"int")));

			TALocation porOptim = threadStarter.createUrgentLocation();

			activeTransition = new TATransition(start, porOptim,
					new TARecvExpression(null, GLOBAL_ACTIVATE_CHAN));
			activeTransition.setGuard(new BinaryExpression(null,
					new ArrayAccessExpression(null,
							new SCArray("enabled", null), list), " == ",
					new ConstantExpression(null, "true")));
			activeTransition.addUpdateExpression(new BinaryExpression(null,
					new ArrayAccessExpression(null, new SCArray("runnable",
							null), list), " = ", new ConstantExpression(null,
							"false")));
			if (ta.getPor().isSleepSet()) {
				activeTransition.addUpdateExpression(new ConstantExpression(
						null, "recompute_sleeps(cln)"));
			}

			TATransition porOptimTransitionTrue = new TATransition(porOptim,
					beforeFunc);
			porOptimTransitionTrue.setGuard(new BinaryExpression(null,
					new SCVariableExpression(null, new SCSimpleType(
							"no_indep_atomicblocks", "bool")), " == ",
					new ConstantExpression(null, "true")));

			porOptimTransitionTrue.addUpdateExpression(

			new BinaryExpression(null, new ArrayAccessExpression(null,
					new SCArray("enabled", null), list), " = ",
					new ConstantExpression(null, "false")));

			threadStarter.addTransition(porOptimTransitionTrue);

			TATransition porOptimTransitionFalse = new TATransition(porOptim,
					beforeFunc);
			porOptimTransitionFalse.setGuard(new BinaryExpression(null,
					new SCVariableExpression(null, new SCSimpleType(
							"no_indep_atomicblocks", "bool")), " == ",
					new ConstantExpression(null, "false")));
			threadStarter.addTransition(porOptimTransitionFalse);

		} else {
			activeTransition = new TATransition(start, beforeFunc,
					new TARecvExpression(null, GLOBAL_ACTIVATE_CHAN));
		}

		threadStarter.addTransition(activeTransition);
		threadStarter.addTransition(new TATransition(beforeFunc, whileFunc,
				new TASendExpression(null, LOCAL_FUNCTION_CTRL_KEYWORD)));
		threadStarter.addTransition(new TATransition(whileFunc, afterFunc,
				new TARecvExpression(null, LOCAL_FUNCTION_CTRL_KEYWORD)));
		TATransition deactivate = new TATransition(afterFunc, end,
				new TASendExpression(null, GLOBAL_DEACTIVATE_CHAN));
		deactivate.addUpdateExpression(new UnaryExpression(null,
				UnaryExpression.POST, "--", new ConstantExpression(null,
						GLOBAL_READY_PROCS_INT)));
		threadStarter.addTransition(deactivate);

		ta.addTemplate(threadStarter);
	}

	/**
	 * Creates a method starter template. An sc_method in SystemC is executed
	 * every time an event that is in the methods sensitivity list is notified.
	 * In general the code of a method does not contain wait statements, its
	 * suspension and reactivation is done by the scheduler. This is why the
	 * method starter template contains an infinite loop which starts the
	 * execution of the thread's function each time the sensitivity channel is
	 * activated.
	 * 
	 * @param ta
	 *            The TAModel to put the template.
	 * @param init
	 *            Create method starter with initialization if true, else
	 *            without init.
	 * @param name
	 *            The name of the method starter template.
	 */

	public static void createMethodStarterTemplate(TAModel ta, boolean init,
			String name) {

		TATemplate methodStarter = new TATemplate(name);

		if (ta.isPorActive()) {
			TAInteger cln = new TAInteger("cln");
			cln.setConstant(true);
			methodStarter.addParameter(cln);
		}

		methodStarter.addParameter(new TAChannel(LOCAL_FUNCTION_CTRL_KEYWORD,
				false));
		methodStarter
				.addParameter(new TAChannel(LOCAL_SENSITIVE_KEYWORD, true));
		TALocation waitForSensitive = methodStarter.createLocation();

		TALocation activate = methodStarter.createLocation();
		activate.setUrgent(true);
		TALocation beforeFunc = methodStarter.createLocation();
		beforeFunc.setUrgent(true);
		TALocation whileFunc = methodStarter.createLocation();
		whileFunc.setUrgent(true);
		TALocation afterFunc = methodStarter.createLocation();
		afterFunc.setUrgent(true);

		TATransition first = new TATransition(waitForSensitive, activate,
				new TARecvExpression(null, LOCAL_SENSITIVE_KEYWORD));
		first.addUpdateExpression(new UnaryExpression(null,
				UnaryExpression.POST, "++", new ConstantExpression(null,
						GLOBAL_READY_PROCS_INT)));
		methodStarter.addTransition(first);

		if (ta.isPorActive()) {

			LinkedList<Expression> list = new LinkedList<Expression>();
			list.add(new SCVariableExpression(null, new SCSimpleType("cln",
					"int")));

			first.addUpdateExpression(

			new BinaryExpression(null, new ArrayAccessExpression(null,
					new SCArray("runnable", null), list), " = ",
					new ConstantExpression(null, "true")));

			TALocation porOptim = methodStarter.createUrgentLocation();

			TATransition activeTransition = new TATransition(activate,
					porOptim, new TARecvExpression(null, GLOBAL_ACTIVATE_CHAN));

			activeTransition.setGuard(new BinaryExpression(null,
					new ArrayAccessExpression(null,
							new SCArray("enabled", null), list), " == ",
					new ConstantExpression(null, "true")));

			activeTransition.addUpdateExpression(new BinaryExpression(null,
					new ArrayAccessExpression(null, new SCArray("runnable",
							null), list), " = ", new ConstantExpression(null,
							"false")));
			if (ta.getPor().isSleepSet()) {
				activeTransition.addUpdateExpression(new ConstantExpression(
						null, "recompute_sleeps(cln)"));
			}

			methodStarter.addTransition(activeTransition);

			TATransition porOptimTransitionTrue = new TATransition(porOptim,
					beforeFunc);
			porOptimTransitionTrue.setGuard(new BinaryExpression(null,
					new SCVariableExpression(null, new SCSimpleType(
							"no_indep_atomicblocks", "bool")), " == ",
					new ConstantExpression(null, "true")));

			porOptimTransitionTrue.addUpdateExpression(

			new BinaryExpression(null, new ArrayAccessExpression(null,
					new SCArray("enabled", null), list), " = ",
					new ConstantExpression(null, "false")));

			methodStarter.addTransition(porOptimTransitionTrue);

			TATransition porOptimTransitionFalse = new TATransition(porOptim,
					beforeFunc);
			porOptimTransitionFalse.setGuard(new BinaryExpression(null,
					new SCVariableExpression(null, new SCSimpleType(
							"no_indep_atomicblocks", "bool")), " == ",
					new ConstantExpression(null, "false")));
			methodStarter.addTransition(porOptimTransitionFalse);

		} else {

			methodStarter.addTransition(new TATransition(activate, beforeFunc,
					new TARecvExpression(null, GLOBAL_ACTIVATE_CHAN)));
		}
		methodStarter.addTransition(new TATransition(beforeFunc, whileFunc,
				new TASendExpression(null, LOCAL_FUNCTION_CTRL_KEYWORD)));
		methodStarter.addTransition(new TATransition(whileFunc, afterFunc,
				new TARecvExpression(null, LOCAL_FUNCTION_CTRL_KEYWORD)));
		TATransition last = new TATransition(afterFunc, waitForSensitive,
				new TASendExpression(null, GLOBAL_DEACTIVATE_CHAN));
		last.addUpdateExpression(new UnaryExpression(null,
				UnaryExpression.POST, "--", new ConstantExpression(null,
						GLOBAL_READY_PROCS_INT)));
		methodStarter.addTransition(last);

		if (init) {
			methodStarter.setInitLocation(activate);
		} else {
			methodStarter.setInitLocation(waitForSensitive);
		}

		ta.addTemplate(methodStarter);
	}

	/**
	 * This template contains some logic to handle the notification mechanisms
	 * of each event. See documentation for picture and details.
	 * 
	 * @param ta
	 *            The Model where to put the template.
	 */
	public static void createSCEventTemplate(TAModel ta) {
		TATemplate eventTemplate = new TATemplate(TEMPLATE_NAME_SC_EVENT);

		eventTemplate.addParameter(new TAChannel(SC_EVENT_TEMPLATE_PARAM_WAIT,
				true));
		eventTemplate.addParameter(new TAChannel(
				SC_EVENT_TEMPLATE_PARAM_NOTIFY, false));
		eventTemplate.addParameter(new TAChannel(
				SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM, false));
		eventTemplate.addParameter(new TAInteger(
				SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME));

		eventTemplate.addLocalVar(new TAClock(SC_EVENT_LOCAL_WAIT_CLOCK));
		eventTemplate.addLocalVar(new TAInteger(SC_EVENT_LOCAL_DELAY_INT));

		TALocation init = eventTemplate.createLocation("initL");
		TALocation notifyRequested = eventTemplate
				.createLocation("notify_requested");
		notifyRequested.setCommitted(true);
		TALocation waitForDelta = eventTemplate
				.createLocation("wait_for_delta");
		waitForDelta.setInvariant(new BinaryExpression(null,
				new ConstantExpression(null, SC_EVENT_LOCAL_WAIT_CLOCK), "<=",
				new ConstantExpression(null, SC_EVENT_LOCAL_DELAY_INT)));

		TALocation notifyNow = eventTemplate.createLocation("notify_now");
		notifyNow.setCommitted(true);
		TALocation advancingTime = eventTemplate
				.createLocation("advancing_time");
		advancingTime.setCommitted(true);

		eventTemplate.setInitLocation(init);

		TATransition startRegNotify = new TATransition(init, notifyRequested,
				new TARecvExpression(null, SC_EVENT_TEMPLATE_PARAM_NOTIFY));
		startRegNotify.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, SC_EVENT_LOCAL_DELAY_INT), "=",
				new ConstantExpression(null,
						SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME)));
		startRegNotify.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, SC_EVENT_LOCAL_WAIT_CLOCK), "=",
				new ConstantExpression(null, "0")));
		eventTemplate.addTransition(startRegNotify);

		TATransition reinitTime = new TATransition(notifyRequested,
				waitForDelta);
		reinitTime.setGuard(new BinaryExpression(null, new BinaryExpression(
				null, new ConstantExpression(null, SC_EVENT_LOCAL_WAIT_CLOCK),
				"+", new ConstantExpression(null,
						SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME)), "<",
				new ConstantExpression(null, SC_EVENT_LOCAL_DELAY_INT)));
		reinitTime.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, SC_EVENT_LOCAL_DELAY_INT), "=",
				new ConstantExpression(null,
						SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME)));
		reinitTime.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, SC_EVENT_LOCAL_WAIT_CLOCK), "=",
				new ConstantExpression(null, "0")));
		eventTemplate.addTransition(reinitTime);

		eventTemplate.addTransition(new TATransition(waitForDelta,
				notifyRequested, new TARecvExpression(null,
						SC_EVENT_TEMPLATE_PARAM_NOTIFY)));

		TATransition noReinitTime = new TATransition(notifyRequested,
				waitForDelta);
		noReinitTime.setGuard(new BinaryExpression(null, new BinaryExpression(
				null, new ConstantExpression(null, SC_EVENT_LOCAL_WAIT_CLOCK),
				"+", new ConstantExpression(null,
						SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME)), ">=",
				new ConstantExpression(null, SC_EVENT_LOCAL_DELAY_INT)));
		eventTemplate.addTransition(noReinitTime);

		TATransition doDelta = new TATransition(waitForDelta, notifyNow,
				new TARecvExpression(null, GLOBAL_DELTA_DELAY_CHAN));
		doDelta.setGuard(new ConstantExpression(null, SC_EVENT_LOCAL_DELAY_INT
				+ " == 0"));
		eventTemplate.addTransition(doDelta);

		TATransition doTimed = new TATransition(waitForDelta, notifyNow,
				new TASendExpression(null, GLOBAL_ADVANCE_TIME_CHAN));
		doTimed.setGuard(new BinaryExpression(null, new BinaryExpression(null,
				new ConstantExpression(null, SC_EVENT_LOCAL_WAIT_CLOCK), "==",
				new ConstantExpression(null, SC_EVENT_LOCAL_DELAY_INT)), "&&",
				new BinaryExpression(null, new ConstantExpression(null,
						SC_EVENT_LOCAL_DELAY_INT), "!=",
						new ConstantExpression(null, "0"))));
		eventTemplate.addTransition(doTimed);

		TATransition otherTimeAdvance = new TATransition(waitForDelta,
				advancingTime, new TARecvExpression(null,
						GLOBAL_ADVANCE_TIME_CHAN));
		otherTimeAdvance.setGuard(new BinaryExpression(null,
				new ConstantExpression(null, SC_EVENT_LOCAL_DELAY_INT), "!=",
				new ConstantExpression(null, "0")));
		eventTemplate.addTransition(otherTimeAdvance);

		TATransition notYet = new TATransition(advancingTime, waitForDelta);
		notYet.setGuard(new BinaryExpression(null, new ConstantExpression(null,
				SC_EVENT_LOCAL_WAIT_CLOCK), "<", new ConstantExpression(null,
				SC_EVENT_LOCAL_DELAY_INT)));
		eventTemplate.addTransition(notYet);

		TATransition advanceToo = new TATransition(advancingTime, notifyNow);
		advanceToo.setGuard(new BinaryExpression(null, new ConstantExpression(
				null, SC_EVENT_LOCAL_WAIT_CLOCK), "==", new ConstantExpression(
				null, SC_EVENT_LOCAL_DELAY_INT)));
		eventTemplate.addTransition(advanceToo);

		TATransition sendWait = new TATransition(notifyNow, init);
		sendWait.setSync(new TASendExpression(null,
				SC_EVENT_TEMPLATE_PARAM_WAIT));
		sendWait.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, SC_EVENT_LOCAL_WAIT_CLOCK), "=",
				new ConstantExpression(null, "0")));
		eventTemplate.addTransition(sendWait);

		eventTemplate
				.addTransition(new TATransition(waitForDelta, notifyNow,
						new TARecvExpression(null,
								SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM)));
		eventTemplate
				.addTransition(new TATransition(init, notifyNow,
						new TARecvExpression(null,
								SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM)));

		ta.addTemplate(eventTemplate);
	}

	/**
	 * Creates the update starter template and inserts it into the given TA
	 * model. An update starter connects the scheduler instance with the update
	 * method of a primitive channel. Therefore it acts similarly to a thread
	 * starter template. Please see the documentation for a more detailed
	 * description.
	 * 
	 * @param ta
	 *            The TAModel instance to insert the update starter template.
	 */
	public static void createUpdateHandlerTemplate(TAModel ta) {

		TATemplate template = new TATemplate(TEMPLATE_NAME_UPDATE_STARTER);

		template.addParameter(new TAChannel(REQUEST_UPDATE_CHANNEL, true));
		template.addParameter(new TAChannel(UPDATE_CTRL_CHANNEL, false));

		if (ta.isUpdateRequestDeterminism()) {
				TAInteger cln = new TAInteger(REQUEST_UPDATE_NUMBER);
				cln.setConstant(true);
				template.addParameter(cln);
		}

		TALocation init = template.createLocation();

		TALocation requested = template.createLocation("requested");
		requested.setUrgent(true);

		TALocation startUpdate = template.createLocation("start");
		startUpdate.setUrgent(true);

		TALocation waitForFun = template.createLocation("wait");
		waitForFun.setUrgent(true);

		TALocation endUpdate = template.createLocation("end");
		endUpdate.setUrgent(true);

		LinkedList<Expression> list = new LinkedList<Expression>();
		list.add(new SCVariableExpression(null, new SCSimpleType(
				REQUEST_UPDATE_NUMBER, "int")));

		TATransition acceptRequest = new TATransition(init, requested,
				new TARecvExpression(null, REQUEST_UPDATE_CHANNEL));

		acceptRequest.addUpdateExpression(new UnaryExpression(null,
				UnaryExpression.POST, "++", new ConstantExpression(null,
						GLOBAL_UPDATE_REQUESTS_INT)));

		if (ta.isUpdateRequestDeterminism()) {
				acceptRequest.addUpdateExpression(

				new BinaryExpression(null, new ArrayAccessExpression(null,
						new SCArray("runnable_ur", null), list), " = ",
						new ConstantExpression(null, "true")));


		}

		template.addTransition(acceptRequest);

		TATransition update_start = new TATransition(requested, startUpdate,
				new TARecvExpression(null, GLOBAL_UPDATE_START_CHAN));

		if (ta.isUpdateRequestDeterminism()) {

				update_start.setGuard(

				new BinaryExpression(null, new ArrayAccessExpression(null,
						new SCArray("enabled_ur", null), list), " == ",
						new ConstantExpression(null, "true")));

		}

		template.addTransition(update_start);

		template.addTransition(new TATransition(startUpdate, waitForFun,
				new TASendExpression(null, UPDATE_CTRL_CHANNEL)));
		template.addTransition(new TATransition(waitForFun, endUpdate,
				new TARecvExpression(null, UPDATE_CTRL_CHANNEL)));

		TATransition returnCtrl = new TATransition(endUpdate, init,
				new TASendExpression(null, GLOBAL_UPDATE_END_CHAN));
		returnCtrl.addUpdateExpression(new UnaryExpression(null,
				UnaryExpression.POST, "--", new ConstantExpression(null,
						GLOBAL_UPDATE_REQUESTS_INT)));

		if (ta.isUpdateRequestDeterminism()) {

				returnCtrl.addUpdateExpression(

				new BinaryExpression(null, new ArrayAccessExpression(null,
						new SCArray("runnable_ur", null), list), " = ",
						new ConstantExpression(null, "false")));

		}

		template.addTransition(returnCtrl);

		template.setInitLocation(init);

		ta.addTemplate(template);
	}

	/**
	 * Concatenates all strings, delimited by the PREFIX_DELIMETER (f.e.
	 * createDelimitedString("Hello", "World") => "Hello#World")
	 * 
	 * @param strings
	 * @return
	 */
	public static String createDelimitedString(String... strings) {
		String ret = "";
		for (int i = 0; i < strings.length; i++) {
			ret += strings[i];
			if (i < strings.length - 1) {
				ret += PREFIX_DELIMITER;
			}
		}
		return ret;
	}

	public static String generateAssertionFailStateName() {
		String ret = ASSERTION_LOCATION_NAME + "_" + (++assertionCounter);
		return ret;
	}
}
