/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.optimization;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.TATemplate;
import de.tub.pes.sc2uppaal.tamodel.TATransition;
import de.tub.pes.sc2uppaal.tamodel.TAVariable;
import de.tub.pes.syscir.sc_model.expressions.Expression;

/**
 * This optimization resets all local variables. This enables the model checker
 * to work faster.
 * 
 */

public class VariableResetOptimizer implements Optimizer {

	private static Logger logger = LogManager
			.getLogger(VariableResetOptimizer.class.getName());

	private int varCount = 0;

	@Override
	public void run(TAModel ta) {

		for (String templateName : ta.getTemplates().keySet()) {
			TATemplate template = ta.getTemplates().get(templateName);

			// find all ingoing transitions to the initial location
			for (TATransition tr : template.getIncomingTransitions(template
					.getInitLocation())) {

				// reset all local variables
				for (TAVariable var : template.getLocalVars()) {
					if (!var.isConstant()) // don't reset constants!
						resetVar(tr, var);
				}
				// reset all local variables which came from the cbv params
				for (TAVariable var : template.getcbvParsAsLocalVars()) {
					resetVar(tr, var);
				}
			}

			// matches '=.*FUNCNAME$return'
			Pattern functret = Pattern.compile(".*=.*?" + "([^\\s]+" + "\\"
					+ Constants.PREFIX_DELIMITER
					+ Constants.LOCAL_FUNCTION_RETURN_KEYWORD + ")");
			Matcher m = null;
			TAVariable var;
			List<TAVariable> varsToReset = new ArrayList<TAVariable>();
			for (TATransition tr : template.getTransitions()) {
				varsToReset = new ArrayList<TAVariable>();
				for (Expression expr : tr.getUpdate()) {
					String s = expr.toStringNoSem();
					m = functret.matcher(s);
					if (m.matches()) {
						var = template.getParameter(m.group(1));
						varsToReset.add(var);
					}
				}
				for (TAVariable v : varsToReset) {
				//	logger.info(v);
					resetVar(tr, v);
				}
			}
		}
		logger.info("Optimization statistics: Resetted " + varCount
				+ " local variables.");
	}

	/**
	 * Resets the variable to a predefined value and adds the expression to the
	 * transition.
	 * 
	 * @param tr
	 * @param var
	 */
	public void resetVar(TATransition tr, TAVariable var) {
		if (var != null && !var.isVolatile()) {
			Expression exp = var.getResetExpr();
			if (exp != null) {
				tr.addUpdateExpression(exp);
				varCount++;
			}
		}
	}

}
