/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.optimization;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.TATemplate;
import de.tub.pes.sc2uppaal.tamodel.TATemplateInstance;
import de.tub.pes.sc2uppaal.tamodel.TAVariable;

/**
 * 
 * @author Joachim Fellmuth, Paula Herber
 * 
 */

public class UnusedTemplateOptimizer implements Optimizer {

	private static Logger logger = LogManager
			.getLogger(UnusedTemplateOptimizer.class.getName());

	private int instCount;
	private int varCount;
	private int templCount;

	@Override
	public void run(TAModel ta) {
		HashMap<TATemplate, List<TAVariable>> calledMethods = ta
				.createMethodCallerMap();
		HashMap<TAVariable, List<TATemplateInstance>> calledChannels = new HashMap<TAVariable, List<TATemplateInstance>>();
		instCount = 0;
		varCount = 0;
		templCount = 0;

		// note: we assume here that unused template parameters have already
		// been optimized
		// another template is only calling the current template if it still has
		// its ctrl channel as a template parameter

		// get all template instances
		// and find out which ctrl channels they are synchronizing on (besides
		// their own ctrl channel)
		for (String tiName : ta.getSystemDeclaration().keySet()) {
			TATemplateInstance instance = ta.getSystemDeclaration().get(tiName);
			List<TAVariable> called = calledMethods.get(instance.getTemplate());
			for (TAVariable localChannel : called) {
				String globalChannelName = instance.getGlobalParam(localChannel
						.getName());
				if (globalChannelName == null)
					continue;

				TAVariable globalChannel = ta
						.getGlobalVariable(globalChannelName);

				if (globalChannel == null)
					continue;

				if (!globalChannel.getName().startsWith(
						instance.getName() + Constants.PREFIX_DELIMITER)) {
					List<TATemplateInstance> insts = calledChannels
							.get(globalChannel);
					if (insts != null)
						insts.add(instance);
					else {
						insts = new LinkedList<TATemplateInstance>();
						insts.add(instance);
						calledChannels.put(globalChannel, insts);
					}
				}
			}
		}

		LinkedList<TATemplateInstance> unused = new LinkedList<TATemplateInstance>();

		// now go through the instances again and see if they are used at all
		for (String tiName : ta.getSystemDeclaration().keySet()) {
			TATemplateInstance instance = ta.getSystemDeclaration().get(tiName);
			TAVariable localCtrlChannel = instance.getTemplate()
					.getCtrlParameter();
			if (localCtrlChannel != null) {
				TAVariable globalCtrlChannel = ta.getGlobalVariable(instance
						.getGlobalParam(localCtrlChannel.getName()));

				if (calledChannels.get(globalCtrlChannel) == null) {
					unused.add(instance);
				}
			}
		}

		// delete the unused instances and see if some other depend on them
		int cnt = 0;
		while (cnt < unused.size()) {
			TATemplateInstance instance = unused.get(cnt);

			for (TAVariable localChannel : calledMethods.get(instance
					.getTemplate())) {
				String globalChannelName = instance.getGlobalParam(localChannel
						.getName());
				if (globalChannelName == null)
					continue;
				TAVariable globalChannel = ta
						.getGlobalVariable(globalChannelName);
				if (globalChannel == null) {
					continue;
				}
				if (!globalChannel.getName().startsWith(
						instance.getName() + Constants.PREFIX_DELIMITER)) {
					List<TATemplateInstance> insts = calledChannels
							.get(globalChannel);
					insts.remove(instance);
					if (insts.size() == 0) {
						TATemplateInstance otherInst = ta
								.getSystemDeclaration()
								.get(globalChannel
										.getName()
										.replaceAll(
												Constants.PREFIX_DELIMITER
														+ Constants.LOCAL_FUNCTION_CTRL_KEYWORD,
												""));
						// if(otherInst != null)
						try {
							if (otherInst.getTemplate().getCtrlParameter() != null) {
								unused.addLast(otherInst);
							}
						} catch (NullPointerException e) {
							// System.out.println("break");
						}
					}

				}
			}
			deleteTemplateInstance(ta, instance);
			cnt++;
		}

		LinkedList<TATemplate> unusedTempl = new LinkedList<TATemplate>();

		// now remove templates if all instances have been removed
		for (String templName : ta.getTemplates().keySet()) {
			boolean used = false;
			for (String instName : ta.getSystemDeclaration().keySet()) {
				TATemplateInstance instance = ta.getSystemDeclaration().get(
						instName);
				if (instance.getTemplateName().equals(templName))
					used = true;
			}

			if (!used) {
				unusedTempl.add(ta.getTemplates().get(templName));
			}
		}

		// remove unused templates
		for (TATemplate templ : unusedTempl) {
			ta.getTemplates().remove(templ.getName());

			templCount++;
		}

		logger.info("Optimization statistics: Deleted " + instCount
				+ " template instances.");
		logger.info("Optimization statistics: Deleted " + varCount
				+ " global variables.");
		logger.info("Optimization statistics: Deleted " + templCount
				+ " templates.");
	}

	private void deleteTemplateInstance(TAModel ta, TATemplateInstance instance) {
		ta.getSystemDeclaration().remove(instance.getName());
		instCount++;
		LinkedList<TAVariable> varsToDelete = new LinkedList<TAVariable>();
		for (TAVariable var : ta.getDeclaration()) {
			String instName = instance.getName();
			int i = instName.lastIndexOf(Constants.PREFIX_DELIMITER);
			String tmp0 = instName.substring(0, i);
			String tmp1 = instName.substring(i);
			i = tmp0.lastIndexOf(Constants.PREFIX_DELIMITER);
			if (i > 0)
				tmp0 = tmp0.substring(0, i);
			instName = tmp0.concat(tmp1);
			if (var.getName().indexOf(instName + Constants.PREFIX_DELIMITER) == 0) {
				varsToDelete.add(var);
			}
		}
		for (TAVariable var : varsToDelete) {
			ta.getDeclaration().remove(var);
			varCount++;
		}
	}

	protected void printCalledMethodsMap(
			HashMap<TATemplate, List<TAVariable>> map) {
		for (TATemplate t : map.keySet()) {
			System.out.print(t.getName() + ": ");
			for (TAVariable c : map.get(t)) {
				System.out.print(c.getName() + " ");
			}
			System.out.println();
		}
	}

	protected void printCallingInstsMap(
			HashMap<TAVariable, List<TATemplateInstance>> map) {
		for (TAVariable c : map.keySet()) {
			System.out.print(c.getName() + ": ");
			for (TATemplateInstance t : map.get(c)) {
				System.out.print(t.getName() + " ");
			}
			System.out.println();
		}
	}

}
