/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.optimization;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.TAChannel;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.TATemplateInstance;
import de.tub.pes.sc2uppaal.tamodel.TAVariable;

/**
 * 
 * @author Joachim Fellmuth, Paula Herber
 * 
 */

public class UnusedEventOptimizer implements Optimizer {

	private static Logger logger = LogManager
			.getLogger(UnusedEventOptimizer.class.getName());

	private int eventCount;
	private int varCount;

	public void run(TAModel ta) {

		eventCount = 0;
		varCount = 0;

		// list to collect global event channels
		List<TAChannel> globalEventChannels = new LinkedList<TAChannel>();

		// hashmap to associate each global event channel with the corresponding
		// event template instance
		HashMap<TAChannel, TATemplateInstance> eventChannelInstances = new HashMap<TAChannel, TATemplateInstance>();

		// collect global event channels
		for (String tiName : ta.getSystemDeclaration().keySet()) {
			TATemplateInstance instance = ta.getSystemDeclaration().get(tiName);

			// SCEventTemplate -> collect global event channels and their
			// association to event template instance
			if (instance.getTemplateName().startsWith(
					Constants.TEMPLATE_NAME_SC_EVENT)) {
				for (TAVariable var : instance.getTemplate().getParameters()) {
					String varName = var.getName();
					if (var instanceof TAChannel
							&& (varName
									.startsWith(Constants.SC_EVENT_TEMPLATE_PARAM_WAIT)
									|| varName
											.startsWith(Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY) || varName
										.startsWith(Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM))) {
						TAChannel globalChannel = (TAChannel) ta
								.getGlobalVariable(instance
										.getGlobalParam(varName));
						globalEventChannels.add(globalChannel);
						eventChannelInstances.put(globalChannel, instance);
					}
				}
			}
		}

		List<TATemplateInstance> usedEvents = new LinkedList<TATemplateInstance>();

		// for each template instance, check if it uses any of the global event
		// channels
		for (String tiName : ta.getSystemDeclaration().keySet()) {

			TATemplateInstance instance = ta.getSystemDeclaration().get(tiName);

			// ignore event template instances
			if (!instance.getTemplateName().startsWith(
					Constants.TEMPLATE_NAME_SC_EVENT)) {
				for (TAVariable var : instance.getTemplate().getParameters()) {
					String varName = var.getName();
					if (var instanceof TAChannel) {
						TAVariable globalChannel = ta
								.getGlobalVariable(instance
										.getGlobalParam(varName));
						if (globalEventChannels.contains(globalChannel)) {
							TATemplateInstance usedEvent = eventChannelInstances
									.get(globalChannel);
							// this TemplateInstance uses a global event channel
							if (!usedEvents.contains(usedEvent)) {
								usedEvents.add(usedEvent);
								// System.out.println("Event " +
								// globalChannel.getName() + " is used");
							}
						}
					}
				}
			}
		}

		LinkedList<TATemplateInstance> unused = new LinkedList<TATemplateInstance>();

		for (TAChannel channel : globalEventChannels) {
			TATemplateInstance event = eventChannelInstances.get(channel);
			if (!usedEvents.contains(event)) {
				// System.out.println("Event " + event.getName() +
				// " is unused ");
				if (!unused.contains(event)) {
					unused.add(event);
					deleteTemplateInstance(ta, event);
					eventCount++;
				}
			}
		}
		logger.info("Optimization statistics: Deleted " + eventCount
				+ " event instances.");
		logger.info("Optimization statistics: Deleted " + varCount
				+ " global variables.");
	}

	private void deleteTemplateInstance(TAModel ta, TATemplateInstance instance) {
		ta.getSystemDeclaration().remove(instance.getName());
		eventCount++;
		LinkedList<TAVariable> varsToDelete = new LinkedList<TAVariable>();
		for (TAVariable var : ta.getDeclaration()) {
			if (var.getName().indexOf(
					instance.getName() + Constants.PREFIX_DELIMITER) == 0) {
				varsToDelete.add(var);
			}
		}
		for (TAVariable var : varsToDelete) {
			ta.getDeclaration().remove(var);
			varCount++;
		}
	}
}
