/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.optimization;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.TATypedef;
import de.tub.pes.sc2uppaal.tamodel.TAVariable;
import de.tub.pes.syscir.engine.util.IOUtil;

/**
 * This Converter adds necessary typedefs to the TAModel when working with
 * complex types. Since UPPAAL only supports a handful of simple types, other
 * types have to be redefined to these simple types. This converter is
 * implemented as a singleton, using the enum singleton pattern.
 * 
 * @author Pfeffer
 * 
 */
public enum TypedefOptimizer implements Optimizer {

	INSTANCE;

	public static String TYPEDEF_CFG_FILE = Constants.CFG_ROOT
			+ "config/typedefs.properties";

	private static final Logger logger = LogManager
			.getLogger(TypedefOptimizer.class.getName());

	/**
	 * Optimizer main method. Adds all typdefs from the config file to the
	 * TAModel and takes care of enumerator constant initialization and cleanup.
	 * 
	 * @param ta
	 *            output parameter
	 */
	@Override
	public void run(TAModel ta) {
		try {
			BufferedReader reader = new BufferedReader(new InputStreamReader(
					IOUtil.getInputStream(TYPEDEF_CFG_FILE)));
			Map<String, TATypedef> typedefs = buildTableFromCfg(reader);
			ta.addTypedefs(typedefs.values());
			List<TAVariable> duplicateDefinitions = getDuplicateEnumDefinitions(
					typedefs, ta.getDeclaration());
			ta.getDeclaration().removeAll(duplicateDefinitions);
		} catch (FileNotFoundException e) {
			logger.warn("Can't find typedef cfg file. Skipping.");
			return;
		} catch (IOException e) {
			logger.warn("Error in config file: " + e.getMessage());
			return;
		}
	}

	/**
	 * Parses the typdefs configuration file to create a table of typdefs and
	 * adds enumrator constants defined in the file.
	 * 
	 * @param reader
	 * @return
	 * @throws IOException
	 */
	private Map<String, TATypedef> buildTableFromCfg(BufferedReader reader)
			throws IOException {
		Map<String, TATypedef> typedefs = new HashMap<String, TATypedef>();
		while (reader.ready()) {
			String line = reader.readLine();
			if (line == null)
				break;
			line = line.intern();
			if (line.startsWith(Constants.COMMENT)) {
				continue;
			} else {
				TATypedef typedef;
				String[] entry = line.split(" ");
				if ("const".equals(entry[0].toLowerCase())) {
					// enum value
					if (entry.length == 4) {
						if (typedefs.containsKey(entry[1])) {
							typedef = typedefs.get(entry[1]);
							typedef.addConstant(entry[2],
									Integer.parseInt(entry[3]));
						} else {
							logger.warn(
									"type {} for const value {} not found. Skipping.",
									entry[1], entry[2]);
							continue;
						}
					} else {
						logger.warn(
								"malformed enum value: {}. Invalid amount of arguments.",
								line);
						continue;
					}
				} else {
					// typedef
					if (entry.length == 2) {
						typedef = new TATypedef(entry[0], entry[1]);
						typedefs.put(entry[1], typedef);
					} else {
						logger.warn(
								"malformed typdef: {}. Invalid amount of arguments.",
								line);
						continue;
					}
				}
			}
		}
		return typedefs;
	}

	/**
	 * Checks all variables in a given list for duplicate definition of a
	 * enumerator constant already defined by a this optimizer.
	 * 
	 * @param typedefs
	 * @param declaration
	 * @return
	 */
	private List<TAVariable> getDuplicateEnumDefinitions(
			Map<String, TATypedef> typedefs, List<TAVariable> declaration) {
		List<TAVariable> duplicateEnumDefinition = new LinkedList<TAVariable>();
		for (TAVariable globalVar : declaration) {
			if (globalVar.isConstant()
					&& typedefs.containsKey(globalVar.getType())) {
				TATypedef typedef = typedefs.get(globalVar.getType());
				for (String name : typedef.getDefinedNames()) {
					if (globalVar.getName().endsWith(name))
						duplicateEnumDefinition.add(globalVar);
				}
			}
		}
		return duplicateEnumDefinition;
	}
}
