/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.optimization;

import java.util.LinkedList;
import java.util.List;

import de.tub.pes.sc2uppaal.tamodel.TALocation;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.TATemplate;
import de.tub.pes.sc2uppaal.tamodel.TATransition;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.ExpressionBlock;

/**
 * This Optimizer splits transitions containing more than one statement into a
 * set of sequential transitions each containing exactly one statement. The
 * order is equivalent to the uppaal execution semantics. This is not a real
 * optimization but enables other optimizers to perform better. It should not be
 * called by the system but by each optimizer in need for this.
 * 
 * @author pockrandt
 * 
 */
public class TransitionSplitOptimizer implements Optimizer {

	@Override
	public void run(TAModel ta) {
		for (TATemplate temp : ta.getTemplates().values()) {
			// we have to prefetch the list to avoid concurrent modification
			// exceptions
			List<TATransition> newTransList = new LinkedList<TATransition>();
			List<TALocation> newLocationList = new LinkedList<TALocation>();
			List<TATransition> removeTransList = new LinkedList<TATransition>();
			for (TATransition trans : temp.getTransitions()) {
				if (trans.getGuard() != null && trans.getSync() != null
						&& trans.getSync().isSend()) {
					continue;
				}
				List<Expression> selects = getExpressions(trans.getSelect());
				List<Expression> updates = getExpressions(trans.getUpdate());
				int statementCount = selects.size() + updates.size()
						+ (trans.getGuard() == null ? 0 : 1)
						+ (trans.getSync() == null ? 0 : 1);
				if (statementCount > 1) {
					removeTransList.add(trans);
					TALocation initialLoc = trans.getStart();
					TALocation finalLoc = trans.getEnd();
					TALocation currentLoc = initialLoc;
					// the execution order is GUARD, RECEIVE, SELECT, UPDATE,
					// SEND
					if (trans.getGuard() != null && trans.getSync() != null
							&& trans.getSync().isRecv()) {
						// a GUARD and RECEIVE should not be separated
						TALocation between;
						if (statementCount > 2) {
							between = createLocation(currentLoc);
							newLocationList.add(between);
						} else {
							between = finalLoc;
						}
						TATransition newTrans = new TATransition(
								currentLoc, between);
						currentLoc = between;
						newTrans.setGuard(trans.getGuard());
						newTrans.setSync(trans.getSync());
						newTransList.add(newTrans);
						statementCount -= 2;
					} else {
						if (trans.getGuard() != null) {
							TALocation between;
							if (statementCount > 1) {
								between = createLocation(currentLoc);
								newLocationList.add(between);
							} else {
								between = finalLoc;
							}
							TATransition newTrans = new TATransition(
									currentLoc, between);
							currentLoc = between;
							newTrans.setGuard(trans.getGuard());
							newTransList.add(newTrans);
							statementCount--;
						}
						if (trans.getSync() != null && trans.getSync().isRecv()) {
							TALocation between;
							if (statementCount > 1) {
								between = createLocation(currentLoc);
								newLocationList.add(between);
							} else {
								between = finalLoc;
							}
							TATransition newTrans = new TATransition(
									currentLoc, between);
							currentLoc = between;
							newTrans.setSync(trans.getSync());
							newTransList.add(newTrans);
							statementCount--;
						}
					}
					if (!selects.isEmpty()) {
						for (Expression exp : selects) {
							TALocation between;
							if (statementCount > 1) {
								between = createLocation(currentLoc);
								newLocationList.add(between);
							} else {
								between = finalLoc;
							}
							TATransition newTrans = new TATransition(
									currentLoc, between);
							currentLoc = between;
							newTrans.addSelectExpression(exp);
							newTransList.add(newTrans);
							statementCount--;
						}
					}
					if (!updates.isEmpty()) {
						for (Expression exp : updates) {
							TALocation between;
							if (statementCount > 1) {
								between = createLocation(currentLoc);
								newLocationList.add(between);
							} else {
								between = finalLoc;
							}
							TATransition newTrans = new TATransition(
									currentLoc, between);
							currentLoc = between;
							newTrans.addUpdateExpression(exp);
							newTransList.add(newTrans);
							statementCount--;
						}
					}
					if (trans.getSync() != null && trans.getSync().isSend()) {
						TALocation between;
						if (statementCount > 1) {
							between = createLocation(currentLoc);
							newLocationList.add(between);
						} else {
							between = finalLoc;
						}
						TATransition newTrans = new TATransition(currentLoc,
								between);
						currentLoc = between;
						newTrans.setSync(trans.getSync());
						newTransList.add(newTrans);
						statementCount--;
					}
				}
			}
			for (TALocation loc : newLocationList) {
				temp.addLocation(loc);
			}
			for (TATransition tra : removeTransList) {
				temp.removeTransition(tra);
			}
			for (TATransition tra : newTransList) {
				temp.addTransition(tra);
			}

		}

	}

	/**
	 * Extracts statements out of block statements.
	 * 
	 * @param list
	 * @return
	 */
	private List<Expression> getExpressions(List<Expression> list) {
		List<Expression> ret = new LinkedList<Expression>();
		for (Expression exp : list) {
			if (exp instanceof ExpressionBlock) {
				ExpressionBlock expBl = (ExpressionBlock) exp;
				ret.addAll(getExpressions(expBl.getBlock()));
			} else {
				ret.add(exp);
			}
		}
		return ret;
	}

	/**
	 * Creates a new location. This is usually an urgent location unless the
	 * starting location is a commited location.
	 * 
	 * @return
	 */
	private TALocation createLocation(TALocation loc) {
		TALocation ret = new TALocation();
		if (loc.isCommitted()) {
			ret.setCommitted(true);
			ret.setUrgent(false);
		} else {
			ret.setUrgent(true);
			ret.setCommitted(false);
		}
		return ret;
	}
}
