/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.optimization;

import java.util.LinkedList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.TALocation;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.TATemplate;
import de.tub.pes.sc2uppaal.tamodel.TATransition;
import de.tub.pes.syscir.sc_model.expressions.Expression;

/**
 * 
 * @author Joachim Fellmuth
 * 
 */

public class TransitionOptimizer implements Optimizer {

	private static Logger logger = LogManager
			.getLogger(TransitionOptimizer.class.getName());

	int transCount;
	int locCount;

	public void run(TAModel ta) {
		transCount = 0;
		locCount = 0;

		for (String templateName : ta.getTemplates().keySet()) {
			TATemplate template = ta.getTemplates().get(templateName);
			deleteEmptyTransitions(template);
			deleteTransitLocations(template);
		}

		logger.info("Optimization statistics: Deleted " + transCount
				+ " Transitions.");
		logger.info("Optimization statistics: Deleted " + locCount
				+ " Locations.");
	}

	private void deleteEmptyTransitions(TATemplate template) {
		for (TATransition t : getEmptyTransitions(template)) {
			TALocation start = t.getStart();
			TALocation end = t.getEnd();
			List<TATransition> concurrent = template
					.getOutgoingTransitions(start);

			if (start.isUrgent() && concurrent.size() == 1
					&& !(start.getName() != null && end.getName() != null)) {
				// we can delete this transition
				if (template.getInitLocation() == start)
					template.setInitLocation(end);
				for (TATransition incoming : template
						.getIncomingTransitions(start)) {
					incoming.setEnd(end);
				}
				if (start.getName() != null && end.getName() == null) {
					// we have to preserve the name of the location.
					end.setName(start.getName());
				}
				template.getLocations().remove(start);
				template.getTransitions().remove(t);
				locCount++;
				transCount++;
			}
		}

	}

	/**
	 * Deletes all locations in an unbranched chain transitions (e.g., -> l -> l
	 * -> l) and merges the transitions. This is always possible if the location
	 * is unnamed and the transitions only contain update statements.
	 * 
	 * @param template
	 * @return
	 */
	private LinkedList<TALocation> deleteTransitLocations(TATemplate template) {
		LinkedList<TALocation> locs = new LinkedList<TALocation>();
		for (TALocation loc : template.getLocations()) {
			if (loc.isUrgent() && loc.getName() == null) {
				List<TATransition> in = template.getIncomingTransitions(loc);
				List<TATransition> out = template.getOutgoingTransitions(loc);
				if (in.size() == 1 && out.size() == 1) {
					TATransition incoming = in.get(0);
					TATransition outgoing = out.get(0);
					if (incoming.getSelect().isEmpty()
							&& incoming.getGuard() == null
							&& (incoming.getSync() == null || incoming
									.getSync().toString().indexOf("?") == incoming
									.getSync().toString().length() - 1)
							&& outgoing.getSelect().isEmpty()
							&& outgoing.getGuard() == null
							&& outgoing.getSync() == null) {
						incoming.setEnd(outgoing.getEnd());
						outgoing.setStart(incoming.getStart());
						for (Expression ex : outgoing.getUpdate()) {
							incoming.addUpdateExpression(ex);
						}
						template.getTransitions().remove(outgoing);
						transCount++;
						locs.add(loc);
					}

				}
			}
		}
		for (TALocation loc : locs) {
			template.getLocations().remove(loc);
			locCount++;
		}
		return locs;
	}

	private LinkedList<TATransition> getEmptyTransitions(TATemplate template) {
		LinkedList<TATransition> ts = new LinkedList<TATransition>();
		for (TATransition t : template.getTransitions()) {
			if (t.getSelect().isEmpty() && t.getGuard() == null
					&& t.getSync() == null && t.getUpdate().isEmpty()) {
				ts.add(t);
			}
		}
		return ts;
	}

}
