/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.optimization;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.ExpressionConverter;
import de.tub.pes.sc2uppaal.tamodel.TALocation;
import de.tub.pes.sc2uppaal.tamodel.TAMemoryType;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.TATemplate;
import de.tub.pes.sc2uppaal.tamodel.TATransition;
import de.tub.pes.syscir.engine.util.Pair;
import de.tub.pes.syscir.sc_model.expressions.BinaryExpression;
import de.tub.pes.syscir.sc_model.expressions.BracketExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.UnaryExpression;

/**
 * Add guards at all locations where outgoing transitions access the memory.
 * Stop verfication if (1) ptr == NULL (2) intMemState[arr_index] !=
 * MEM_ARR_HEAD|TAIL
 * 
 * @author rschroeder
 */

public class PtrChkOptimizer implements Optimizer {
	private static Logger logger = LogManager.getLogger(PtrChkOptimizer.class
			.getName());

	@Override
	public void run(TAModel ta) {
		// first we have to ensure that we can access all update statements
		TransitionSplitOptimizer tso = new TransitionSplitOptimizer();
		tso.run(ta);

		int i = 0;
		for (String tmplName : ta.getTemplates().keySet()) {
			TATemplate tmpl = ta.getTemplates().get(tmplName);
			List<Pair<TALocation, Expression>> failStatesToAdd = new LinkedList<Pair<TALocation, Expression>>();
			// Generate guards for each memory access
			for (TATransition trans : tmpl.getTransitions()) {
				Expression guard = getGuard(trans.getUpdate());
				if (guard != null) {
					trans.setGuard(guard);
					failStatesToAdd.add(new Pair<TALocation, Expression>(trans
							.getStart(), guard));
				}
			}
			// generate failstates and transitions
			for (Pair<TALocation, Expression> p : failStatesToAdd) {
				// Create failstate with unique name and add to template
				String name = "mem_access_fail_" + i;
				i++;
				TALocation fail = new TALocation(name);
				fail.setUrgent(true);
				tmpl.addLocation(fail);

				// create transition to failstate with negated guard. Also sets
				// the memory access error flag to true
				TATransition trans = new TATransition(p.getFirst(), fail);
				Expression guard = p.getSecond();
				Expression negatedGuard = new UnaryExpression(guard
						.getNode(), UnaryExpression.PRE, "!", new BracketExpression(guard.getNode(), guard));
				trans.setGuard(negatedGuard);
				trans.addUpdateExpression(new BinaryExpression(guard
						.getNode(), new ConstantExpression(guard
						.getNode(), "MEM_ERR_ACCESS_FAIL"), "=",
						new ConstantExpression(guard.getNode(), "true")));
				tmpl.addTransition(trans);
			}
		}
	}

	private Expression getGuard(List<Expression> update) {
		/*
		 * If we don't use defragmentation, the following statements are
		 * possible: intMem[num_readable] = intMem[size] - intMem[free],
		 * intMem[num_written] = intMem[num_written] + 1, intMem[buf +
		 * intMem[wi]] = intMem[val], intMem[wi] = (intMem[wi] + 1) %
		 * intMem[size], intMem[free] = intMem[free] - 1
		 */

		// check every mem access
		// matches "intMem["
		Pattern access = Pattern.compile("(\\w+)"
				+ TAMemoryType.getMemArrName() + "\\[");
		Map<String, String> ptrsToCheck = new HashMap<String, String>();
		for (Expression expr : update) {
			String curUpdate = expr.toStringNoSem();
			Matcher varMatcher = access.matcher(curUpdate);
			while (varMatcher.find()) {
				// get the data type for which the memory is used
				String type = varMatcher.group(1);
				for (int i = 1; i <= varMatcher.groupCount(); i++) {
					StringBuffer inside = new StringBuffer();
					int count = 1;
					int j = varMatcher.end();
					while (count > 0) {
						inside.append(curUpdate.charAt(j));
						j++;
						// make sure we get everything inside the "outer" []
						if (curUpdate.charAt(j) == ']') {
							count--;
						} else if (curUpdate.charAt(j) == '[') {
							count++;
						}
					}
					String index = inside.toString();
					if (!ptrsToCheck.containsKey(index)) {
						ptrsToCheck.put(index, type);
					}
				}
			}
		}
		if (ptrsToCheck.size() > 0) {
			List<String> guardList = new ArrayList<String>();
			access = Pattern.compile("([^\\s]*)\\s\\W+(.*)"); // "arr + i" -
																// "(head)\s\W+(offset)"
			for (Map.Entry<String, String> entry : ptrsToCheck.entrySet()) {
				String type = entry.getValue();
				String headPlusIndex = entry.getKey();
				String funcName = TAMemoryType.getFuncName(
						TAMemoryType.getAddrChkFuncName(), type);
				guardList.add(ExpressionConverter.getFCE(funcName,
						headPlusIndex).toStringNoSem());
				// check if all array accesses are valid, too
				Matcher arrMatcher = access.matcher(headPlusIndex);
				if (arrMatcher.find()) {
					String head = arrMatcher.group(1);
					String index = arrMatcher.group(2);
					String params = head + ", " + index;
					guardList.add(0, ExpressionConverter.getFCE(funcName, head).toStringNoSem());
					funcName = TAMemoryType.getFuncName(
							TAMemoryType.getIsArrElemFuncName(), type);
					guardList.add(ExpressionConverter.getFCE(funcName, params)
							.toStringNoSem());
				}
			}
			// if we get here we have at least one entry so this is safe
			StringBuffer guard = new StringBuffer();
			guard.append(guardList.get(0));
			if (guardList.size() > 1) {
				int i = 1;
				while (i < guardList.size() - 1) {
					guard.append(" && ");
					guard.append(guardList.get(i));
					i++;
				}
				guard.append(" && ");
				guard.append(guardList.get(i));
			}
			return new ConstantExpression(null, guard.toString());
		} else {
			return null;
		}
	}
}
