/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.optimization;

import java.util.ArrayList;

import de.tub.pes.syscir.analysis.data_race_analyzer.AtomicBlock;
import de.tub.pes.syscir.analysis.data_race_analyzer.DataRaceAnalyzer;
import de.tub.pes.syscir.engine.util.Pair;
import de.tub.pes.syscir.sc_model.expressions.Expression;

/**
 * 
 * @author Björn Beckmann
 *
 */

public class POR {
	
	/**
	 *  Data Race Analyzer
	 */
	public DataRaceAnalyzer dra;
	
	/**
	 * current number of atomic blocks
	 */
	private int atomicBlockCounter;
		
	/**
	 * wait statement offset
	 */
	private ArrayList<Pair<Integer,Expression>> waitOffsetList;
	
	/**
	 * Partial Order Reduction with persistents and sleep sets
	 */
	private boolean sleepSet;
	
	public POR(DataRaceAnalyzer dra){
		this.dra=dra;
		this.setAtomicBlockCounter(0);
		this.setWaitOffsetList(new ArrayList<Pair<Integer,Expression>>());
		this.setSleepSet(false);
	}

	public POR(DataRaceAnalyzer dra, boolean sleepSet){
		this.dra=dra;
		this.setAtomicBlockCounter(0);
		this.setWaitOffsetList(new ArrayList<Pair<Integer,Expression>>());
		this.setSleepSet(sleepSet);
	}
	
	public String getGlobalPORVariables(){
		String var = getGlobalPORVariablesDefault();
		if(sleepSet){
			return var.concat(getGlobalPORVariablesSleepSets());
		}else{
			return var;
		}
	}
	
	public String getGlobalPORFunctionDefinitions(){
		if(sleepSet){
			return getGlobalPORFunctionDefinitionsSleepSets();
		}else{
			return getGlobalPORFunctionDefinitionsDefault();
		}
	}
	
	public String getGlobalPORVariablesDefault(){
		dra.sortAtomicBlocks();
		dra.mergeAtomicBlockDependencys();
		StringBuilder ret = new StringBuilder();
		ret.append("\n");
		ret.append("///////////////////\n");
		ret.append("// POR \n");
		ret.append("///////////////////\n");
		ret.append("\n");
		ret.append("const int BLOCKCOUNT = "+this.dra.getAtomicBlocks().size()+";\n");
		ret.append("meta bool runnable[BLOCKCOUNT]={");
		for(int i=0; i<this.dra.getAtomicBlocks().size();i++){
			if(this.dra.getAtomicBlocks().get(i).isProcessNoInit()){
				ret.append("false");
			}else{
				ret.append("true");
			}
			
			if(i!=this.dra.getAtomicBlocks().size()-1){
				ret.append(",");
			}
		}
		ret.append("};\n\n");
		ret.append("meta bool enabled[BLOCKCOUNT]={");
		for(int i=0; i<this.dra.getAtomicBlocks().size();i++){
			ret.append("false");
			if(i!=this.dra.getAtomicBlocks().size()-1){
				ret.append(",");
			}
		}
		ret.append("};\n\n");
		ret.append("const bool dependency[BLOCKCOUNT][BLOCKCOUNT]={");
		
		for(int i=0; i<this.dra.getAtomicBlocks().size();i++){
		ret.append("{");
		
			for(int k=0; k<this.dra.getAtomicBlocks().size();k++){
				
				if(this.dra.getDataRaces().get(this.dra.getAtomicBlocks().get(i)).contains(this.dra.getAtomicBlocks().get(k))){
					ret.append("true");
				}else{
					ret.append("false");
				}
				
				if(k!=this.dra.getAtomicBlocks().size()-1){
					ret.append(",");
				}
				
			}

		ret.append("}");

		if(i!=this.dra.getAtomicBlocks().size()-1){
			ret.append(",");
		}

		}
		ret.append("};\n\n");
		ret.append("meta int last_readyprocs=-1;\n");
		ret.append("meta bool no_indep_atomicblocks=false;\n");
		return ret.toString();
		
	}
	
	public String getGlobalPORFunctionDefinitionsDefault() {
		StringBuilder ret = new StringBuilder();
		ret.append("	void recompute_enabled() {\n\n");
		ret.append(" if(last_readyprocs==readyprocs && no_indep_atomicblocks==true){\n\n");
		ret.append("    last_readyprocs=last_readyprocs-1;\n\n");
		ret.append(" }else{\n\n");
		ret.append("		int i;\n\n");
		ret.append("		int j;\n\n");
		ret.append("		int l;\n\n");
		ret.append("    	last_readyprocs=readyprocs-1;\n\n");
		ret.append("			for (i = 0; i < BLOCKCOUNT; i++) {\n\n");
		ret.append("				if(runnable[i]==true){\n\n");
		ret.append("					int k = 0;\n\n");
		ret.append("					bool b = false;\n\n");
		ret.append("					for (k=0; k < BLOCKCOUNT && b==false; k++){\n\n");
		ret.append("						if(runnable[k]==true){\n\n");
		ret.append("							if(dependency[i][k]==true){\n\n");
		ret.append("								b=true;\n\n");
		ret.append("							}\n\n");
		ret.append("						}\n\n");
		ret.append("					}\n\n");
		ret.append("					if(b==false){\n\n");
		ret.append("						for(j=0; j < BLOCKCOUNT; j++){\n\n");
		ret.append("							if(j==i){\n\n");
		ret.append("								enabled[j]=true;\n\n");
		ret.append("							}else{\n\n");
		ret.append("								enabled[j]=false;\n\n");
		ret.append("							}\n\n");
		ret.append("						}\n\n");
		ret.append("						no_indep_atomicblocks=false;\n\n");     
		ret.append("						return;\n\n");
		ret.append("					}\n\n");
		ret.append("				}\n\n");
		ret.append("		}\n\n");
		ret.append("		for(l=0; l < BLOCKCOUNT; l++){\n\n");
		ret.append("			enabled[l]=runnable[l];\n\n");
		ret.append("		}\n\n");
		ret.append("		no_indep_atomicblocks=true;\n\n");
		ret.append("		}\n\n");

		ret.append("	}	\n\n");
		return ret.toString();
	}
	
	public String getGlobalPORVariablesSleepSets(){
		StringBuilder ret = new StringBuilder();
		ret.append("meta bool persistents[BLOCKCOUNT]={");
		for(int i=0; i<this.dra.getAtomicBlocks().size();i++){
			ret.append("false");
			if(i!=this.dra.getAtomicBlocks().size()-1){
				ret.append(",");
			}
		}
		ret.append("};\n\n");
		ret.append("meta bool sleeps[BLOCKCOUNT]={");
		for(int i=0; i<this.dra.getAtomicBlocks().size();i++){
			ret.append("false");
			if(i!=this.dra.getAtomicBlocks().size()-1){
				ret.append(",");
			}
		}
		ret.append("};\n\n");
		ret.append("meta bool next_sleeps[BLOCKCOUNT][BLOCKCOUNT]={");
		
		for(int i=0; i<this.dra.getAtomicBlocks().size();i++){
		ret.append("{");

			for(int k=0; k<this.dra.getAtomicBlocks().size();k++){
				ret.append("false");
				
				if(k!=this.dra.getAtomicBlocks().size()-1){
					ret.append(",");
				}
				
			}
		ret.append("}");

		if(i!=this.dra.getAtomicBlocks().size()-1){
			ret.append(",");
		}

		}
		ret.append("};\n\n");
		ret.append("meta int a,b,c,i,j,k,l,f,g,aA,aB,aC;\n");
		return ret.toString();
		
	}

	public String getGlobalPORFunctionDefinitionsSleepSets() {
		StringBuilder ret = new StringBuilder();
		ret.append("void get_pers(){\n\n");
		ret.append("	for(f=0; f<BLOCKCOUNT; f++){\n\n");
		ret.append("			persistents[f] = false;\n\n");		
		ret.append("	}\n\n");
		ret.append("	for(f=0; f<BLOCKCOUNT; f++){\n\n");
		ret.append("			if(runnable[f]==true){\n\n");
		ret.append("					for(g=0; g<BLOCKCOUNT; g++){\n\n");
		ret.append("						if(runnable[g]==true){\n\n");
		ret.append("							if(!(g>=f)){\n\n");
		ret.append("								if(dependency[f][g]==false){\n\n");
		ret.append("									if(persistents[f]==false){\n\n");
		ret.append("										persistents[g]=true;\n\n");
		ret.append("									}\n\n");
		ret.append("								}else{\n\n");
		ret.append("										persistents[f]=true;\n\n");
		ret.append("										persistents[g]=true;\n\n");
		ret.append("								}\n\n");
		ret.append("							}\n\n");
		ret.append("						}\n\n");
		ret.append("					}\n\n");
		ret.append("				}\n\n");
		ret.append("			}\n\n");
		ret.append("		}\n\n");
		
		ret.append("void recompute_enabled() {\n\n");
		ret.append("    aA = false;\n\n");
		ret.append("    aB = false;\n\n"); 
		ret.append("    aC = false;\n\n");
		ret.append("	for (i = 0; i < BLOCKCOUNT; i++) {\n\n");
		ret.append("		if(runnable[i]==true){\n\n");
		ret.append("            aA = true;\n\n");
		ret.append("			aB = false;\n\n");
		ret.append("			for (k=0; k < BLOCKCOUNT && aB==false; k++){\n\n");
		ret.append("				if(i!=k){\n\n");
		ret.append("					if(runnable[k]==true){\n\n");
		ret.append("							if(dependency[i][k]==true){\n\n");
		ret.append("								aB=true;\n\n");
		ret.append("							}\n\n");
		ret.append("					}\n\n");
		ret.append("				}\n\n");
		ret.append("			}\n\n");
		ret.append("			if(aB==false){\n\n");
		ret.append("				for(j=0; j < BLOCKCOUNT; j++){\n\n");
		ret.append("					if(j==i){\n\n");
		ret.append("						enabled[j]=true;\n\n");
		ret.append("					}else{\n\n");
		ret.append("						enabled[j]=false;\n\n");
		ret.append("					}\n\n");
		ret.append("				}\n\n");
		ret.append("				return;\n\n");
		ret.append("			}\n\n");
		ret.append("		}\n\n");
		ret.append("	}\n\n");
		ret.append("    if(aA==false){\n\n");
		ret.append("        return;\n\n");
		ret.append("    }\n\n");
		ret.append("	get_pers();\n\n");
		ret.append("	for(a=0; a<BLOCKCOUNT; a++){\n\n");
		ret.append("		enabled[a]=persistents[a];\n\n");
		ret.append("		if(sleeps[a]==true){\n\n");
		ret.append("			enabled[a]=false;\n\n");
		ret.append("		}\n\n");
		ret.append("	}\n\n");
		ret.append("    for(a=0; a<BLOCKCOUNT; a++){\n\n");
		ret.append("        if(enabled[a]==true){\n\n");
		ret.append("			aC=true;\n\n");
		ret.append("        }\n\n");
		ret.append("    }\n\n");
		ret.append("    if(aC==false){\n\n");
		ret.append("   		for(a=0; a<BLOCKCOUNT; a++){\n\n");
		ret.append("        	if(persistents[a]==true){\n\n");
		ret.append("		    	enabled[a]=true;\n\n");
		ret.append("       		}\n\n");
		ret.append("    	}\n\n");
		ret.append("		return;\n\n");
		ret.append("    }\n\n");
		ret.append("	for(a=0; a<BLOCKCOUNT; a++){\n\n");
		ret.append("        for(b=0; b<BLOCKCOUNT; b++){\n\n");
		ret.append("			next_sleeps[a][b]=false;\n\n");
		ret.append("        }\n\n");
		ret.append("        sleeps[a]=false;\n\n");
		ret.append("    }\n\n");
		ret.append("	for(b=0; b<BLOCKCOUNT; b++){\n\n");
		ret.append("		if(enabled[b]==true){\n\n");
		ret.append("			for(c=0; c<BLOCKCOUNT; c++){\n\n");
		ret.append("				if(sleeps[c]==true){\n\n");
		ret.append("					if(dependency[b][c]==false){\n\n");
		ret.append("						next_sleeps[b][c]=true;\n\n");
		ret.append("					}\n\n");
		ret.append("				}\n\n");
		ret.append("			}\n\n");
		ret.append("			sleeps[b]=true;\n\n");
		ret.append("		}\n\n");
		ret.append("	}\n\n");
		ret.append("}\n\n");	
		ret.append("void recompute_sleeps(int op){\n\n");
		ret.append("	for(a=0; a<BLOCKCOUNT; a++){\n\n");
		ret.append("		sleeps[a]=next_sleeps[op][a];\n\n");
		ret.append("	}\n\n");
		ret.append("}\n\n");
		ret.append("void reset_sleeps(){\n\n");
		ret.append("    for(i=0; i<BLOCKCOUNT; i++){\n\n");
		ret.append("        sleeps[i]=false;\n\n");
		ret.append("    }\n\n");
		ret.append("}\n\n");
		return ret.toString();
	}
		
	public ArrayList<Pair<Integer,Expression>> getWaitOffsetList() {
		return waitOffsetList;
	}

	public void setWaitOffsetList(ArrayList<Pair<Integer,Expression>> waitOffsetList) {
		this.waitOffsetList = waitOffsetList;
	}

	public int getAtomicBlockCounter() {
		return atomicBlockCounter;
	}

	public void setAtomicBlockCounter(int atomicBlockCounter) {
		this.atomicBlockCounter = atomicBlockCounter;
	}

	public boolean isSleepSet() {
		return sleepSet;
	}

	public void setSleepSet(boolean sleepSet) {
		if(sleepSet){
			this.sleepSet=sleepSet;
			for(AtomicBlock aB : dra.getAtomicBlocks()){
				if(aB.getNotifySCEvent().size()>0){
					this.sleepSet=false;
					return;
				}
			}
		}else{
			this.sleepSet=sleepSet;
		}
	}
	
}
