/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.optimization;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLClassLoader;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.syscir.engine.util.IOUtil;

/**
 * 
 * @author Joachim Fellmuth, Marcel Pockrandt
 * 
 */

public class OptimizationEngine {

	private static Logger logger = LogManager
			.getLogger(OptimizationEngine.class.getName());

	public static final String OPTIMIZERS_CFG_FILE = Constants.CFG_ROOT
			+ "config/optimizers.properties";

	public static void run(TAModel ta) {
		try {
			BufferedReader reader = new BufferedReader(new InputStreamReader(
					IOUtil.getInputStream(OPTIMIZERS_CFG_FILE)));

			while (reader.ready()) {
				String className = reader.readLine();
				if (className == null)
					break;
				className = className.replaceAll("^[ \\s\\t]*", "");
				className = className.replaceAll("[ \\s\\t]*$", "");
				if (className.length() == 0
						|| className.startsWith(Constants.COMMENT))
					continue;

				// now do the magic
				URL[] urls = {};
				URLClassLoader cl = new URLClassLoader(urls);
				try {
					Class<?> clazz = cl.loadClass(className);
					runOptimizer(clazz, ta);
				} catch (ClassNotFoundException e) {
					logger.warn("Can not find optimizer '{}'. Skipping.",
							className);
				} finally {
					// and end the magic
					cl.close();
				}
			}
		} catch (FileNotFoundException e) {
			logger.warn("Can not find optimization cfg file: " + e.getMessage());
			return;
		} catch (IOException e) {
			logger.warn("Can not open optimization cfg file: " + e.getMessage());
			return;
		}
	}

	private static void runOptimizer(Class<?> clazz, TAModel ta) {
		try {
			((Optimizer) clazz.newInstance()).run(ta);
		} catch (InstantiationException e) {
			// might be singleton (enumerator pattern)
			if (clazz.getEnumConstants().length > 0
					&& clazz.getEnumConstants()[0] instanceof Optimizer) {
				((Optimizer) clazz.getEnumConstants()[0]).run(ta);
			} else {
				logger.warn("Can not instantiate optimizer '{}'. Skipping.",
						clazz.getName());
			}
		} catch (IllegalAccessException e) {
			logger.warn("Can not open optimizer '{}'. Skipping.",
					clazz.getName());
		}
	}
}
