/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.engine;

import java.util.LinkedList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.ExpressionConverter;
import de.tub.pes.sc2uppaal.tamodel.SCConstants;
import de.tub.pes.sc2uppaal.tamodel.TAInteger;
import de.tub.pes.sc2uppaal.tamodel.TALocation;
import de.tub.pes.sc2uppaal.tamodel.TATemplate;
import de.tub.pes.sc2uppaal.tamodel.TATransition;
import de.tub.pes.sc2uppaal.tamodel.TAVariable;
import de.tub.pes.sc2uppaal.tamodel.expressions.TARecvExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TASendExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TAVariableAssignmentExpression;
import de.tub.pes.syscir.engine.util.Pair;
import de.tub.pes.syscir.sc_model.SCClass;
import de.tub.pes.syscir.sc_model.SCFunction;
import de.tub.pes.syscir.sc_model.expressions.ArrayAccessExpression;
import de.tub.pes.syscir.sc_model.expressions.BinaryExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableExpression;
import de.tub.pes.syscir.sc_model.expressions.UnaryExpression;
import de.tub.pes.syscir.sc_model.variables.SCArray;
import de.tub.pes.syscir.sc_model.variables.SCSimpleType;

/**
 * Creates logic in TA to implement the SystemC wait statement.
 * <p>
 * 
 * Types of wait:
 * <p>
 * <ul>
 * <li>wait() -> sensitive
 * <li>wait(sc_event) -> event wait
 * <li>wait(sc_event_or_list) -> events divided by '|'
 * <li>wait(sc_event_and_list) -> event separated by '&'
 * <li>wait(sc_time) -> regular timed wait
 * <li>wait(sc_time, sc_event) -> not allowed
 * <li>wait(sc_time, sc_event_or_list) -> not allowed
 * <li>wait(sc_time, sc_event_and_list) -> not allowed
 * <li>wait(double, sc_time_unit) -> regular timed wait
 * <li>wait(double, sc_time_unit, sc_event) -> timed wait and event wait
 * <li>wait(double, sc_time_unit, sc_event_or_list)
 * <li>wait(double, sc_time_unit, sc_event_and_list)
 * </ul>
 * <br>
 * All wait calls with an sc_time object are marked with 'not allowed' in the
 * above list. This is because creation of objects at runtime can not be
 * translated.
 * 
 * @author Joachim Fellmuth, Paula Herber, Marcel Pockrandt, Björn Beckmann
 * 
 */
public class WaitFunctionTransformer {

	private static Logger logger = LogManager
			.getLogger(WaitFunctionTransformer.class.getName());

	// @SuppressWarnings("unchecked")
	public void createFunctionCall(SCClass scclass, TATemplate template,
			TALocation startLoc, TALocation endLoc, SCFunction fun,
			// Stack<Expression> prefixes,
			List<Expression> arguments, Expression expression) {

		// if(prefixes.size() != 0) {
		// reportError(name, "Don't know how to evaluate this wait statement.");
		// return;
		// }

		// fun.setUsesWait(true);

		// prepare the wait
		TALocation waitForEvent = template.createLocation();
		TALocation waitForActivate = template.createUrgentLocation();
		TALocation afterWait = endLoc;
		
		//Partial Order Reduction
		if(template.isPorActive()){
		
			TATransition toWait = new TATransition(startLoc, waitForEvent,
					new TASendExpression(null, Constants.GLOBAL_DEACTIVATE_CHAN));
			toWait.addUpdateExpression(new UnaryExpression(null,
					UnaryExpression.POST, "--", new ConstantExpression(null,
							Constants.GLOBAL_READY_PROCS_INT)));
	
			template.addTransition(toWait);
	
			String currentBlockNumber="";
			
			currentBlockNumber=template.getWaitOffset()+"";

			template.getPor().getWaitOffsetList().add(new Pair<Integer,Expression>(template.getWaitOffset(),expression));

			template.setWaitOffset(template.getWaitOffset()+1);
			
			if(currentBlockNumber.equals("")){
				
				logger.error("Could not find atomic block for expression: "+expression.toString()+scclass.getName());
			}

			TALocation por = template.createUrgentLocation();
			
			TATransition setNewRunnable = new TATransition(waitForActivate, por);
			

			LinkedList<Expression> list = new LinkedList<Expression>();
			list.add(new BinaryExpression(
					null,
					new SCVariableExpression(null,new SCSimpleType("cln","int")),
					"+",				
					new ConstantExpression(null,currentBlockNumber)));

			setNewRunnable.addUpdateExpression(
			 
				new BinaryExpression(
					null,
					new ArrayAccessExpression(
						null,
						new SCArray("runnable",null),
						list
					)
					," = ",new ConstantExpression(null,"true")));
			
			template.addTransition(setNewRunnable);
						
			TALocation porOptim = template.createUrgentLocation();
			
			TATransition porOptimTransitionTrue = new TATransition(porOptim,afterWait);
			porOptimTransitionTrue.setGuard(	new BinaryExpression(
					null,
					new SCVariableExpression(
							null,
							new SCSimpleType("no_indep_atomicblocks","bool")
					)
					," == ",
					new ConstantExpression(null,"true")
				)
			);
			porOptimTransitionTrue.addUpdateExpression(
					 
					new BinaryExpression(
						null,
						new ArrayAccessExpression(
							null,
							new SCArray("enabled",null),
							list
						)
						," = ",new ConstantExpression(null,"false")));
			
			template.addTransition(porOptimTransitionTrue);
			
			TATransition porOptimTransitionFalse = new TATransition(porOptim,afterWait);
			porOptimTransitionFalse.setGuard(	new BinaryExpression(
					null,
					new SCVariableExpression(
							null,
							new SCSimpleType("no_indep_atomicblocks","bool")
					)
					," == ",
					new ConstantExpression(null,"false")
				)
			);
			template.addTransition(porOptimTransitionFalse);

			TATransition activate = new TATransition(por, porOptim,
					new TARecvExpression(null, Constants.GLOBAL_ACTIVATE_CHAN));
			
			activate.setGuard(	new BinaryExpression(
										null,
										new ArrayAccessExpression(
												null,
												new SCArray("enabled",null),
												list
										)
										," == ",
										new ConstantExpression(null,"true")
									)
								);
			
			activate.addUpdateExpression(
					new BinaryExpression(
							null,
							new ArrayAccessExpression(
									null,
									new SCArray("runnable",null),
									list
							)
							," = ",
							new ConstantExpression(null,"false")
						)
			);

			if(template.getPor().isSleepSet()){
				activate.addUpdateExpression(new ConstantExpression(null,"recompute_sleeps(cln+"+currentBlockNumber+")"));
			}
			
			template.addTransition(activate);

		}else{
		
			TATransition toWait = new TATransition(startLoc, waitForEvent,
					new TASendExpression(null, Constants.GLOBAL_DEACTIVATE_CHAN));
			toWait.addUpdateExpression(new UnaryExpression(null,
					UnaryExpression.POST, "--", new ConstantExpression(null,
							Constants.GLOBAL_READY_PROCS_INT)));

			template.addTransition(toWait);

			TATransition activate = new TATransition(waitForActivate, afterWait,
					new TARecvExpression(null, Constants.GLOBAL_ACTIVATE_CHAN));
			template.addTransition(activate);

		}
		// now only the gap between deactivate and activate is left
		Expression eventExp = null;
		boolean timed = false;

		switch (arguments.size()) {
			case 0 :
				// wait without arguments -> static sensitivity
				addStaticSensWait(template, waitForEvent, waitForActivate);
				break;
			case 1 :
				Expression arg = arguments.get(0);
				// wait with one argument -> either wait for one or more
				// event(s)
				// or wait for time with sc_time object, which is handled as int
				String name = Expression.Expr2VarName(arg);
				if (scclass.getMemberByName(name) != null
						|| template.containsLocalVarOrCbvParAsLocVar(name)) {
					TALocation afterNotify = template.createLocation();
					waitForEvent.setCommitted(true);
					addTimedWait(template, waitForEvent, afterNotify,
							waitForActivate, arg);
				} else
					eventExp = arguments.get(0);
				break;
			case 3 :
				eventExp = arguments.get(2);
				// NO BREAK !
			case 2 :
				Expression arg1 = arguments.get(0);
				Expression arg2 = arguments.get(1);

				waitForEvent.setCommitted(true);
				TALocation afterNotify = template.createLocation();
				name = Expression.Expr2VarName(arg1);
				if (scclass.getMemberByName(name) != null
						|| template.containsLocalVarOrCbvParAsLocVar(name)) {
					int factor = SCConstants
							.getTimeFactor(arg2.toStringNoSem());
					Expression waitExpr = (factor == 1)
							? arg1
							: new BinaryExpression(null, arg1, "*",
									new ConstantExpression(null,
											Integer.toString(factor)));
					addTimedWait(template, waitForEvent, afterNotify,
							waitForActivate, waitExpr);
				} else {
					try {
						int time = SCConstants.getTimeUnits(
								arg1.toStringNoSem(), arg2.toStringNoSem());
						addTimedWait(template, waitForEvent, afterNotify,
								waitForActivate, new ConstantExpression(null,
										Integer.toString(time)));
					} catch (Exception e) {
						logger.error("Illegal argument in wait statement.");
					}
				}
				waitForEvent = afterNotify;
				timed = true;
				break;
		}

		if (eventExp != null) {
			if (eventExp.toStringNoSem().indexOf('|') > 0) {
				// wait for one of at least two events
				addEventOrListWait(template, waitForEvent, waitForActivate,
						eventExp, timed);
			} else if (eventExp.toStringNoSem().indexOf('&') > 0) {
				// wait for all of at least two events
				addEventAndListWait(template, waitForEvent, waitForActivate,
						eventExp, timed);
				// TODO
				// } else if(scclass..isEvent(eventExp, sc)) {
			} else if (scclass.getEventByName(eventExp.toStringNoSem()) != null) {
				// wait for exactly one event
				addEventWait(template, waitForEvent, waitForActivate, eventExp,
						timed);
			} else {
				// wait for period of time
				// BUT: time is initialized with sc_time() constructor
				// and that is not allowed in a function call here
				logger.error("Illegal argument in wait statement.");
			}
		}

	}
	private void addStaticSensWait(TATemplate template, TALocation before,
			TALocation after) {
		TATransition trans = new TATransition(before, after,
				new TARecvExpression(null, Constants.LOCAL_SENSITIVE_KEYWORD));
		trans.addUpdateExpression(new UnaryExpression(null,
				UnaryExpression.POST, "++", new ConstantExpression(null,
						Constants.GLOBAL_READY_PROCS_INT)));
		template.addTransition(trans);
	}

	private void addEventWait(TATemplate template, TALocation before,
			TALocation after, Expression event, boolean timed) {
		String evWait = event.toStringNoSem() + Constants.PREFIX_DELIMITER
				+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT;
		if (timed) {
			TALocation gotEvent = template.createLocation();
			gotEvent.setCommitted(true);
			template.addTransition(new TATransition(before, gotEvent,
					new TARecvExpression(null, evWait)));

			TATransition notifyTEvent = new TATransition(
					gotEvent,
					after,
					new TASendExpression(
							null,
							Constants.LOCAL_TIMEOUT_EVENT_KEYWORD
									+ Constants.PREFIX_DELIMITER
									+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM));
			notifyTEvent.addUpdateExpression(new UnaryExpression(null,
					UnaryExpression.POST, "++", new ConstantExpression(null,
							Constants.GLOBAL_READY_PROCS_INT)));
			template.addTransition(notifyTEvent);
		} else {
			TATransition trans = new TATransition(before, after,
					new TARecvExpression(null, evWait));
			trans.addUpdateExpression(new UnaryExpression(null,
					UnaryExpression.POST, "++", new ConstantExpression(null,
							Constants.GLOBAL_READY_PROCS_INT)));
			template.addTransition(trans);
		}
	}

	private void addEventOrListWait(TATemplate template, TALocation before,
			TALocation after, Expression eventList, boolean timed) {
		String evWaitSuffix = Constants.PREFIX_DELIMITER
				+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT;

		List<Expression> events = getEvents(eventList);
		if (timed) {
			TALocation gotEvent = template.createLocation();
			gotEvent.setCommitted(true);
			for (Expression ev : events) {
				template.addTransition(new TATransition(before, gotEvent,
						new TARecvExpression(null, ev.toStringNoSem()
								+ evWaitSuffix)));
			}
			TATransition notifyTEvent = new TATransition(
					gotEvent,
					after,
					new TASendExpression(
							null,
							Constants.LOCAL_TIMEOUT_EVENT_KEYWORD
									+ Constants.PREFIX_DELIMITER
									+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM));
			notifyTEvent.addUpdateExpression(new UnaryExpression(null,
					UnaryExpression.POST, "++", new ConstantExpression(null,
							Constants.GLOBAL_READY_PROCS_INT)));
			template.addTransition(notifyTEvent);
		} else {
			for (Expression ev : events) {
				TATransition trans = new TATransition(before, after,
						new TARecvExpression(null, ev.toStringNoSem()
								+ evWaitSuffix));
				trans.addUpdateExpression(new UnaryExpression(null,
						UnaryExpression.POST, "++", new ConstantExpression(
								null, Constants.GLOBAL_READY_PROCS_INT)));
				template.addTransition(trans);
			}
		}
	}

	private List<Expression> getEvents(Expression eventList) {
		List<Expression> events = new LinkedList<Expression>();
		for (Expression expr : eventList.getInnerExpressions()) {
			if (expr instanceof SCVariableExpression) {
				events.add(expr);
			}
		}
		return events;
	}

	private void addEventAndListWait(TATemplate template, TALocation before,
			TALocation after, Expression eventList, boolean timed) {
		// TODO: sc_event_and_list wait implementation
	}

	private void addTimedWait(TATemplate template, TALocation before,
			TALocation middle, TALocation after, Expression time) {
		String pref = Constants.LOCAL_TIMEOUT_EVENT_KEYWORD
				+ Constants.PREFIX_DELIMITER;
		TATransition notifyt = new TATransition(before, middle,
				new TASendExpression(null, pref
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY));
		TAVariable tavarNotifyt = new TAInteger(pref
				+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME);
		time = ExpressionConverter.convert(time, null);
		notifyt.addUpdateExpression(new TAVariableAssignmentExpression(null,
				tavarNotifyt, time));

		TATransition waitt = new TATransition(middle, after,
				new TARecvExpression(null, pref
						+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT));
		waitt.addUpdateExpression(new UnaryExpression(null,
				UnaryExpression.POST, "++", new ConstantExpression(null,
						Constants.GLOBAL_READY_PROCS_INT)));

		template.addTransition(notifyt);
		template.addTransition(waitt);
	}

}
