/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.engine;

import java.util.LinkedList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.TATemplate;
import de.tub.pes.sc2uppaal.tamodel.TATypedef;
import de.tub.pes.sc2uppaal.tamodel.TAVariable;
import de.tub.pes.sc2uppaal.tamodel.VariableConverter;
import de.tub.pes.syscir.sc_model.SCClass;
import de.tub.pes.syscir.sc_model.SCConnectionInterface;
import de.tub.pes.syscir.sc_model.SCEnumElement;
import de.tub.pes.syscir.sc_model.SCEnumType;
import de.tub.pes.syscir.sc_model.SCFunction;
import de.tub.pes.syscir.sc_model.SCPortInstance;
import de.tub.pes.syscir.sc_model.SCSystem;
import de.tub.pes.syscir.sc_model.SCVariable;
import de.tub.pes.syscir.sc_model.variables.SCClassInstance;

/**
 * Main transformer class for the SCSystem. It handles the transformation of a
 * all SCSystem variables, classes and instances into a TAModel. This transformer is the
 * root of the transformer hierarchy tree.
 * 
 * @author Timm Liebrenz
 * 
 */
public class Transformer extends HierarchicalTransformer {

	private static Logger logger = LogManager.getLogger(Transformer.class
			.getName());

	private final SCSystem scSystem;
	private final TAModel ta;

	/**
	 * List that contains all ClassInstanceTransformer which are used to
	 * transform the model
	 */
	private final List<ClassInstanceTransformer> clInstTransList;

	/**
	 * The instance of the KnownTypeTranformer which should be used for all
	 * KnownTypes of the given model
	 */
	private final KnownTypeTransformer ktTransformer;

	public Transformer(SCSystem scSystem, TAModel ta) {
		super(null, "");

		this.scSystem = scSystem;
		this.ta = ta;
		clInstTransList = new LinkedList<ClassInstanceTransformer>();
		ktTransformer = new KnownTypeTransformer();
	}

	public SCSystem getSCSystem() {
		return scSystem;
	}

	public TAModel getTAModel() {
		return ta;
	}

	public void addClassInstTrans(ClassInstanceTransformer clInstTrans) {
		clInstTransList.add(clInstTrans);
	}

	public KnownTypeTransformer getKnownTypeTransformer() {
		return ktTransformer;
	}

	public void createTAModel() {

		// Enums
		for (SCEnumType et : scSystem.getEnumTypes()) {
			TATypedef td = new TATypedef("int", et.getName());

			for (SCEnumElement ee : et.getElements()) {
				td.addConstant(ee.getName(), ee.getValue());
			}

			ta.addTypedef(td);
		}

		// Global Variables
		for (SCVariable var : scSystem.getGlobalVariables()) {
			TAVariable tavar = VariableConverter.convertVariable(var);
			if (tavar != null) {
				ta.addVariable(tavar);
			}
		}

		// TODO: create only SCPortSCSocketInstances which are created sc_main
		// Global SCPortSCSocketInstances
		for (SCConnectionInterface psi : scSystem.getPortSocketInstances()) {

			if (psi instanceof SCPortInstance) {
				SCPortInstance pi = (SCPortInstance) psi;
				// TODO: can there be more channels?
				if (pi.getChannels().size() > 0) {
					ktTransformer.transformKnownType(pi.getChannels().get(0),
							"", this, this);
				}
			}

			// TODO: handle sockets
		}

		// Global Functions
		for (SCFunction func : scSystem.getGlobalFunctions()) {
			if (func.getName().equals("sc_main")) {
				// don't translate this here
				continue;
			}
			TATemplate template = new TATemplate(func.getSCClass().getName()
					+ Constants.PREFIX_DELIMITER + func.getName());
			ta.addTemplate(template);
			new FunctionTransformer(func, template).createFunction();
			List<TAVariable> vars = FunctionTransformer
					.getNeededTmplParamsFromFunc(func, "");
			template.setParameters(vars);
		}

		// create all ClassInstanceTransfomer
		// creates also all KnownTypes and must be executed before the
		// ClassTemplates are created
		for (SCClassInstance classIns : scSystem.getInstances()) {
			if (classIns.getOuterClass() == null) {
				ClassInstanceTransformer classInsTrans = new ClassInstanceTransformer(
						classIns, "", this, this);
				classInsTrans.createChildClassInstanceTransformers();
			}
		}

		// Classes
		List<SCClass> classes = scSystem.getClasses();
		for (SCClass cl : classes) {
			if (cl.hasInstances()) {
				ClassTransformer classTrans = new ClassTransformer(cl, this);
				classTrans.createClass("", scSystem, ta);
			}
		}

		// Class Instances
		for (HierarchicalTransformer hTrans : this.getChildTransformer()) {
			if (hTrans instanceof ClassInstanceTransformer) {
				ClassInstanceTransformer classInsTrans = (ClassInstanceTransformer) hTrans;
				classInsTrans.createClassInstance(scSystem, ta);
			}
		}
	}
}
