/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.engine;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.TAAddressPointer;
import de.tub.pes.sc2uppaal.tamodel.TAArray;
import de.tub.pes.sc2uppaal.tamodel.TABoolean;
import de.tub.pes.sc2uppaal.tamodel.TAChannel;
import de.tub.pes.sc2uppaal.tamodel.TAClock;
import de.tub.pes.sc2uppaal.tamodel.TAFunction;
import de.tub.pes.sc2uppaal.tamodel.TAInteger;
import de.tub.pes.sc2uppaal.tamodel.TALocation;
import de.tub.pes.sc2uppaal.tamodel.TAMemoryType;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.TATemplate;
import de.tub.pes.sc2uppaal.tamodel.TATemplateInstance;
import de.tub.pes.sc2uppaal.tamodel.TATransition;
import de.tub.pes.sc2uppaal.tamodel.TAVariable;
import de.tub.pes.sc2uppaal.tamodel.expressions.TARecvExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TASendExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TAVariableExpression;
import de.tub.pes.syscir.engine.util.IOUtil;
import de.tub.pes.syscir.sc_model.SCVariable;
import de.tub.pes.syscir.sc_model.expressions.ArrayAccessExpression;
import de.tub.pes.syscir.sc_model.expressions.BinaryExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.FunctionCallExpression;
import de.tub.pes.syscir.sc_model.expressions.RefDerefExpression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableExpression;
import de.tub.pes.syscir.sc_model.expressions.UnaryExpression;
import de.tub.pes.syscir.sc_model.variables.SCArray;
import de.tub.pes.syscir.sc_model.variables.SCPeq;

/**
 * 
 * @author Pfeffer, Timm Liebrenz
 * 
 */
public class PeqTransformer {

	private static final Logger logger = LogManager
			.getLogger(PeqTransformer.class.getName());

	public static final String PEQ_CFG_FILE = Constants.CFG_ROOT
			+ "config/peqs.properties";

	public static final String ERR_OUT_OF_ELEMENTS = "PEQ_ERR_OUT_OF_ELEMENTS";

	private static final String PEQ_PREFIX = "peq_";
	public static final String NOTIFY_TEMPLATE_NAME = PEQ_PREFIX + "notify";
	public static final String FETCH_TEMPLATE_NAME = PEQ_PREFIX + "fetch";
	public static final String ELEMENT_TEMPLATE_NAME = PEQ_PREFIX + "element";

	private static final String NOTIFY_CTRL = Constants.createDelimitedString(
			NOTIFY_TEMPLATE_NAME, "ctrl");
	private static final String FETCH_CTRL = Constants.createDelimitedString(
			FETCH_TEMPLATE_NAME, "ctrl");

	private static final String ELEMENT_CTRL = Constants.createDelimitedString(
			ELEMENT_TEMPLATE_NAME, "ctrl");
	public static final String ELEMENT_PARAM_TRANS = Constants
			.createDelimitedString(ELEMENT_TEMPLATE_NAME, "param", "trans");
	public static final String ELEMENT_PARAM_PHASE = Constants
			.createDelimitedString(ELEMENT_TEMPLATE_NAME, "param", "phase");
	public static final String ELEMENT_PARAM_DELAY = Constants
			.createDelimitedString(ELEMENT_TEMPLATE_NAME, "param", "delay");

	public static final String PEQ_CB_CTRL = Constants.createDelimitedString(
			"peq_cb", "ctrl");
	private static final String PEQ_CB_PARAM_PHASE = Constants
			.createDelimitedString("peq_cb", "param", "phase");
	private static final String PEQ_CB_PARAM_TRANS = Constants
			.createDelimitedString("peq_cb", "param", "tran");

	public static final String SC_PEQ_TEMPLATE_PARAM_ELEMENT = Constants
			.createDelimitedString("peq_element", "ctrl");
	public static final String SC_PEQ_TEMPLATE_PARAM_NOTIFY = Constants
			.createDelimitedString("peq_notify", "ctrl");
	public static final String SC_PEQ_TEMPLATE_PARAM_ELEMENT_TRANS = "peq_element"
			+ Constants.PREFIX_DELIMITER
			+ "param"
			+ Constants.PREFIX_DELIMITER
			+ "trans";
	public static final String SC_PEQ_TEMPLATE_PARAM_ELEMENT_PHASE = "peq_element"
			+ Constants.PREFIX_DELIMITER
			+ "param"
			+ Constants.PREFIX_DELIMITER
			+ "phase";
	public static final String SC_PEQ_TEMPLATE_PARAM_ELEMENT_DELAY = "peq_element"
			+ Constants.PREFIX_DELIMITER
			+ "param"
			+ Constants.PREFIX_DELIMITER
			+ "delay";

	public static final String SC_PEQ_TEMPLATE_PEQ_IN_USE = "peqInUse";

	private static final Map<String, Integer> sizeTable = new HashMap<String, Integer>();
	private static TAModel ta = null;
	private static boolean initialized = false;

	// collection of templates for each PEQ data type
	private static final HashMap<String, TATemplate> notifyTmpls = new HashMap<String, TATemplate>();
	private static final HashMap<String, TATemplate> elementTmpls = new HashMap<String, TATemplate>();
	private static final HashMap<String, TATemplate> fetchTmpls = new HashMap<String, TATemplate>();

	// size for this instance
	private final int size;

	// templates for current data type
	private final TATemplate notifyTmpl;
	private final TATemplate fetchTmpl;
	private final TATemplate elementTmpl;

	// template instances
	private TATemplateInstance notifyInst;
	private TATemplateInstance fetchInst;
	private List<TATemplateInstance> elementInsts;

	private TAVariable notifyCtrlParam;
	TAVariable elementDelayParam;
	TAVariable elementTransParam;
	TAVariable elementPhaseParam;

	private static final HashMap<String, PeqTransformer> transformers = new HashMap<String, PeqTransformer>();

	private final String prefix;
	private final String peqName;

	static public void initialize(TAModel tamodel) {
		if (initialized) {
			logger.warn("PeqTransformer is already initilized");
		}
		if (tamodel == null) {
			logger.error("Initializing PeqTransformer with null model.");
			return;
		}
		// clear the static hash maps
		notifyTmpls.clear();
		elementTmpls.clear();
		fetchTmpls.clear();
		// set ta model
		ta = tamodel;
		// add error case
		TABoolean errCase = new TABoolean(ERR_OUT_OF_ELEMENTS);
		errCase.setInitValue(Boolean.FALSE);
		ta.addVariable(errCase);
		// get sizes
		loadPeqSizes();
		initialized = true;
	}

	static public PeqTransformer getInstance(String prefix, SCPeq peq) {
		// check if initialized
		if (!initialized) {
			logger.error("PeqTransformer class has not been initialized. Please call initialize first.");
			return null;
		}
		// get peq data type
		String peqDataType = peq.getPayloadType();
		// get peq size
		Integer size = Constants.GLOBAL_PEQ_SIZE;
		if (sizeTable.containsKey(prefix + peq.getName())) {
			size = sizeTable.get(prefix + peq.getName());
		}
		// make sure notify templates for this type and size exist
		if (!notifyTmpls.containsKey(peqDataType + "_" + size)) {
			addNotifyTemplate(peqDataType, size);
		}
		// make sure other templates exist for this type
		if (!fetchTmpls.containsKey(peqDataType)) {
			addElementFetchTemplates(peqDataType);
		}
		// get transformer
		PeqTransformer transformer = transformers.get(prefix + peq.getName());
		if (transformer == null) {
			// create new instance
			transformer = new PeqTransformer(prefix, peq.getName(),
					peqDataType, size);
			transformers.put(prefix + peq.getName(), transformer);
		}
		return transformer;
	}

	private PeqTransformer(String prefix, String peqName, String type, int size) {
		// get templates for type
		this.notifyTmpl = notifyTmpls.get(type + "_" + size);
		this.fetchTmpl = fetchTmpls.get(type);
		this.elementTmpl = elementTmpls.get(type);
		// init names
		this.prefix = prefix;
		this.peqName = peqName;
		// init size
		this.size = size;
		// create instances
		addPeqInstances();
		preBindInstances();
	}

	private void addPeqInstances() {
		assert this.notifyTmpl != null;
		assert this.fetchTmpl != null;
		assert this.elementTmpl != null;
		String fullName = prefix + peqName + Constants.PREFIX_DELIMITER;
		notifyInst = notifyTmpl.getInstance(fullName + NOTIFY_TEMPLATE_NAME);
		fetchInst = fetchTmpl.getInstance(fullName + FETCH_TEMPLATE_NAME);
		elementInsts = new ArrayList<TATemplateInstance>(this.size);
		for (int i = 0; i < this.size; i++) {
			elementInsts.add(elementTmpl.getInstance(fullName
					+ ELEMENT_TEMPLATE_NAME + i));
		}
		ta.addTemplateInstance(notifyInst);
		ta.addTemplateInstance(fetchInst);
		for (TATemplateInstance elementInst : elementInsts) {
			ta.addTemplateInstance(elementInst);
		}
	}

	private void preBindInstances() {
		assert notifyInst != null;
		assert elementInsts != null;
		assert fetchInst != null;
		// create parameters
		String fullName = prefix + peqName + Constants.PREFIX_DELIMITER;
		notifyCtrlParam = notifyTmpl.getParameter(NOTIFY_CTRL).createCopy(
				fullName + "notify" + Constants.PREFIX_DELIMITER);
		TAVariable elementArrayParam = notifyTmpl.getParameter(
				Constants.SC_PEQ_TEMPLATE_ELEMENT + Constants.PREFIX_DELIMITER
						+ "ctrl").createCopy(
				fullName + "notify" + Constants.PREFIX_DELIMITER);
		TAVariable peqInUseArrayParam = notifyTmpl.getParameter("peqInUse")
				.createCopy(fullName + "notify" + Constants.PREFIX_DELIMITER);

		ta.addVariable(notifyCtrlParam);
		ta.addVariable(elementArrayParam);
		ta.addVariable(peqInUseArrayParam);

		notifyInst.connectParameter(PeqTransformer.NOTIFY_CTRL,
				notifyCtrlParam.getName());
		notifyInst.connectParameter(Constants.SC_PEQ_TEMPLATE_ELEMENT
				+ Constants.PREFIX_DELIMITER + "ctrl",
				elementArrayParam.getName());
		notifyInst.connectParameter("peqInUse", peqInUseArrayParam.getName());

		TAVariable fetchCtrlParam = null;
		for (TAVariable param : fetchTmpl.getParameters()) {
			if (!param.getName().equals(PeqTransformer.PEQ_CB_CTRL)) {
				TAVariable copy = param.createCopy(fullName + "fetch"
						+ Constants.PREFIX_DELIMITER);
				ta.addVariable(copy);
				fetchInst.connectParameter(param.getName(), copy.getName());
				if (param.getName().equals(FETCH_CTRL)) {
					fetchCtrlParam = copy;
				}
			}
		}

		elementDelayParam = elementTmpl.getParameter(ELEMENT_PARAM_DELAY)
				.createCopy(fullName);
		elementTransParam = elementTmpl.getParameter(ELEMENT_PARAM_TRANS)
				.createCopy(fullName);
		elementPhaseParam = elementTmpl.getParameter(ELEMENT_PARAM_PHASE)
				.createCopy(fullName);
		ta.addVariable(elementDelayParam);
		ta.addVariable(elementTransParam);
		ta.addVariable(elementPhaseParam);
		for (int i = 0; i < this.size; i++) {
			TATemplateInstance elementInst = elementInsts.get(i);
			elementInst.connectParameter(ELEMENT_CTRL,
					elementArrayParam.getName() + "[" + i + "]");
			elementInst.connectParameter(FETCH_CTRL, fetchCtrlParam.getName());
			elementInst.connectParameter(SC_PEQ_TEMPLATE_PEQ_IN_USE,
					peqInUseArrayParam + "[" + i + "]");
			elementInst.connectParameter(ELEMENT_PARAM_DELAY,
					elementDelayParam.getName());
			elementInst.connectParameter(ELEMENT_PARAM_TRANS,
					elementTransParam.getName());
			elementInst.connectParameter(ELEMENT_PARAM_PHASE,
					elementPhaseParam.getName());
		}
	}

	public void connectCallback(String cbName) {
		assert fetchInst != null;
		assert elementInsts != null;
		assert !elementInsts.isEmpty();
		fetchInst.connectParameter(PEQ_CB_CTRL, cbName);
		for (TATemplateInstance elementInst : elementInsts) {
			elementInst.connectParameter(PEQ_CB_PARAM_PHASE,
					cbName.replace("$ctrl", "$param$phase"));
			elementInst.connectParameter(PEQ_CB_PARAM_TRANS,
					cbName.replace("$ctrl", "$param$tran"));
		}
	}

	public void addParamterConnections(HashMap<String, String> params) {
		// notify ctrl
		params.put(Constants.createDelimitedString(peqName, "notify$ctrl"),
				notifyCtrlParam.getName());
		// element params
		params.put(Constants.createDelimitedString(peqName,
				PeqTransformer.ELEMENT_PARAM_DELAY), elementDelayParam
				.getName());
		params.put(Constants.createDelimitedString(peqName,
				PeqTransformer.ELEMENT_PARAM_PHASE), elementPhaseParam
				.getName());
		params.put(Constants.createDelimitedString(peqName,
				PeqTransformer.ELEMENT_PARAM_TRANS), elementTransParam
				.getName());

	}

	static public Collection<TAVariable> getConnectionParamters(SCPeq peq) {
		Collection<TAVariable> parameters = new ArrayList<TAVariable>(4);
		// notify ctrl
		TAVariable tavar = new TAChannel(Constants.createDelimitedString(
				peq.getName(), "notify$ctrl"));
		parameters.add(tavar);
		// element params
		tavar = new TAInteger(Constants.createDelimitedString(peq.getName(),
				PeqTransformer.ELEMENT_PARAM_DELAY));
		parameters.add(tavar);
		tavar = new TAAddressPointer(Constants.createDelimitedString(
				peq.getName(), PeqTransformer.ELEMENT_PARAM_PHASE),
				peq.getPhaseType());
		parameters.add(tavar);
		tavar = new TAAddressPointer(Constants.createDelimitedString(
				peq.getName(), PeqTransformer.ELEMENT_PARAM_TRANS),
				peq.getPayloadType());
		parameters.add(tavar);
		return parameters;
	}

	static public TALocation createNotifyEvent(TALocation currentLoc,
			TATemplate tmpl, SCPeq peq, List<Expression> parameters) {
		String prefix = peq.getName();
		Expression transParam = parameters.get(0);
		Expression phaseParam = parameters.get(1);
		Expression delayParam = parameters.get(2);

		// deref
		transParam = transParam instanceof RefDerefExpression ? ((RefDerefExpression) transParam)
				.getExpression() : transParam;
		phaseParam = phaseParam instanceof RefDerefExpression ? ((RefDerefExpression) phaseParam)
				.getExpression() : phaseParam;
		delayParam = delayParam instanceof RefDerefExpression ? ((RefDerefExpression) delayParam)
				.getExpression() : delayParam;
		if (delayParam instanceof FunctionCallExpression
				&& ((FunctionCallExpression) delayParam).getFunction()
						.getName().equalsIgnoreCase("sc_time")) {
			delayParam = ((FunctionCallExpression) delayParam).getParameters()
					.get(0);
		} else if (delayParam instanceof SCVariableExpression) {
			SCVariable delayVar = ((SCVariableExpression) delayParam).getVar();
			// TODO: the hardcoded "int" is not so nice, convert sc_time instead
			delayParam = new ArrayAccessExpression(null, new SCArray(
					TAMemoryType.getMemArrName("int"), delayVar.getName()),
					new ConstantExpression(null, delayVar.getName()));
		}

		// XXX: ugly!
		TAChannel chan = new TAChannel(prefix + Constants.PREFIX_DELIMITER
				+ "notify$ctrl");
		TALocation calledLoc = tmpl.createUrgentLocation();
		TALocation newLoc = tmpl.createUrgentLocation();

		TATransition trans = new TATransition(currentLoc, calledLoc,
				new TASendExpression(null, chan.getName()));
		trans.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, Constants.createDelimitedString(
						prefix, PeqTransformer.ELEMENT_PARAM_DELAY)), "=", delayParam));
		trans.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, Constants.createDelimitedString(
						prefix, PeqTransformer.ELEMENT_PARAM_TRANS)), "=",
				transParam));
		trans.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, Constants.createDelimitedString(
						prefix, PeqTransformer.ELEMENT_PARAM_PHASE)), "=",
				phaseParam));
		tmpl.addTransition(trans);

		TATransition returnTrans = new TATransition(calledLoc, newLoc,
				new TARecvExpression(null, chan.getName()));
		tmpl.addTransition(returnTrans);

		return newLoc;
	}

	static private void addNotifyTemplate(String type, Integer size) {
		TATemplate notifyTmpl = generateNotifyTemplate(type, size);
		notifyTmpls.put(type + "_" + size, notifyTmpl);
		ta.addTemplate(notifyTmpl);
	}

	static private void addElementFetchTemplates(String type) {
		// generate typed templates
		TATemplate fetchTmpl = generateFetchTemplate(type);
		TATemplate elementTmpl = generateElementTemplate(type);
		// add to templates collection
		fetchTmpls.put(type, fetchTmpl);
		elementTmpls.put(type, elementTmpl);
		// add to ta model
		ta.addTemplate(fetchTmpl);
		ta.addTemplate(elementTmpl);
	}

	static private TATemplate generateNotifyTemplate(String type, Integer size) {
		TATemplate notifyTemplate = new TATemplate(
				Constants.createDelimitedString(NOTIFY_TEMPLATE_NAME, type,
						size.toString()));

		TAChannel ctrl = new TAChannel(NOTIFY_CTRL);

		TAArray notifyArr = new TAArray(Constants.SC_PEQ_TEMPLATE_ELEMENT
				+ Constants.PREFIX_DELIMITER + "ctrl", "chan", false,
				new ConstantExpression(null, Integer.toString(size)));

		TAArray peqInUse = new TAArray("peqInUse", "bool", false,
				new ConstantExpression(null, Integer.toString(size)));
		peqInUse.setInitValue(new ConstantExpression(null, ""));

		notifyTemplate.addParameter(ctrl);
		notifyTemplate.addParameter(notifyArr);
		notifyTemplate.addParameter(peqInUse);

		TAInteger index = new TAInteger("index");
		index.setMin("-1");
		index.setMax(Integer.toString(size - 1));
		index.setInitValue(new ConstantExpression(null, "-1"));
		notifyTemplate.addLocalVar(index);

		TAInteger i = new TAInteger("i");
		i.setMin("-1");
		i.setMax(Integer.toString(size));
		i.setInitValue(new ConstantExpression(null, "0"));
		notifyTemplate.addLocalVar(i);

		TAFunction getIndex = new TAFunction.Builder("getIndex", "int[-1, "
				+ (size - 1) + "]").body(
				new ConstantExpression(null, "\tfor (" + i.getName() + " = 0; "
						+ i.getName() + " < " + size + "; " + i.getName()
						+ "++) {\n\t\tif (!" + peqInUse.getName() + "["
						+ i.getName() + "]) {\n\t\t\treturn " + i.getName()
						+ ";\n\t\t}\n\t}\n\treturn -1;")).build();
		notifyTemplate.addLocalFun(getIndex);

		TALocation initialLocation = new TALocation();
		TALocation loc1 = new TALocation("invoked");
		loc1.setUrgent(true);
		TALocation loc2 = new TALocation();
		loc2.setUrgent(true);
		TALocation fail = new TALocation();

		TATransition tran1 = new TATransition(initialLocation, loc1);
		tran1.setSync(new TARecvExpression(null, ctrl.getName()));
		tran1.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, index.getName()), "=",
				new ConstantExpression(null, getIndex.getName() + "()")));

		TATransition tran2 = new TATransition(loc1, loc2);
		tran2.setSync(new TASendExpression(null, notifyArr.getName() + "["
				+ index.getName() + "]"));
		tran2.setGuard(new ConstantExpression(null, "index >= 0"));

		TATransition errt = new TATransition(loc1, fail);
		errt.setGuard(new ConstantExpression(null, "index < 0 && !"
				+ ERR_OUT_OF_ELEMENTS));
		errt.addUpdateExpression(new ConstantExpression(null,
				ERR_OUT_OF_ELEMENTS + " = true"));

		TATransition tran3 = new TATransition(loc2, initialLocation);
		tran3.setSync(new TASendExpression(null, ctrl.getName()));
		tran3.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, index.toString()), "=",
				new ConstantExpression(null, "-1")));
		tran3.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, i.getName()), "=",
				new ConstantExpression(null, "0")));

		notifyTemplate.addLocations(initialLocation, loc1, loc2, fail);
		notifyTemplate.addTransitions(tran1, tran2, tran3, errt);
		notifyTemplate.setInitLocation(initialLocation);

		return notifyTemplate;
	}

	static private TATemplate generateElementTemplate(String type) {
		TATemplate elementTemplate = new TATemplate(
				Constants.createDelimitedString(ELEMENT_TEMPLATE_NAME, type));

		TABoolean peqInUse = new TABoolean(SC_PEQ_TEMPLATE_PEQ_IN_USE);

		// the parameters of the notify-function
		elementTemplate.addParameter(new TAChannel(ELEMENT_CTRL));
		TAAddressPointer transparam = new TAAddressPointer(ELEMENT_PARAM_TRANS,
				type);
		elementTemplate.addParameter(transparam);
		TAAddressPointer phaseparam = new TAAddressPointer(ELEMENT_PARAM_PHASE,
				"int");
		elementTemplate.addParameter(phaseparam);

		elementTemplate.addParameter(new TAInteger(ELEMENT_PARAM_DELAY));

		// parameters for the callback function
		elementTemplate.addParameter(new TAChannel(FETCH_CTRL));
		TAVariable fetchtransparam = new TAAddressPointer(PEQ_CB_PARAM_TRANS,
				type);
		elementTemplate.addParameter(fetchtransparam);
		TAVariable fetchphaseparam = new TAAddressPointer(PEQ_CB_PARAM_PHASE,
				"int");
		elementTemplate.addParameter(fetchphaseparam);

		elementTemplate.addParameter(peqInUse);

		TAAddressPointer locPhase = new TAAddressPointer("phase", "int");
		TAInteger locDelay = new TAInteger("delay");
		TAClock locClock = new TAClock("c");
		TAAddressPointer locPayload = new TAAddressPointer("payload", type);

		// do not reset these variables
		locPhase.setVolatile(true);
		locPayload.setVolatile(true);

		elementTemplate.addLocalVar(locPhase);
		elementTemplate.addLocalVar(locDelay);
		elementTemplate.addLocalVar(locClock);
		elementTemplate.addLocalVar(locPayload);

		TALocation initLocation = new TALocation();
		initLocation.setName("init_location");

		TALocation active = elementTemplate.createCommittedLocation();
		TALocation payloadInit = elementTemplate.createCommittedLocation();
		TALocation phaseInit = elementTemplate.createCommittedLocation();

		TALocation loc1 = new TALocation();
		loc1.setInvariant(new BinaryExpression(null, new TAVariableExpression(
				null, locClock), "<=", new TAVariableExpression(null, locDelay)));

		TALocation loc2 = new TALocation();
		loc2.setCommitted(true);

		TALocation loc3 = new TALocation();
		loc3.setCommitted(true);

		TALocation loc4 = new TALocation();
		loc4.setUrgent(true);

		TALocation loc5 = new TALocation();
		loc5.setUrgent(true);

		TATransition activateTrans = new TATransition(initLocation, active);
		TATransition payloadNotNullTrans = new TATransition(active, payloadInit);
		TATransition payloadNullTrans = new TATransition(active, payloadInit);
		TATransition phaseNotNullTrans = new TATransition(payloadInit,
				phaseInit);
		TATransition phaseNullTrans = new TATransition(payloadInit, phaseInit);

		payloadNotNullTrans.setGuard(new BinaryExpression(null,
				new ConstantExpression(null, locPayload.getName()), "!=",
				new ConstantExpression(null, "NULL")));
		payloadNullTrans.setGuard(new BinaryExpression(null,
				new ConstantExpression(null, locPayload.getName()), "==",
				new ConstantExpression(null, "NULL")));
		payloadNullTrans.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, locPayload.getName()), "=",
				new ConstantExpression(null, "allocate_dynamic_"
						+ locPayload.getMemType() + "()")));
		phaseNotNullTrans.setGuard(new BinaryExpression(null,
				new ConstantExpression(null, locPhase.getName()), "!=",
				new ConstantExpression(null, "NULL")));
		phaseNullTrans.setGuard(new BinaryExpression(null,
				new ConstantExpression(null, locPhase.getName()), "==",
				new ConstantExpression(null, "NULL")));
		phaseNullTrans.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, locPhase.getName()), "=",
				new ConstantExpression(null, "allocate_dynamic_"
						+ locPhase.getMemType() + "()")));
		activateTrans.setSync(new TARecvExpression(null, ELEMENT_TEMPLATE_NAME
				+ Constants.PREFIX_DELIMITER + "ctrl"));

		TATransition trans1 = new TATransition(phaseInit, loc1);
		trans1.addUpdateExpression(new BinaryExpression(null,
				new ArrayAccessExpression(null, new SCArray(TAMemoryType
						.getMemArrName(type), type), new ConstantExpression(
						null, locPayload.getName())), "=",
				new ArrayAccessExpression(null, new SCArray(TAMemoryType
						.getMemArrName(type), type), new ConstantExpression(
						null, ELEMENT_PARAM_TRANS))));
		trans1.addUpdateExpression(new BinaryExpression(null,
				new ArrayAccessExpression(null, new SCArray(TAMemoryType
						.getMemArrName("int"), "int"), new ConstantExpression(
						null, locPhase.getName())), "=",
				new ArrayAccessExpression(null, new SCArray(TAMemoryType
						.getMemArrName("int"), "int"), new ConstantExpression(
						null, ELEMENT_PARAM_PHASE))));
		trans1.addUpdateExpression(new BinaryExpression(null,
				new TAVariableExpression(null, locDelay), "=",
				/*
				 * new ArrayAccessExpression(null, new SCArray(TAMemoryType
				 * .getMemArrName("int"), "int"), new ConstantExpression( null,
				 * ELEMENT_PARAM_DELAY))));
				 */
				new ConstantExpression(null, ELEMENT_PARAM_DELAY)));
		trans1.addUpdateExpression(new BinaryExpression(null,
				new TAVariableExpression(null, peqInUse), "=",
				new ConstantExpression(null, "true")));
		trans1.addUpdateExpression(new BinaryExpression(null,
				new TAVariableExpression(null, locClock), "=",
				new ConstantExpression(null, "0")));

		TATransition trans2 = new TATransition(loc1, loc2);
		trans2.setSync(new TARecvExpression(null,
				Constants.GLOBAL_ADVANCE_TIME_CHAN));
		trans2.setGuard(new BinaryExpression(null, new TAVariableExpression(
				null, locDelay), "!=", new ConstantExpression(null, "0")));

		TATransition trans3 = new TATransition(loc2, loc1);
		trans3.setGuard(new BinaryExpression(null, new TAVariableExpression(
				null, locClock), "<", new TAVariableExpression(null, locDelay)));

		TATransition trans4 = new TATransition(loc2, loc3);
		trans4.setGuard(new BinaryExpression(null, new TAVariableExpression(
				null, locClock), "==", new TAVariableExpression(null, locDelay)));

		TATransition trans5 = new TATransition(loc1, loc3);
		trans5.setSync(new TARecvExpression(null,
				Constants.GLOBAL_DELTA_DELAY_CHAN));
		trans5.setGuard(new BinaryExpression(null, new TAVariableExpression(
				null, locDelay), "==", new ConstantExpression(null, "0")));

		TATransition trans6 = new TATransition(loc1, loc3);
		trans6.setSync(new TASendExpression(null,
				Constants.GLOBAL_ADVANCE_TIME_CHAN));
		trans6.setGuard(new BinaryExpression(null, new BinaryExpression(null,
				new TAVariableExpression(null, locDelay), "!=",
				new ConstantExpression(null, "0")), "&&", new BinaryExpression(
				null, new TAVariableExpression(null, locClock), "==",
				new TAVariableExpression(null, locDelay))));

		TATransition trans7 = new TATransition(loc3, loc4);
		trans7.addUpdateExpression(new UnaryExpression(null,
				UnaryExpression.POST, "++", new ConstantExpression(null,
						Constants.GLOBAL_READY_PROCS_INT)));

		TATransition trans8 = new TATransition(loc4, loc5);
		trans8.setSync(new TARecvExpression(null,
				Constants.GLOBAL_ACTIVATE_CHAN));

		TATransition trans9 = new TATransition(loc5, initLocation);
		trans9.setSync(new TASendExpression(null, FETCH_CTRL));

		trans9.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, PEQ_CB_PARAM_TRANS), "=",
				new TAVariableExpression(null, locPayload)));
		trans9.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, PEQ_CB_PARAM_PHASE), "=",
				new TAVariableExpression(null, locPhase)));
		trans9.addUpdateExpression(new BinaryExpression(null,
				new TAVariableExpression(null, peqInUse), "=",
				new ConstantExpression(null, "false")));

		elementTemplate.addLocation(initLocation);
		elementTemplate.addLocation(loc1);
		elementTemplate.addLocation(loc2);
		elementTemplate.addLocation(loc3);
		elementTemplate.addLocation(loc4);
		elementTemplate.addLocation(loc5);

		elementTemplate.addTransition(activateTrans);
		elementTemplate.addTransition(payloadNotNullTrans);
		elementTemplate.addTransition(payloadNullTrans);
		elementTemplate.addTransition(phaseNotNullTrans);
		elementTemplate.addTransition(phaseNullTrans);

		elementTemplate.addTransition(trans1);
		elementTemplate.addTransition(trans2);
		elementTemplate.addTransition(trans3);
		elementTemplate.addTransition(trans4);
		elementTemplate.addTransition(trans5);
		elementTemplate.addTransition(trans6);
		elementTemplate.addTransition(trans7);
		elementTemplate.addTransition(trans8);
		elementTemplate.addTransition(trans9);

		elementTemplate.setInitLocation(initLocation);
		return elementTemplate;
	}

	static private TATemplate generateFetchTemplate(String type) {
		// template creation
		TATemplate fetchTemplate = new TATemplate(
				Constants.createDelimitedString(FETCH_TEMPLATE_NAME, type));

		// parameters
		fetchTemplate.addParameter(new TAChannel(FETCH_CTRL));

		// parameters for the callback function
		fetchTemplate.addParameter(new TAChannel(PEQ_CB_CTRL));
		TAAddressPointer cbtransparam = new TAAddressPointer(
				PEQ_CB_PARAM_TRANS, type);
		fetchTemplate.addParameter(cbtransparam);
		TAAddressPointer cbphaseparam = new TAAddressPointer(
				PEQ_CB_PARAM_PHASE, "int");
		fetchTemplate.addParameter(cbphaseparam);

		// locations
		TALocation initLocation = fetchTemplate.createInitLocation();
		TALocation activeLoc = fetchTemplate.createUrgentLocation();
		TALocation calledLoc = fetchTemplate.createStandardLocation();
		TALocation returnLoc = fetchTemplate.createUrgentLocation();

		// transitions
		TATransition activateTrans = new TATransition(initLocation, activeLoc);
		TATransition callTrans = new TATransition(activeLoc, calledLoc);
		TATransition returnTrans = new TATransition(calledLoc, returnLoc);
		TATransition deactivateTrans = new TATransition(returnLoc, initLocation);

		// transition design
		activateTrans.setSync(new TARecvExpression(null, FETCH_CTRL));
		callTrans.setSync(new TASendExpression(null, PEQ_CB_CTRL));
		returnTrans.setSync(new TARecvExpression(null, PEQ_CB_CTRL));
		deactivateTrans.setSync(new TASendExpression(null,
				Constants.GLOBAL_DEACTIVATE_CHAN));
		deactivateTrans.addUpdateExpression(new UnaryExpression(null,
				UnaryExpression.POST, "--", new ConstantExpression(null,
						Constants.GLOBAL_READY_PROCS_INT)));

		// build
		fetchTemplate.addTransitions(activateTrans, callTrans, returnTrans,
				deactivateTrans);
		return fetchTemplate;
	}

	static private void loadPeqSizes() {
		try {
			BufferedReader reader = new BufferedReader(new InputStreamReader(
					IOUtil.getInputStream(PEQ_CFG_FILE)));
			sizeTable.putAll(buildTableFromCfg(reader));
			checkSizes();
		} catch (FileNotFoundException e) {
			logger.warn("Can't find peq cfg file. Skipping.");
			return;
		} catch (IOException e) {
			logger.warn("Error in config file: " + e.getMessage());
			return;
		}
	}

	static private Map<String, Integer> buildTableFromCfg(BufferedReader reader)
			throws IOException {
		Map<String, Integer> sizes = new HashMap<String, Integer>();
		while (reader.ready()) {
			String line = reader.readLine();
			if (line == null)
				break;
			line = line.intern();
			if (line.startsWith(Constants.COMMENT)) {
				continue;
			} else {
				String[] entry = line.split(" ");
				int size = Integer.parseInt(entry[1]);
				if (size <= 0) {
					throw new IOException("PEQ size is <= 0.");
				}
				if (entry[0].equalsIgnoreCase("global")) {
					Constants.GLOBAL_PEQ_SIZE = size;
				} else {
					sizes.put(entry[0], size);
				}
			}
		}
		return sizes;
	}

	static private void checkSizes() {
		if (!sizeTable.isEmpty()) {
			if (Collections.min(sizeTable.values()) > Constants.GLOBAL_PEQ_SIZE) {
				logger.warn("Global PEQ size is not the highest PEQ size. This could cause errors.");
			}
		}
	}

	static public int getPeqSize(String fullName) {
		if (!initialized) {
			logger.error("PeqTransformer is not initialized. Please call initilize first!");
			throw new IllegalStateException("PeqTransformer not initialized.");
		}
		if (sizeTable.containsKey(fullName)) {
			return sizeTable.get(fullName);
		} else {
			return Constants.GLOBAL_PEQ_SIZE;
		}
	}
}
