/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.engine;

import java.util.LinkedList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.syscir.sc_model.SCClass;
import de.tub.pes.syscir.sc_model.variables.SCKnownType;

/**
 * Transformer that handles SystemC variables which have a known type.
 * KnownTypes are handled as classes and variables of a KnownType are handled
 * like class instances. Therefore for each KnownType variable a new
 * ClassInstanceTransformer is created.
 * 
 * @author Timm Liebrenz
 * 
 */
public class KnownTypeTransformer {

	private static Logger logger = LogManager
			.getLogger(KnownTypeTransformer.class.getName());

	/**
	 * List that contains all SCClasses which were already transformed once with
	 * the KnownTypeTransformer
	 */
	private List<SCClass> knownTypeClassList;

	/**
	 * List that contains the names of the instances which were created with the
	 * KnownTypeTransformer
	 */
	private List<String> knownTypeInstList;

	public KnownTypeTransformer() {
		knownTypeClassList = new LinkedList<SCClass>();
		knownTypeInstList = new LinkedList<String>();
	}

	/**
	 * Creates a ClassInstanceTransformer for a given instance of a KnownType.
	 * Creates the Template of the appropriate SCClass if necessary. If this
	 * functions was already called before for this instance it does nothing.
	 * nothing.
	 * 
	 * @param ktInst
	 * @param prefix
	 * @param mainTransformer
	 * @param outerClInstTrans
	 */
	public void transformKnownType(SCKnownType ktInst, String prefix,
			Transformer mainTransformer,
			HierarchicalTransformer outerClInstTrans) {
		// check if the instance was already transformed before
		// TODO: is there a better way to create each instance only once?
		if (knownTypeInstList.contains(prefix + ktInst.getName())) {
			return;
		}

		// create the Template for the SCClass
		createKnownTypeTemplate(ktInst.getSCClass(), "", mainTransformer);

		// create the ClassInstanceTransformer for the given instance and all of
		// its child classInstances
		ClassInstanceTransformer modInsTrans = new ClassInstanceTransformer(
				ktInst, prefix, mainTransformer, outerClInstTrans);
		modInsTrans.createChildClassInstanceTransformers();

		knownTypeInstList.add(prefix + ktInst.getName());
	}

	/**
	 * Creates all Templates for the given KnownType. If the given Templates
	 * were already created the function does nothing.
	 * 
	 * @param ktclass
	 * @param prefix
	 * @param mainTransformer
	 */
	public void createKnownTypeTemplate(SCClass ktclass, String prefix,
			Transformer mainTransformer) {
		if (!knownTypeClassList.contains(ktclass)) {
			ClassTransformer classTrans = new ClassTransformer(ktclass,
					mainTransformer);
			classTrans.createClass(prefix, mainTransformer.getSCSystem(),
					mainTransformer.getTAModel());

			knownTypeClassList.add(ktclass);
		}
	}

	/**
	 * Returns the SCClass of a KnownType for a given String. Only returns the
	 * SCClass if the KnownType was already transformed otherwise it returns
	 * null.
	 * 
	 * @param name
	 * @return
	 */
	public SCClass getKnownTypeClassByName(String name) {
		for (SCClass scclass : knownTypeClassList) {
			// TODO: make this better; "contains" is used and not "equals" to
			// get a class for sc_fifo with custom buffer size (name is sc_fifo,
			// but class name is sc_fifo__x)
			if (scclass.getName().contains(name)) {
				return scclass;
			}
		}
		return null;
	}
}
