/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.engine;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.ExpressionConverter;
import de.tub.pes.sc2uppaal.tamodel.SCConstants;
import de.tub.pes.sc2uppaal.tamodel.TAChannel;
import de.tub.pes.sc2uppaal.tamodel.TALocation;
import de.tub.pes.sc2uppaal.tamodel.TAMemoryType;
import de.tub.pes.sc2uppaal.tamodel.TATemplate;
import de.tub.pes.sc2uppaal.tamodel.TATransition;
import de.tub.pes.sc2uppaal.tamodel.TAVariable;
import de.tub.pes.sc2uppaal.tamodel.VariableConverter;
import de.tub.pes.sc2uppaal.tamodel.expressions.TAAccessExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TAFunctionCallExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TARecvExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TASendExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TAVariableAssignmentExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TAVariableExpression;
import de.tub.pes.syscir.analysis.VariableScopeAnalyzer;
import de.tub.pes.syscir.engine.util.Pair;
import de.tub.pes.syscir.sc_model.SCFunction;
import de.tub.pes.syscir.sc_model.SCParameter;
import de.tub.pes.syscir.sc_model.SCVariable;
import de.tub.pes.syscir.sc_model.expressions.AccessExpression;
import de.tub.pes.syscir.sc_model.expressions.AssertionExpression;
import de.tub.pes.syscir.sc_model.expressions.BinaryExpression;
import de.tub.pes.syscir.sc_model.expressions.BracketExpression;
import de.tub.pes.syscir.sc_model.expressions.BreakExpression;
import de.tub.pes.syscir.sc_model.expressions.CaseExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.ContinueExpression;
import de.tub.pes.syscir.sc_model.expressions.DoWhileLoopExpression;
import de.tub.pes.syscir.sc_model.expressions.EventNotificationExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.ExpressionBlock;
import de.tub.pes.syscir.sc_model.expressions.ForLoopExpression;
import de.tub.pes.syscir.sc_model.expressions.FunctionCallExpression;
import de.tub.pes.syscir.sc_model.expressions.IfElseExpression;
import de.tub.pes.syscir.sc_model.expressions.OutputExpression;
import de.tub.pes.syscir.sc_model.expressions.QuestionmarkExpression;
import de.tub.pes.syscir.sc_model.expressions.ReturnExpression;
import de.tub.pes.syscir.sc_model.expressions.SCStopExpression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableExpression;
import de.tub.pes.syscir.sc_model.expressions.SwitchExpression;
import de.tub.pes.syscir.sc_model.expressions.UnaryExpression;
import de.tub.pes.syscir.sc_model.expressions.WhileLoopExpression;
import de.tub.pes.syscir.sc_model.variables.SCPeq;

/**
 * 
 * The FunctionTransformer class creates a TATemplate for a given function.
 * 
 * @author Timm Liebrenz, rschroeder
 * 
 */
public class FunctionTransformer {

	private static Logger logger = LogManager
			.getLogger(FunctionTransformer.class.getName());

	private final SCFunction func;
	private final TATemplate tmpl;

	// stores every created template
	private static HashMap<String, TATemplate> tmplByFunc = new HashMap<String, TATemplate>();

	/**
	 * Need to remember which vars we need to remove from the stack afterwards.
	 * But: there may be var decl inside loops/if-else-scopes. So we build some
	 * sort of stack of scopes and some helperfunctions around it. We use the
	 * barrierIndices to remember when we actually need to delete vars from
	 * multiple scopes, e.g. break/ continue. Think of all the following
	 * combinations we need to assure here: void f(...) { int h; for(...) { ...
	 * int i; if(...) int j; break else ... int k; if (...) break else int l;
	 * return; } // end for int m; return; } See testcase dyn_mem/break for some
	 * examples. Also, try to invent a better data structure :)
	 */
	private final Stack<Map<TAVariable, SCVariable>> scopeStack = new Stack<Map<TAVariable, SCVariable>>();
	private final Stack<Integer> barrierIndices = new Stack<Integer>();
	private static final VariableScopeAnalyzer vsa = new VariableScopeAnalyzer();

	/**
	 * Creates a new FunctionTransformer instance.
	 * 
	 * @param func
	 *            the Function which should be transformed
	 * @param template
	 *            an empty template, in which the functionality should be
	 *            written
	 */
	public FunctionTransformer(SCFunction func, TATemplate template) {
		this.func = func;
		this.tmpl = template;
		template.setFunction(func);
	}

	private static String getKey(SCFunction func) {
		StringBuffer key = new StringBuffer();
		if (func.getSCClass() != null) {
			key.append(func.getSCClass().getName());
		}
		key.append(Constants.PREFIX_DELIMITER);
		key.append(func.getName());
		return key.toString();
	}

	/**
	 * Returns the template to a given function. Returns null if no template was
	 * created with this function.
	 * 
	 * @param func
	 *            function to which a template should be found
	 * @return the template to the function func
	 */
	public static TATemplate getTemplate(SCFunction func) {
		TATemplate temp = tmplByFunc.get(getKey(func));
		if (temp == null) {
			logger.error("Could not find template for function {}",
					func.getName());
		}
		return temp;
	}

	private static void putTemplate(SCFunction func, TATemplate tmpl) {
		tmplByFunc.put(getKey(func), tmpl);
	}

	/**
	 * Creates the function in a new template.
	 */
	public void createFunction() {
		TALocation initLoc = tmpl.createInitLocation();
		// create ctrl transition
		TALocation startLoc = tmpl.createUrgentLocation();
		TATransition getCtrl = new TATransition(initLoc, startLoc,
				new TARecvExpression(null, getCtrlChanName(func)));
		tmpl.addTransition(getCtrl);

		// Create a 'virtual' scope above all others.
		// It'll include only the parameters.
		newScope();
		// create local var for each parameter
		for (SCParameter par : func.getParameters()) {
			SCVariable scvar = par.getVar();
			TAVariable tmpTAVar = VariableConverter.convertVariable(scvar);
			Expression update = null;
			if (VariableConverter.useMemModel(scvar)) {
				// alloc on local stack for call by value params in mem model
				update = ExpressionConverter.getAllocExprForParAsLocVar(par,
						func);
				tmpl.addCbvParsAsLocalVar(tmpTAVar);
			} else {
				// all params not in mem (either not referenced [convert to
				// normal tavar] or pointers[convert to addr ptr]) get the
				// value of the tmpl param assigned
				tmpl.addLocalVar(tmpTAVar);
				String value = VariableConverter.getTemplParamName(
						func.getName(), scvar.getName());
				update = new TAVariableAssignmentExpression(null, tmpTAVar,
						value);
			}
			getCtrl.addUpdateExpression(update);
			addToScope(par.getVar(), tmpTAVar.getName());
		}

		// create local variables
		for (SCVariable scvar : func.getLocalVariables()) {
			TAVariable tmpTAVar = null;
			tmpTAVar = VariableConverter.convertVariable(scvar);
			if (tmpTAVar != null) {
				tmpl.addLocalVar(tmpTAVar);
			}
		}

		// TODO: review if we can't use tmpl.getUrgentLocation() (which adds
		// this template)
		TALocation endLoc = new TALocation();
		endLoc.setUrgent(true);

		// create the body
		createExpressionBody(func.getBody(), startLoc, endLoc, null, null);

		// check if function loop is closed
		if (tmpl.getLocations().contains(endLoc)) {
			TATransition retCtrl = new TATransition(endLoc, initLoc,
					new TASendExpression(null, getCtrlChanName(func)));
			// Delete the parameter scope
			emptyScope(retCtrl);
			tmpl.addTransition(retCtrl);
		}

		// now put the template and function in the HashMap
		putTemplate(func, tmpl);
	}

	/**
	 * Transforms the given list of expressions in equivalent TATransitions and
	 * TALocations.
	 * 
	 * @param body
	 * @param startLoc
	 * @param endLoc
	 * @param breakLoc
	 * @param continueLoc
	 */
	private void createExpressionBody(List<Expression> body,
			TALocation startLoc, TALocation endLoc, TALocation breakLoc,
			TALocation continueLoc) {
		newScope();
		addToScope(vsa.getVarsExclInnerScopes(body));

		// transform Expressions
		TALocation currentLoc = startLoc;

		for (Expression expr : body) {
			currentLoc = createSingleExpression(expr, currentLoc, endLoc,
					breakLoc, continueLoc, "");
			if (currentLoc == null) {
				// at return/continue/break we are done here, dont loop further
				// (the rest is done by another createExprBody()-call)
				return;
			}
		}

		if (!tmpl.getLocations().contains(endLoc)) {
			tmpl.addLocation(endLoc);
		}

		TATransition exit = new TATransition(currentLoc, endLoc);
		emptyScope(exit);
		tmpl.addTransition(exit);
	}

	private TALocation createSingleExpression(Expression expr,
			TALocation currentLoc, TALocation endLoc, TALocation breakLoc,
			TALocation continueLoc, String prefix) {
		if (expr instanceof QuestionmarkExpression) {
			// TODO: pretty sure, this doesn't work
			return transformIfExpression(
					((QuestionmarkExpression) expr).toIfElseExpression(),
					currentLoc, breakLoc, continueLoc);
		} else if (expr instanceof AssertionExpression) {
			return handleAssertionExpression((AssertionExpression) expr,
					currentLoc);
		} else if (expr instanceof BreakExpression) {
			return handleBreakExpression(currentLoc, breakLoc);
		} else if (expr instanceof ExpressionBlock) {
			return handleExpressionBlock((ExpressionBlock) expr, currentLoc,
					endLoc, breakLoc, continueLoc, prefix);
		} else if (expr instanceof ContinueExpression) {
			return handleContinueExpression(currentLoc, continueLoc);
		} else if (expr instanceof FunctionCallExpression) {
			return handleFunctionCallExpression((FunctionCallExpression) expr,
					currentLoc, prefix);
		} else if (expr instanceof IfElseExpression) {
			return transformIfExpression((IfElseExpression) expr, currentLoc,
					breakLoc, continueLoc);
		} else if (expr instanceof SwitchExpression) {
			return transformSwitchExpression((SwitchExpression) expr,
					currentLoc, continueLoc);
		} else if (expr instanceof DoWhileLoopExpression) {
			return transformDoWhile((DoWhileLoopExpression) expr, currentLoc);
		} else if (expr instanceof ForLoopExpression) {
			return transformFor((ForLoopExpression) expr, currentLoc);
		} else if (expr instanceof WhileLoopExpression) {
			return transformWhile((WhileLoopExpression) expr, currentLoc);
		} else if (expr instanceof ReturnExpression) {
			return handleReturnExpression((ReturnExpression) expr, currentLoc,
					prefix);
		} else if (expr instanceof AccessExpression) {
			return handleAccess((AccessExpression) expr, currentLoc, prefix);
		} else if (expr instanceof EventNotificationExpression) {
			return handleEventNotification((EventNotificationExpression) expr,
					currentLoc);
		} else if (expr instanceof OutputExpression) {
			return handleOutputExpression((OutputExpression) expr, currentLoc);
		} else if (expr instanceof SCStopExpression) {
			return handleStopExpression((SCStopExpression) expr, currentLoc);
		} else {
			// when we arrive here, we can safely execute the inner function
			// before assignments, so 'int x = f();' actually becomes
			// 'f(); int x = "$f$return"'
			currentLoc = handleInnerFunctionCallExpr(expr, currentLoc);
			Expression replacement = ExpressionConverter.convert(expr, func);

			// create the a new location if needed
			if (replacement != null) {
				TALocation newLoc = tmpl.createUrgentLocation();
				TATransition trans = new TATransition(currentLoc, newLoc);
				trans.addUpdateExpression(replacement);
				tmpl.addTransition(trans);
				// add return paramter
				currentLoc = newLoc;
			}
		}
		return currentLoc;
	}

	private TALocation handleExpressionBlock(ExpressionBlock block,
			TALocation currentLoc, TALocation endLoc, TALocation breakLoc,
			TALocation continueLoc, String prefix) {
		TALocation loc;
		for (Expression inner : block.getBlock()) {
			loc = createSingleExpression(inner, currentLoc, endLoc, breakLoc,
					continueLoc, prefix);
			if (loc != null) {
				currentLoc = loc;
			}
		}

		return currentLoc;
	}

	private TALocation handleBreakExpression(TALocation currentLoc,
			TALocation breakLoc) {
		if (breakLoc == null) {
			logger.error("Break expression found, but no break location defined");
			return null;
		}

		if (!tmpl.getLocations().contains(breakLoc)) {
			tmpl.addLocation(breakLoc);
		}

		TATransition breakT = new TATransition(currentLoc, breakLoc);
		emptyAllScopesTillCurrentBarrier(breakT);
		tmpl.addTransition(breakT);

		return null;
	}

	private TALocation handleContinueExpression(TALocation currentLoc,
			TALocation continueLoc) {
		if (continueLoc == null) {
			logger.error("Continue expression found, but no continue location defined");
			return null;
		}

		if (!tmpl.getLocations().contains(continueLoc)) {
			tmpl.addLocation(continueLoc);
		}

		TATransition continueT = new TATransition(currentLoc, continueLoc);
		emptyAllScopesTillCurrentBarrier(continueT);
		tmpl.addTransition(continueT);

		return null;
	}

	private TALocation handleReturnExpression(ReturnExpression retExpr,
			TALocation currentLoc, String prefix) {
		currentLoc = handleInnerFunctionCallExpr(retExpr, currentLoc);
		TALocation initLoc = tmpl.getInitLocation();
		TATransition retCtrl = new TATransition(currentLoc, initLoc,
				new TASendExpression(null, getCtrlChanName(func)));
		Expression retStat = ExpressionConverter.convert(retExpr, func);
		if (retStat != null) {
			retCtrl.addUpdateExpression(retStat);
		}
		emptyAllScopes(retCtrl);
		tmpl.addTransition(retCtrl);
		return null;
	}

	private TALocation handleEventNotification(
			EventNotificationExpression enexpr, TALocation currentLoc) {
		Expression event = enexpr.getEvent();
		List<Expression> parameters = enexpr.getParameters();

		TALocation newLoc = null;

		if (tmpl.getName().startsWith("sc_clock")) {
			// System.out.println("clock process is: " + template.getName());
			currentLoc.setUrgent(false);
			currentLoc.setCommitted(true);
		}

		// TODO: test this
		// immediate notification
		if (parameters.size() == 0) {
			newLoc = tmpl.createUrgentLocation();
			tmpl.addTransition(new TATransition(currentLoc, newLoc,
					new TASendExpression(null, event.toStringNoSem()
							+ Constants.PREFIX_DELIMITER
							+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM)));

			// delta notification or delayed notification, called with sc_time
			// argument
		} else if (parameters.size() == 1) {
			Expression arg = parameters.get(0);

			if (arg.toStringNoSem().equals(SCConstants.SC_ZERO_TIME)) {
				// this is a delayed notification (time notification with zero
				// time)
				newLoc = createSimpleSyncTransition(event.toStringNoSem()
						+ Constants.PREFIX_DELIMITER, new ConstantExpression(
						null, "0"), tmpl, currentLoc);
			} else {
				// sc_time objects are already converted to sc_time_resolution
				// through sc_time()
				// reportError(node,
				// "Timed notify() with sc_time argument is not supported. Use notify(double, sc_time_unit).");
//				if (arg instanceof SCVariableExpression
//						&& ((SCVariableExpression) arg).getVar() instanceof SCTime) {
//					if (Engine.ALWAYS_USE_MEM_MODEL) {
//						
//					}
//					arg = new ConstantExpression(arg.getNode(), val)
//					arg = new ArrayAccessExpression(arg.getNode(), new SCArray(
//							TAMemoryType.getMemArrName(tap.getMemType()),
//							tap.getMemType()), arg);
//				}
				arg = ExpressionConverter.convert(arg, this.func);
				newLoc = createSimpleSyncTransition(event.toStringNoSem()
						+ Constants.PREFIX_DELIMITER, arg, tmpl, currentLoc);

			}

			// delta notification or delayed notification, called with two
			// arguments (double, sc_time_unit)
		} else if (parameters.size() == 2) {
			Expression time = parameters.get(0);
			Expression unit = parameters.get(1);

			int intTime = SCConstants.getTimeUnits(time.toStringNoSem(),
					unit.toStringNoSem());

			newLoc = createSimpleSyncTransition(event.toStringNoSem()
					+ Constants.PREFIX_DELIMITER, new ConstantExpression(null,
					Integer.toString(intTime)), tmpl, currentLoc);

		}

		// PEQ notification (has 3 parameters)
		if (event instanceof SCVariableExpression
				&& ((SCVariableExpression) event).getVar() instanceof SCPeq) {
			SCPeq peq = (SCPeq) ((SCVariableExpression) event).getVar();
			newLoc = PeqTransformer.createNotifyEvent(currentLoc, tmpl, peq,
					parameters);
		}

		return newLoc;
	}

	private TALocation createSimpleSyncTransition(String pref,
			Expression delay, TATemplate template, TALocation curLoc) {
		TALocation newLoc = template.createUrgentLocation();
		TATransition trans = new TATransition(curLoc, newLoc,
				new TASendExpression(null, pref
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY));
		trans.addUpdateExpression(new BinaryExpression(null,
				new ConstantExpression(null, pref
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME), "=",
				delay));
		template.addTransition(trans);
		return newLoc;
	}

	// TODO: are breakLoc... necessary and should be added?
	private TALocation handleAccess(AccessExpression acexpr,
			TALocation currentLoc, String prefix) {
		Expression left = acexpr.getLeft();
		Expression right = acexpr.getRight();

		// TODO: is this sufficient?
		prefix = prefix + left.toStringNoSem() + Constants.PREFIX_DELIMITER;

		return createSingleExpression(right, currentLoc, null, null, null,
				prefix);
	}

	/**
	 * Handles all assertion expressions. For each assertion, a new transition
	 * is generated with the assertion condition as a guard and the starting
	 * location of the transition gets a unique label. Strings which are part of
	 * an assertion are transformed into the boolean value "true" which is C++
	 * conform.
	 * 
	 * @param assexpr
	 * @param currentLoc
	 * @return
	 */
	private TALocation handleAssertionExpression(AssertionExpression assexpr,
			TALocation currentLoc) {
		// first step: handle inner functions
		currentLoc = handleInnerFunctionCallExpr(assexpr, currentLoc);

		// second step: convert the expression
		AssertionExpression newAssExpr = (AssertionExpression) ExpressionConverter
				.convert(assexpr, func);

		// Intermediate step: we cannot handle strings in conditions, therefore
		// we have to
		// replace all strings with true. This is c++ conform as all Strings are
		// evaluated to true.
		TALocation nextLoc = tmpl.createUrgentLocation();
		TATransition trans = new TATransition(currentLoc, nextLoc);
		List<Pair<Expression, Expression>> replacements = new LinkedList<Pair<Expression, Expression>>();
		for (Expression exp : newAssExpr.getInnerExpressions()) {
			if (exp instanceof ConstantExpression
					&& ((ConstantExpression) exp).getValue().contains("\"")) {
				Expression rep = new ConstantExpression(exp.getNode(), "true");
				replacements.add(new Pair<Expression, Expression>(exp, rep));
			}
		}
		newAssExpr.replaceInnerExpressions(replacements);

		// third step: create the guarded transition
		trans.setGuard(newAssExpr.getCondition());
		tmpl.addTransition(trans);

		// fourth step: create the failstate location and connect it
		TALocation failstate = tmpl.createUrgentLocation();
		failstate.setName(Constants.generateAssertionFailStateName());
		TATransition failstatetrans = new TATransition(currentLoc, failstate);
		Expression negatedCondition = new UnaryExpression(newAssExpr.getNode(),
				UnaryExpression.PRE, "!", new BracketExpression(
						newAssExpr.getNode(), newAssExpr.getCondition()));
		Expression fail = new BinaryExpression(newAssExpr.getNode(),
				new ConstantExpression(newAssExpr.getNode(),
						Constants.ERR_ASSERTION_VIOLATION), "=",
				new ConstantExpression(newAssExpr.getNode(), "true"));
		failstatetrans.setGuard(negatedCondition);
		failstatetrans.addUpdateExpression(fail);

		tmpl.addLocation(failstate);
		tmpl.addTransition(failstatetrans);

		return nextLoc;
	}

	private TALocation handleFunctionCallExpression(
			FunctionCallExpression fcexpr, TALocation currentLoc, String prefix) {
		currentLoc = handleInnerFunctionCallExpr(fcexpr, currentLoc);
		return transformFunctionCallExpression(fcexpr, currentLoc, prefix);
	}

	/**
	 * This functions handles all FunctionCallExpressions which are in the inner
	 * expressions of the given Expression. It creates new transitions for the
	 * call and return and it replaces all FunctionCallExpressions with
	 * equivalent TAFunctionCallExpressions. Also handles AccessExpressions if a
	 * FunctionCallExpression has one as parent. Replaces parent
	 * AccessExpressions with equivalent TAAccessExpressions.
	 * 
	 * @param expr
	 * @param currentLoc
	 * @return
	 */
	private TALocation handleInnerFunctionCallExpr(Expression expr,
			TALocation currentLoc) {
		boolean repeat = true;
		while (repeat) {
			repeat = false;
			List<Expression> innerExpr = expr.getInnerExpressions();
			for (Expression iexpr : innerExpr) {
				if ((iexpr instanceof FunctionCallExpression)
						&& !(iexpr instanceof TAFunctionCallExpression)) {
					currentLoc = handleInnerFunctionCallExpr(iexpr, currentLoc);

					FunctionCallExpression fcexpr = (FunctionCallExpression) iexpr;
					Expression currParent = fcexpr.getParent();
					List<Pair<Expression, Expression>> replacements = new LinkedList<Pair<Expression, Expression>>();
					TAVariableExpression taFcExpr = new TAVariableExpression(
							fcexpr.getNode(),
							VariableConverter.getTrnsprtVarForFuncReturn(fcexpr
									.getFunction()));
					taFcExpr.setParent(currParent);
					replacements.add(new Pair<Expression, Expression>(fcexpr,
							taFcExpr));
					currParent.replaceInnerExpressions(replacements);
					replacements.clear();
					//fcexpr.setParent(taFcExpr);
					String accessPrefix = "";
					// check if the parent is an AccessExpression
					while (currParent instanceof AccessExpression) {
						// TODO: maybe we should return the new expression, too
						// because we actually need to check if the cond has
						// changed in the
						// caller (ie. handleIfElseExpr and
						// handleInnerFunccall(condition))
						AccessExpression acexpr = (AccessExpression) currParent;
						currParent = currParent.getParent();
						accessPrefix = acexpr.getLeft().toStringNoSem()
								+ Constants.PREFIX_DELIMITER + accessPrefix;

						TAAccessExpression taAcExpr = new TAAccessExpression(
								acexpr.getNode(), acexpr.getLeft(),
								acexpr.getOp(), acexpr.getRight(), acexpr);
						taAcExpr.setParent(currParent);
						replacements.add(new Pair<Expression, Expression>(
								acexpr, taAcExpr));
						currParent.replaceInnerExpressions(replacements);
						replacements.clear();
					}

					// create the call and return Transitions
					TALocation afterCall = transformFunctionCallExpression(
							fcexpr, currentLoc, accessPrefix);

					currentLoc = afterCall;

					repeat = true;
					break;
				}
			}
		}
		return currentLoc;
	}

	private TALocation transformFunctionCallExpression(
			FunctionCallExpression fcexpr, TALocation startLoc, String prefix) {
		SCFunction function = fcexpr.getFunction();
		List<Expression> arguments = fcexpr.getParameters();
		TALocation afterCall = tmpl.createUrgentLocation();
		if (function.getName().equals("wait")) {
			// special handling for wait function
			WaitFunctionTransformer wft = new WaitFunctionTransformer();
			wft.createFunctionCall(func.getSCClass(), tmpl, startLoc,
					afterCall, function, arguments, fcexpr);
		} else if (function.getName().equals("request_update")) {
			// special handling for request update
			TATransition request = new TATransition(
					startLoc,
					afterCall,
					new TASendExpression(null, Constants.REQUEST_UPDATE_CHANNEL));
			tmpl.addTransition(request);
		} else {
			// TODO: time consumption?
			TALocation funcLoc = tmpl.createLocation();
			String ctrlChanName = getCtrlChanName(function, prefix);
			tmpl.addParameter(getCtrlChan(function, prefix));
			// call the function
			TATransition callFunc = new TATransition(startLoc, funcLoc,
					new TASendExpression(null, ctrlChanName));
			// update transport vars
			int i = 0;
			for (SCParameter par : function.getParameters()) {
			//	if(arguments.size() > i) 
					Expression arg = arguments.get(i); 
					TAVariable tavar = VariableConverter.getTmplParFromFuncPar(par,
						prefix);
					if (tavar != null) {
					// use 'func' here because when calling, we are still
					// outside of 'function'
						callFunc.addUpdateExpression(new TAVariableAssignmentExpression(
							null, tavar, ExpressionConverter.convert(arg, func)));
					}
					
				i++;
			}
			tmpl.addTransition(callFunc);
			// return from function
			TATransition getCtrl = new TATransition(funcLoc, afterCall,
					new TARecvExpression(null, ctrlChanName));
			// reset transport vars
			for (SCParameter par : function.getParameters()) {
				TAVariable tavar = VariableConverter.getTmplParFromFuncPar(par,
						prefix);
				// migh
				getCtrl.addUpdateExpression(tavar.getResetExpr());

				// the function might not be in the same module (e.g. it's
				// called via sockets), so we might need extra paramters
				tmpl.addParameter(tavar);
			}
			// maybe add an return parameter
			if (VariableConverter.getTrnsprtVarForFuncReturn(function) != null) {
				TAVariable retVar = VariableConverter
						.getTrnsprtVarForFuncReturn(function, prefix);
				tmpl.addParameter(retVar);
			}

			tmpl.addTransition(getCtrl);
		}
		return afterCall;
	}

	// TODO: extend handling of functionCallExpression, the condition could be a
	// function call
	private TALocation transformIfExpression(IfElseExpression expr,
			TALocation startLoc, TALocation breakLoc, TALocation continueLoc) {

		TALocation afterIf = new TALocation();
		afterIf.setUrgent(true);

		Expression cond = expr.getCondition();
		List<Expression> ifPath = expr.getThenBlock();
		List<Expression> elsePath = expr.getElseBlock();

		// handle FunctionCallExpressions for condition
		startLoc = handleInnerFunctionCallExpr(cond, startLoc);
		cond = expr.getCondition(); // condition may have changed during
									// handleInnerFuncCall()
		if (cond instanceof FunctionCallExpression) {
			FunctionCallExpression fcexpr = (FunctionCallExpression) cond;
			TALocation afterCall = transformFunctionCallExpression(fcexpr,
					startLoc, "");

			cond = new ConstantExpression(cond.getNode(),
					VariableConverter.getTrnsprtVarNameForFuncReturn(fcexpr
							.getFunction()));
			startLoc = afterCall;
		} else {
			cond = ExpressionConverter.convert(cond, func);
		}

		// if Path
		TALocation ifPathStart = tmpl.createLocation();
		ifPathStart.setUrgent(true);

		TATransition ifCondT = new TATransition(startLoc, ifPathStart);
		ifCondT.setGuard(cond);
		tmpl.addTransition(ifCondT);

		createExpressionBody(ifPath, ifPathStart, afterIf, breakLoc,
				continueLoc);

		// else Path
		if (!elsePath.isEmpty()) {
			TALocation elsePathStart = tmpl.createLocation();
			elsePathStart.setUrgent(true);

			TATransition elseCondT = new TATransition(startLoc, elsePathStart);
			elseCondT
					.setGuard(new UnaryExpression(cond.getNode(),
							UnaryExpression.PRE, "!", new BracketExpression(
									null, cond)));
			tmpl.addTransition(elseCondT);

			createExpressionBody(elsePath, elsePathStart, afterIf, breakLoc,
					continueLoc);
		} else {
			if (!tmpl.getLocations().contains(afterIf)) {
				tmpl.addLocation(afterIf);
			}
			TATransition elseCondT = new TATransition(startLoc, afterIf);
			elseCondT
					.setGuard(new UnaryExpression(cond.getNode(),
							UnaryExpression.PRE, "!", new BracketExpression(
									null, cond)));
			tmpl.addTransition(elseCondT);
		}

		if (tmpl.getLocations().contains(afterIf)) {
			return afterIf;
		} else {
			return null;
		}
	}

	private TALocation transformSwitchExpression(SwitchExpression expr,
			TALocation startLoc, TALocation continueLoc) {

		TALocation afterSwitch = new TALocation();
		afterSwitch.setUrgent(true);

		Expression swExpr = expr.getSwitchExpression();

		List<Expression> cases = expr.getCases();

		Expression defCond = createDefCond(cases, swExpr);
		defCond = ExpressionConverter.convert(defCond, func);

		// handle FunctionCallExpressions for condition
		startLoc = handleInnerFunctionCallExpr(swExpr, startLoc);
		swExpr = expr.getSwitchExpression(); // condition may have changed
												// during handleInnerFuncCall()
		if (swExpr instanceof FunctionCallExpression) {
			FunctionCallExpression fcexpr = (FunctionCallExpression) swExpr;
			TALocation afterCall = transformFunctionCallExpression(fcexpr,
					startLoc, "");

			swExpr = new ConstantExpression(swExpr.getNode(),
					VariableConverter.getTrnsprtVarNameForFuncReturn(fcexpr
							.getFunction()));
			startLoc = afterCall;
		}

		TALocation pathStart = new TALocation();
		pathStart.setUrgent(true);

		boolean defaulthandled = false;

		for (Expression ce : cases) {
			if (ce instanceof CaseExpression) {
				CaseExpression cas = (CaseExpression) ce;
				TALocation nextPath = new TALocation();
				nextPath.setUrgent(true);

				if (!tmpl.getLocations().contains(pathStart)) {
					tmpl.addLocation(pathStart);
				}

				Expression cond;

				if (!cas.isDefaultCase()) {
					cond = new BinaryExpression(ce.getNode(), swExpr, "==",
							cas.getCondition());
					cond = ExpressionConverter.convert(cond, func);
				} else {
					cond = defCond;
					defaulthandled = true;
				}

				TATransition condT = new TATransition(startLoc, pathStart);

				condT.setGuard(cond);
				tmpl.addTransition(condT);

				markUpcomingScopeAsBarrier();
				createExpressionBody(cas.getBody(), pathStart, nextPath,
						afterSwitch, continueLoc);
				unmarkCurrentScopeAsBarrier();

				pathStart = nextPath;
			} else {
				logger.error(
						"Encountered case which is not a case expression: {}",
						ce);
			}

		}

		if (tmpl.getLocations().contains(pathStart)) {
			TATransition toEndT = new TATransition(pathStart, afterSwitch);
			tmpl.addTransition(toEndT);
		}

		if (!defaulthandled) {
			if (!tmpl.getLocations().contains(afterSwitch)) {
				tmpl.addLocation(afterSwitch);
			}

			TATransition defCondT = new TATransition(startLoc, afterSwitch);
			defCondT.setGuard(defCond);
			tmpl.addTransition(defCondT);

			if (!tmpl.getLocations().contains(pathStart)) {
				TATransition exitT = new TATransition(pathStart, afterSwitch);
				tmpl.addTransition(exitT);
			}
		}

		if (tmpl.getLocations().contains(afterSwitch)) {
			return afterSwitch;
		} else {
			return null;
		}

	}

	private Expression createDefCond(List<Expression> cases, Expression swExpr) {
		Expression defCond = null;
		for (Expression ce : cases) {
			if (ce instanceof CaseExpression) {
				CaseExpression cas = (CaseExpression) ce;
				if (!cas.isDefaultCase()) {
					Expression cond = new BinaryExpression(cas.getNode(),
							swExpr, "==", cas.getCondition());
					if (defCond != null) {
						defCond = new BinaryExpression(null, defCond, "&&",
								new UnaryExpression(null, UnaryExpression.PRE,
										"!", new BracketExpression(null, cond)));
					} else {
						defCond = new UnaryExpression(null,
								UnaryExpression.PRE, "!",
								new BracketExpression(null, cond));
					}
				}
			} else {
				logger.error(
						"Encountered case which is not a case expression: {}",
						ce.toString());
			}

		}
		return defCond;
	}

	// TODO: extend handling of functionCallExpression, the condition could be a
	// function call
	private TALocation transformDoWhile(DoWhileLoopExpression expr,
			TALocation startLoc) {

		TALocation afterLoop = new TALocation();
		afterLoop.setUrgent(true);

		Expression cond = expr.getCondition();
		List<Expression> loopBody = expr.getLoopBody();

		TALocation afterLoopBody = new TALocation();
		afterLoopBody.setUrgent(true);

		markUpcomingScopeAsBarrier();
		createExpressionBody(loopBody, startLoc, afterLoopBody, afterLoop,
				afterLoopBody);
		unmarkCurrentScopeAsBarrier();

		if (!tmpl.getLocations().contains(afterLoopBody)) {
			tmpl.addLocation(afterLoopBody);
		}

		if (!tmpl.getLocations().contains(afterLoop)) {
			tmpl.addLocation(afterLoop);
		}

		// handle InnerFunctionCallExpressions for condition
		afterLoopBody = handleInnerFunctionCallExpr(cond, afterLoopBody);
		cond = expr.getCondition(); // condition may have changed during
									// handleInnerFuncCall()

		TATransition loopCondT = new TATransition(afterLoopBody, startLoc);
		cond = ExpressionConverter.convert(cond, func);
		loopCondT.setGuard(cond);
		tmpl.addTransition(loopCondT);

		TATransition loopEndCondT = new TATransition(afterLoopBody, afterLoop);
		loopEndCondT.setGuard(new UnaryExpression(cond.getNode(),
				UnaryExpression.PRE, "!", new BracketExpression(null, cond)));
		tmpl.addTransition(loopEndCondT);

		if (tmpl.getLocations().contains(afterLoop)) {
			return afterLoop;
		} else {
			return null;
		}
	}

	// TODO: extend handling of functionCallExpression, the condition could be a
	// function call
	private TALocation transformWhile(WhileLoopExpression expr,
			TALocation startLoc) {

		TALocation afterLoop = new TALocation();
		afterLoop.setUrgent(true);

		Expression cond = expr.getCondition();
		List<Expression> loopBody = expr.getLoopBody();

		// handle InnerFunctionCallExpressions for condition
		TALocation beforeCond = startLoc;
		beforeCond = handleInnerFunctionCallExpr(cond, beforeCond);
		cond = expr.getCondition(); // condition may have changed during
									// handleInnerFuncCall()

		TALocation beforeLoopBody = tmpl.createUrgentLocation();

		TATransition loopCondT = new TATransition(beforeCond, beforeLoopBody);
		cond = ExpressionConverter.convert(cond, func);
		loopCondT.setGuard(cond);
		tmpl.addTransition(loopCondT);

		markUpcomingScopeAsBarrier();
		createExpressionBody(loopBody, beforeLoopBody, startLoc, afterLoop,
				startLoc);
		unmarkCurrentScopeAsBarrier();

		if (!tmpl.getLocations().contains(afterLoop)) {
			// TODO: when this happens, the rest is dead code
			tmpl.addLocation(afterLoop);
		}

		TATransition loopEndCondT = new TATransition(beforeCond, afterLoop);
		loopEndCondT.setGuard(new UnaryExpression(cond.getNode(),
				UnaryExpression.PRE, "!", new BracketExpression(null, cond)));
		tmpl.addTransition(loopEndCondT);

		return afterLoop;
	}

	// TODO: extend handling of functionCallExpression, the condition could be a
	// function call
	private TALocation transformFor(ForLoopExpression expr, TALocation startLoc) {

		TALocation afterLoop = new TALocation();
		afterLoop.setUrgent(true);

		Expression init = expr.getInitializer();
		Expression cond = expr.getCondition();
		Expression iter = expr.getIterator();
		List<Expression> loopBody = expr.getLoopBody();

		// init
		// handle InnerFunctionCallExpressions for initializer
		startLoc = handleInnerFunctionCallExpr(init, startLoc);
		init = expr.getInitializer(); // may have changed during inner func call

		TALocation afterInit = tmpl.createLocation();
		afterInit.setUrgent(true);

		TATransition initT = new TATransition(startLoc, afterInit);
		initT.addUpdateExpression(ExpressionConverter.convert(init, func));
		tmpl.addTransition(initT);

		// condition
		// handle InnerFunctionCallExpressions for condition
		TALocation beforeCond = afterInit;
		beforeCond = handleInnerFunctionCallExpr(cond, beforeCond);
		cond = expr.getCondition(); // condition may have changed during
									// handleInnerFuncCall()

		TALocation beforeLoopBody = new TALocation();
		beforeLoopBody.setUrgent(true);

		tmpl.addLocation(beforeLoopBody);

		TATransition loopCondT = new TATransition(beforeCond, beforeLoopBody);
		cond = ExpressionConverter.convert(cond, func);
		loopCondT.setGuard(cond);
		tmpl.addTransition(loopCondT);

		// body
		TALocation beforeIter = tmpl.createLocation();
		beforeIter.setUrgent(true);

		markUpcomingScopeAsBarrier();
		createExpressionBody(loopBody, beforeLoopBody, beforeIter, afterLoop,
				beforeIter);
		unmarkCurrentScopeAsBarrier();

		// iterate
		if (!tmpl.getLocations().contains(beforeIter)) {
			// TODO: when this happens, the rest is dead code
			tmpl.addLocation(beforeIter);
		}
		// handle InnerFunctionCallExpressions for iterator
		beforeIter = handleInnerFunctionCallExpr(iter, beforeIter);
		iter = expr.getIterator(); // may have changed during hanlde
									// func call

		TATransition iterT = new TATransition(beforeIter, afterInit);
		iterT.addUpdateExpression(ExpressionConverter.convert(iter, func));
		tmpl.addTransition(iterT);

		// end loop
		if (!tmpl.getLocations().contains(afterLoop)) {
			tmpl.addLocation(afterLoop);
		}
		TATransition loopEndCondT = new TATransition(beforeCond, afterLoop);
		loopEndCondT.setGuard(new UnaryExpression(cond.getNode(),
				UnaryExpression.PRE, "!", new BracketExpression(null, cond)));
		tmpl.addTransition(loopEndCondT);

		return afterLoop;
	}

	/**
	 * We ignore output expressions in sc2uppaal as we cannot produce the output
	 * as in c++/systemc
	 * 
	 * @param expr
	 * @param currentLoc
	 * @return
	 */
	private TALocation handleOutputExpression(OutputExpression expr,
			TALocation currentLoc) {
		return currentLoc;
	}

	/**
	 * Currently we ignore a call of sc_stop. TODO mp: we should have a new
	 * state in the scheduler (stopped) and a global signal sc_stop.
	 * 
	 * @param expr
	 * @param currentLoc
	 * @return
	 */
	private TALocation handleStopExpression(SCStopExpression expr,
			TALocation currentLoc) {
		return currentLoc;
	}

	private void newScope() {
		scopeStack.push(new HashMap<TAVariable, SCVariable>());
	}

	/**
	 * Add a barrierIndex. Use this when calling continue/break
	 */
	private void markUpcomingScopeAsBarrier() {
		barrierIndices.push(scopeStack.size());
	}

	private void unmarkCurrentScopeAsBarrier() {
		barrierIndices.pop();
	}

	private void emptyScope(Stack<Map<TAVariable, SCVariable>> stack,
			TATransition trans) {
		TAVariable tavar;
		SCVariable scvar;
		Map<TAVariable, SCVariable> currentMap = stack.pop();
		for (Map.Entry<TAVariable, SCVariable> entry : currentMap.entrySet()) {
			scvar = entry.getValue();
			tavar = entry.getKey();
			trans.addUpdateExpression(TAMemoryType
					.getDeleteExpressionForLocVar(tavar, scvar));
		}
	}

	private void emptyScope(TATransition trans) {
		emptyScope(scopeStack, trans);
	}

	private void addToScope(SCVariable scvar, String newName) {
		if (VariableConverter.useMemModel(scvar)) {
			TAVariable tavar = VariableConverter.convertVariable(scvar);
			if (newName != null && !newName.equals("")) {
				tavar.setName(newName);
			}
			scopeStack.peek().put(tavar, scvar);
		}
	}

	private void addToScope(SCVariable scvar) {
		addToScope(scvar, null);
	}

	private void addToScope(List<SCVariable> scvars) {
		if (scvars != null) {
			for (SCVariable scvar : scvars) {
				if (VariableConverter.useMemModel(scvar)) {
					addToScope(scvar);
				}
			}
		}
	}

	/**
	 * Call this whenever there is a return statement. Don't actually remove the
	 * scope stack because we still need it for other programm flows.
	 * 
	 * @param trans
	 */
	private void emptyAllScopes(TATransition trans) {
		Stack<Map<TAVariable, SCVariable>> shallowCopy = new Stack<Map<TAVariable, SCVariable>>();
		shallowCopy.addAll(scopeStack);
		while (shallowCopy.size() > 0) {
			emptyScope(shallowCopy, trans);
		}
	}

	/**
	 * Call this for continue/break. Use current barrierIndex to know how many
	 * leves we need to go up.
	 * 
	 * @param trans
	 */
	private void emptyAllScopesTillCurrentBarrier(TATransition trans) {
		Stack<Map<TAVariable, SCVariable>> shallowCopy = new Stack<Map<TAVariable, SCVariable>>();
		int currentBarrierIndex = barrierIndices.peek();
		for (int i = scopeStack.size() - 1; i >= currentBarrierIndex; i--) {
			shallowCopy.add(0, scopeStack.get(i));
		}
		while (shallowCopy.size() > 0) {
			emptyScope(shallowCopy, trans);
		}
		// remove the very last scope from the stack because we just left it
		// with our break/continue
		scopeStack.pop();
	}

	public static List<TAVariable> getNeededTmplParamsFromFunc(SCFunction func,
			String prefix) {
		List<TAVariable> ls = new LinkedList<TAVariable>();
		ls.add(getCtrlChan(func, prefix));
		for (SCParameter par : func.getParameters()) {
			ls.add(VariableConverter.getTmplParFromFuncPar(par, prefix));
		}
		String retType = func.getReturnType();
		if (!retType.equals("void")) {
			ls.add(VariableConverter.getTrnsprtVarForFuncReturn(func, prefix));
		}
		return ls;
	}

	public static String getCtrlChanName(SCFunction func, String prefix) {
		return prefix + func.getName() + Constants.PREFIX_DELIMITER
				+ Constants.LOCAL_FUNCTION_CTRL_KEYWORD;
	}

	public static String getCtrlChanName(SCFunction func) {
		return getCtrlChanName(func, "");
	}

	private static TAChannel getCtrlChan(SCFunction func, String prefix) {
		return new TAChannel(getCtrlChanName(func, prefix));
	}

}
