/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.engine;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.optimization.OptimizationEngine;
import de.tub.pes.sc2uppaal.optimization.POR;
import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.SCMainConverter;
import de.tub.pes.sc2uppaal.tamodel.TAMemoryType;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.VariableConverter;
import de.tub.pes.sc2uppaal.util.XMLWriter;
import de.tub.pes.syscir.analysis.MemoryConsumptionAnalyzer;
import de.tub.pes.syscir.analysis.TimeConsumptionAnalyzer;
import de.tub.pes.syscir.analysis.data_race_analyzer.DataRaceAnalyzer;
import de.tub.pes.syscir.engine.TransformerFactory;
import de.tub.pes.syscir.engine.modeltransformer.CallByReferenceTransformer;
import de.tub.pes.syscir.engine.modeltransformer.SCTime2IntTransformer;
import de.tub.pes.syscir.engine.modeltransformer.SocketCallTransformer;
import de.tub.pes.syscir.engine.modeltransformer.StructMethodTransformer;
import de.tub.pes.syscir.engine.modeltransformer.TypeNameTransformer;
import de.tub.pes.syscir.engine.util.CommandLineParser;
import de.tub.pes.syscir.sc_model.SCClass;
import de.tub.pes.syscir.sc_model.SCSystem;
import de.tub.pes.syscir.sc_model.SCVariable;
import de.tub.pes.syscir.sc_model.variables.SCClassInstance;
import de.tub.pes.syscir.sc_model.variables.SCPeq;

/**
 * This class is the starting class for the sc2uppaal application. It controls
 * the phases of the transformation.
 * 
 * @author Timm Liebrenz
 * 
 */
public class Engine {

	private static Logger logger = LogManager.getLogger(Engine.class.getName());

	// TODO: when refactoring the arg parsing, put this somewhere it belongs to
	// by default, use mem model only for vars where needed
	public static boolean ALWAYS_USE_MEM_MODEL = false;
	public static boolean USE_DEFRAGMENTATION = false;

	/**
	 * The SC System instance, exists only once in the system.
	 */
	private SCSystem scSystem;

	public SCSystem getScSystem() {
		return scSystem;
	}

	/**
	 * The TA model instance, exists only one in the system.
	 */
	private final TAModel ta;

	public TAModel getTa() {
		return ta;
	}

	/**
	 * Standard constructor creates instances of the internal models.
	 * 
	 */
	public Engine() {
		ta = new TAModel();
		// set references in transformers
		PeqTransformer.initialize(ta);
	}

	/**
	 * Performs the transformation by executing the following tasks:
	 * <ul>
	 * <li>loading SysCIR model</li>
	 * <li>...</li>
	 * <li>Instantiation/Binding Phase - Creates TA model entries from the SC
	 * model.</li>
	 * <li>Optimization phase - Performs optimization routines to reduce the
	 * number of elements in the TA model.
	 * <li/>
	 * </ul>
	 * The time consumed by every phase is measured and printed out after
	 * completing the transformation.
	 * 
	 * @param in
	 * @param out
	 * @param peq
	 * @return True
	 */
	public boolean transform(String in, String out, String tb, String peq,
			String query, boolean updateRequestDeterminism,
			boolean partialOrderReduction, boolean persistentsSleeps) {
		long time1, time2, time3, time4, time5, time6;

		time1 = System.currentTimeMillis();

		File f = new File(Constants.CFG_ROOT + "config");
		if (!(f.exists() && f.isDirectory())) {
			logger.warn(
					"Couldn't find config folder at {}, searching at jar location",
					Constants.CFG_ROOT + "config");
			File tmp = new File(TransformerFactory.class.getProtectionDomain()
					.getCodeSource().getLocation().getPath());
			String location = tmp.getParentFile().getPath() + "/";
			File fj = new File(location + "config");
			if (fj.exists() && fj.isDirectory()) {
				Constants.CFG_ROOT = location;
			} else
				logger.error("Couldn't find config folder at {}", location);
		}

		// loading file
		// TODO for testing purpose use SysCIR to build the model, later it
		// should be possible to load the model from file
		scSystem = de.tub.pes.syscir.engine.Engine.buildModelFromFile(in);
		// for later use
		// scSystem = SCSystem.load(in);

		transformSystem(scSystem);

		ta.setUpdateRequestDeterminism(updateRequestDeterminism);

		//afterwards, the basic SCSystem is build and the analysis begins
		time2 = System.currentTimeMillis();
		
		// Partial Order Reduction
		if (partialOrderReduction) {
			DataRaceAnalyzer dra = new DataRaceAnalyzer();
			dra.analyze(scSystem);
			POR por = new POR(dra);
			ta.setPor(por);
		}
		// Partial Order Reduction with persistents and sleeps sets
		if (persistentsSleeps && !partialOrderReduction) {
			DataRaceAnalyzer dra = new DataRaceAnalyzer();
			dra.analyze(scSystem);
			POR por = new POR(dra, true);
			ta.setPor(por);
		}

		TimeConsumptionAnalyzer tca = new TimeConsumptionAnalyzer();
		tca.analyze(scSystem);

		MemoryConsumptionAnalyzer mca = new MemoryConsumptionAnalyzer();
		mca.analyze(scSystem);
		VariableConverter.init(scSystem);
		TAMemoryType mt = null;
		HashSet<String> usedTypes = new HashSet<String>();
		boolean memArrayOnlyForRefVars = !ALWAYS_USE_MEM_MODEL;
		for (SCVariable scvar : mca.getVars()) {
			String type = scvar.getTypeWithoutSize();
			if (!usedTypes.contains(type)) {
				if (!TAMemoryType.isForbiddenType(type)
						&& scvar.isNotSCModule() && scvar.isNotChannel()) {
					String typedef = TAMemoryType.getTypedef(scvar, scSystem);
					String defaultValue = TAMemoryType.getDefaultValueString(
							scvar, scSystem);
					// the basic memory size without PEQs
					
					int memSize = mca.getTotalCountForAllSizes(type, memArrayOnlyForRefVars);
					// this size has to be increased by the number of elements
					// per peq for both the phase data type and the payload data
					// type
					memSize += calcPeqMemConsumptionForType(mca, type);
					mt = new TAMemoryType(type, typedef, memSize, defaultValue,
							scSystem, ta);
					ta.addMemoryType(mt);
					usedTypes.add(type);
				}
			}
		}
		
		// finished analysis, creating timed automata model
		time3 = System.currentTimeMillis();

		// create predefined Timed Automata
		Constants.createPredefined(ta);

		// create Timed Automata
		Transformer transformer = new Transformer(scSystem, ta);
		transformer.createTAModel();

		// create Scheduler
		Constants.createScheduler(ta, scSystem.countInitProcesses());
		SCMainConverter scmc = new SCMainConverter(ta, scSystem);
		scmc.createSCMainTemplate();

		//automata model generated, starting optimizations
		time4 = System.currentTimeMillis();

		optimize();

		//optimizations done, write out design
		time5 = System.currentTimeMillis();

		ta.labelLocations();

		// create uppaal-xml-file
		if (out != null && !out.equals("")) {
			writeOut(out);
		}
		
		time6 = System.currentTimeMillis();

		logger.info("SCSystem build time: {}", time2 - time1);
		logger.info("Analysis time:       {}", time3 - time2);
		logger.info("Transformation time: {}", time4 - time3);
		logger.info("Optimization time:   {}", time5 - time4);
		logger.info("Write out time:      {}", time6 - time5);

		return true;
	}

	public boolean transform(String in) {
		return transform(in, in + ".uppaal.xml", null, null, null, false,
				false, false);
	}

	/**
	 * Launches the optimization in the optimization engine.
	 */
	public void optimize() {
		OptimizationEngine.run(ta);
	}

	/**
	 * Write TAModel to a freshly created XML file.
	 * 
	 * @param file
	 */
	public void writeOut(String file) {
		try {
			XMLWriter writer = new XMLWriter(file);
			if (writer.canWrite() && ta != null) {
				ta.print(writer);
				writer.flush();
			}
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		String MEM_MODEL_FLAG = "m";
		String DEFRAG_FLAG = "defrag";

		Map<String, String> argMap = CommandLineParser.parseArgs(args);

		if (argMap.size() == 0) {
			System.out
					.println("Insufficent number of parameters. Use -h for help.");
			return;
		}

		if (argMap.containsKey("h") || argMap.containsKey("H")) {
			// System.out.println("SystemC to Timed Automata Transformation Engine ver. "
			// + Constants.VERSION);
			System.out
					.println("SysCIR to Timed Automata Transformation Engine");
			System.out.println("");
			System.out.println("Mandatory Arguments:");
			System.out
					.println("-i [INPUTFILE] - absolute path to the AST xml-File which should be transformed");
			System.out.println();
			System.out.println("Optional Arguments:");
			System.out
					.println("-o [OUTPUTFILE] - absolute path where the resulting UPPAAL-file should be created. (default: inputfilename + .uppaal.xml)");
			System.out.println("-tr [INT] - time resolution in nano seconds (default: 1, -tr 10000 => time resolution set to 10 us)");
			System.out
					.println("-tb [TESTBENCHFILE] - absolute path to the testbench-file");
			System.out
					.println("-maxpeq [INT] - number of entries per PEQ. (default: 4)");
			System.out
					.println("-p [PEQFILE] - absolute path to a file containing the sizes for all PEQs. (default: inputfilename + .peq)");
			System.out
					.println("-pq [FILE] - absolute path to a query file where all automatically generated queries (e.g. assertions, out of range memory access) are stored. (default: inputfilename + _auto.q");
			System.out.println("-urdet - update request determinism");
			System.out.println("-por - partial order reduction");
			System.out
					.println("-por-ps - partial order reduction with persistents and sleeps sets");
			System.out
					.println("-"
							+ MEM_MODEL_FLAG
							+ " [1|0] - use dyn mem model for all vars (1) or only where needed (0, default)");
			// System.out
			// .println("-"
			// + DEFRAG_FLAG
			// +
			// " [1|0] - use defragmentation [sound] (1) or not [faster] (0, default). Aktivating defragmentation also activates the \"-"
			// + MEM_MODEL_FLAG + " 1\" option.");
			// System.out.println("-peqsize - calculates the minimum size for each peq in the model. Needs the parameters p and q.");
			return;
		}

		String inFile = null;
		String outFile = null;
		String tbFile = null;
		String peqFile = null;
		String pqFile = null;

		boolean urdet = false;
		boolean por = false;
		boolean persistentsSleeps = false;

		// String queryFile = null;
		int max_peq_entries = 4;
		int time_resolution = 1;
		// boolean calcPeqSize = false;

		if (argMap.containsKey("i")) {
			inFile = argMap.get("i");
		} else {
			System.out.println("No input file specified. Use -h for help.");
			return;
		}

		if (argMap.containsKey("o")) {
			outFile = argMap.get("o");
		} else {
			outFile = createFileName(inFile, ".uppaal.xml");
			System.out
					.println("No output file specified. Generating output file name from input file name.");
			System.out.println("Outputfile is " + outFile + ".");
		}

		if (argMap.containsKey("tr")) {
			try {
				time_resolution = Integer.parseInt(argMap.get("tr"));
			} catch (NumberFormatException ex) {
				logger.error("Submitted time resolution is no integer value.");
				return;
			}
			Constants.SC_TIME_RESOLUTION = time_resolution;
		} else {
			Constants.SC_TIME_RESOLUTION = 1; //default: time resolution in nanoseconds
		}
		
		if (argMap.containsKey("tb")) {
			tbFile = argMap.get("tb");
		}

		if (argMap.containsKey("maxpeq")) {
			try {
				max_peq_entries = Integer.parseInt(argMap.get("maxpeq"));
			} catch (NumberFormatException ex) {
				logger.error("Submitted maxpeq-argument is no integer value.");
				return;
			}
			Constants.GLOBAL_PEQ_SIZE = max_peq_entries;
		}

		if (argMap.containsKey("p")) {
			peqFile = argMap.get("p");
		}

		if (argMap.containsKey("pq")) {
			pqFile = argMap.get("pq");
			if (pqFile == "true") {
				pqFile = createFileName(inFile, "_auto.q");
			}
		}

		if (argMap.containsKey("urdet")) {
			urdet = true;
		}

		if (argMap.containsKey("por")) {
			por = true;
		}

		if (argMap.containsKey("por-ps")) {
			persistentsSleeps = true;
		}

		if (argMap.containsKey(MEM_MODEL_FLAG)
				&& argMap.get(MEM_MODEL_FLAG).equals("1")) {
			ALWAYS_USE_MEM_MODEL = true;
		} else {
			ALWAYS_USE_MEM_MODEL = false;
		}

		// This option has to be handled AFTER the MEM_MODEL_FLAG as it may
		// override the user settings. Defragmentation only works if we use the
		// full memory model.
		if (argMap.containsKey(DEFRAG_FLAG)
				&& argMap.get(DEFRAG_FLAG).equals("1")) {
			USE_DEFRAGMENTATION = true;
			ALWAYS_USE_MEM_MODEL = true;
		} else {
			USE_DEFRAGMENTATION = false;
		}

		Engine e = new Engine();

		if (!e.transform(inFile, outFile, tbFile, peqFile, pqFile, urdet, por,
				persistentsSleeps)) {
			logger.error("An error occured during transformation.");
			return;
		}
	}

	/**
	 * Creates a new file name out of a given name. Replaces the old file
	 * extension (after the last .) with extension.
	 * 
	 * 
	 * @param name
	 *            Input file name
	 * @return extension file name
	 */
	public static String createFileName(String name, String extension) {

		if (name.matches(".*\\.xml")) {
			name = name.substring(0, name.lastIndexOf("."));
		}

		return name + extension;
	}

	private void transformSystem(SCSystem scsystem) {
		Map<String, String> typeReplacements = new HashMap<String, String>();
		typeReplacements.put("unsigned int", "unsigned_int");
		typeReplacements.put("unsigned short", "unsigned_short");
		TypeNameTransformer tnTrans = new TypeNameTransformer();
		tnTrans.setReplacement(typeReplacements);
		tnTrans.transformModel(scsystem);

		// convert struct methods to global methods
		StructMethodTransformer smTrans = new StructMethodTransformer();
		smTrans.transformModel(scSystem);

		// convert sc_time2int
		SCTime2IntTransformer sc2intTrans = new SCTime2IntTransformer();
		sc2intTrans.transformModel(scSystem);

		// resolve socket RMI
		SocketCallTransformer scTrans = new SocketCallTransformer();
		scTrans.transformModel(scSystem);

		// convert C++ references to pointers
		CallByReferenceTransformer cbrTrans = new CallByReferenceTransformer();
		cbrTrans.transformModel(scSystem);
	}

	/**
	 * Calculates the amount of memory needed for the submitted data type for
	 * all PEQs in the system. This is currently Constants.GLOBAL_PEQ_SIZE *
	 * amount of PEQs for ints and for those data types used in the PEQ as a
	 * payload and 0 for all others.
	 * 
	 * @TODO mp: WARNING: This currently does not take instance count into
	 *       account and therefore might not allocate enough memory!
	 * 
	 * @param mca
	 * @param type
	 * @return
	 */
	public int calcPeqMemConsumptionForType(MemoryConsumptionAnalyzer mca,
			String type) {
		int ret = 0;
		for (SCPeq peq : mca.getPeqs()) {
			if (peq.getPayloadType().equals(type)
					|| peq.getPhaseType().equals(type)) {
				// get instances
				SCClass scclass = peq.getOwner();
				if (scclass != null) {
					List<SCClassInstance> instances = scclass.getInstances();
					for (SCClassInstance instance : instances) {
						// get size
						int size = PeqTransformer.getPeqSize(Constants
								.createDelimitedString(instance.getName(),
										peq.getName()));
						if (peq.getPayloadType().equals(type)) {
							ret += size;
						}
						if (peq.getPhaseType().equals(type)) {
							ret += size;
						}
					}
				}
			}
		}
		return ret;
	}

}
