/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.engine;

import java.util.LinkedList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.TABroadcastChannel;
import de.tub.pes.sc2uppaal.tamodel.TAChannel;
import de.tub.pes.sc2uppaal.tamodel.TAInteger;
import de.tub.pes.sc2uppaal.tamodel.TALocation;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.TATemplate;
import de.tub.pes.sc2uppaal.tamodel.TATransition;
import de.tub.pes.sc2uppaal.tamodel.TAVariable;
import de.tub.pes.sc2uppaal.tamodel.VariableConverter;
import de.tub.pes.sc2uppaal.tamodel.expressions.TARecvExpression;
import de.tub.pes.sc2uppaal.tamodel.expressions.TASendExpression;
import de.tub.pes.syscir.sc_model.SCClass;
import de.tub.pes.syscir.sc_model.SCFunction;
import de.tub.pes.syscir.sc_model.SCProcess;
import de.tub.pes.syscir.sc_model.SCSystem;
import de.tub.pes.syscir.sc_model.SCVariable;
import de.tub.pes.syscir.sc_model.expressions.FunctionCallExpression;
import de.tub.pes.syscir.sc_model.variables.SCClassInstance;
import de.tub.pes.syscir.sc_model.variables.SCEvent;
import de.tub.pes.syscir.sc_model.variables.SCPeq;

/**
 * Transformer for SystemC classes. Creates all templates that are needed for
 * this class.
 * 
 * @author Timm Liebrenz
 * 
 */
public class ClassTransformer {

	private static Logger logger = LogManager.getLogger(ClassTransformer.class
			.getName());

	private static final String DIV = Constants.PREFIX_DELIMITER;

	private final SCClass scclass;
	private final List<TATemplate> templates;
	private final Transformer mainTransformer;

	public ClassTransformer(SCClass scclass, Transformer mainTransformer) {
		this.scclass = scclass;
		this.templates = new LinkedList<TATemplate>();
		this.mainTransformer = mainTransformer;
	}

	/**
	 * Creates all necessary Templates for the SCClass this Transformer was
	 * created for. Uses the given prefix for the function Templates of the
	 * class.
	 * 
	 * @param prefix
	 * @param sc
	 * @param ta
	 */
	public void createClass(String prefix, SCSystem sc, TAModel ta) {

		String className = scclass.getName();
		String modPrefix = className + DIV;
		String fullPrefix = prefix + modPrefix;

		// TODO: Hierarchy

		// Members/local Variables
		// for (SCVariable scvar : scclass.getMembers()) {
		// }

		// Member Functions
		for (SCFunction fun : scclass.getMemberFunctions()) {
			// we obmit constructors as they are handled in a different way.
			if (fun.equals(scclass.getConstructor())) {
				// TODO Constructor
				continue;
			}

			TATemplate template;

			if (ta.isPorActive()) {
				template = new TATemplate(fullPrefix + fun.getName(),
						ta.getPor());
			} else {
				template = new TATemplate(fullPrefix + fun.getName());
			}

			ta.addTemplate(template);
			templates.add(template);

			FunctionTransformer funcTrans = new FunctionTransformer(fun,
					template);
			funcTrans.createFunction();
		}

		// // Events
		// for (SCEvent event : scclass.getEvents()) {
		// }
		//
		// // Class Instances
		// for (SCClassInstance classIns : scclass.getInstances()) {
		// }
		//
		// // PortSockets
		// for (SCPort portSocket : scclass.getPortsSockets()) {
		// }

		createSensitivityTemplates(ta);

		createTemplateHeader(sc);
	}

	/**
	 * This module creates the sensitivity template for each thread definition
	 * in the constructor. Please see documentation for the exact definition of
	 * sensitivity templates and their intention. As stated there, this method
	 * creates a template that emit a signal through a channel name sensitive
	 * each time an event on the sensitivity list is notified. To achieve that,
	 * it creates a transition for each entry of the sensitivity list, no matter
	 * of what type, connects it properly. All entries are parameters of this
	 * template and have to be connected in the instantiation/binding phase.
	 * 
	 * @param ta
	 */
	private void createSensitivityTemplates(TAModel ta) {

		List<SCProcess> moduleProcesses = scclass.getProcesses();
		int i = 0;
		for (SCProcess modProcess : moduleProcesses) {
			TATemplate sensitivity = new TATemplate(scclass.getName() + DIV
					+ "thread" + i + DIV + Constants.LOCAL_SENSITIVE_KEYWORD
					+ "Temp");

			TALocation start = new TALocation();
			sensitivity.addLocation(start);
			sensitivity.setInitLocation(start);
			TALocation end = new TALocation();
			end.setCommitted(true);
			sensitivity.addLocation(end);
			sensitivity.addTransition(new TATransition(end, start,
					new TASendExpression(null,
							Constants.LOCAL_SENSITIVE_KEYWORD)));
			sensitivity.addParameter(new TABroadcastChannel(
					Constants.LOCAL_SENSITIVE_KEYWORD));

			for (SCEvent entry : modProcess.getSensitivity()) {
				sensitivity
						.addParameter(new TABroadcastChannel(entry.getName()));
				sensitivity.addTransition(new TATransition(start, end,
						new TARecvExpression(null, entry.getName())));
			}
			ta.addTemplate(sensitivity);

			i++;
		}
	}

	/**
	 * This method creates the template header of the method templates. The
	 * header consists of reference parameters which represent all identifiers
	 * (simple TA Variables) that can possibly be used within the method
	 * template. More precisely, these identifiers are the simple members of
	 * this module and the elements of the complex members, the method signature
	 * variables of all methods that are visible. The
	 * template headers of all methods templates of a module are the same, so
	 * one actual list instances can be applied to all templates.
	 * 
	 * @param modules
	 */
	public void createTemplateHeader(SCSystem sc) {
		LinkedList<TAVariable> header = new LinkedList<TAVariable>();

		// sensitivity channel parameter and timeout event parameters
		header.add(new TABroadcastChannel(Constants.LOCAL_SENSITIVE_KEYWORD));
		header.add(new TABroadcastChannel(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD
				+ DIV + Constants.SC_EVENT_TEMPLATE_PARAM_WAIT));
		header.add(new TAChannel(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY));
		header.add(new TAChannel(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM));
		header.add(new TAInteger(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME));

		// Events
		for (SCEvent event : scclass.getEvents()) {
			String evName = event.getName();

			header.add(new TABroadcastChannel(evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT));
			header.add(new TAChannel(evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY));
			header.add(new TAChannel(evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM));
			header.add(new TAInteger(evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME));
		}

		// Ports and sockets
	/*	for (SCPort portsocket : scclass.getPortsSockets()) {
			String name = portsocket.getName();
			String type = portsocket.getType();
			SCPORTSCSOCKETTYPE conType = portsocket.getConType();
			if (portsocket instanceof SCPort) {
				SCPort port = portsocket;

				SCClass cl = null;
				// TODO
				if (conType == SCPORTSCSOCKETTYPE.SC_PORT) {
					cl = sc.getClassByName(type);
					// if no class was found, it could be a KnownType
					if (cl == null) {
						cl = mainTransformer.getKnownTypeTransformer()
								.getKnownTypeClassByName(type);
					}
				}
				if (SCPORTSCSOCKETTYPE.SC_FIFO.contains(conType)) {
					cl = mainTransformer.getKnownTypeTransformer()
							.getKnownTypeClassByName("sc_fifo");
				}
				if (SCPORTSCSOCKETTYPE.SC_SIGNAL.contains(conType)) {
					cl = mainTransformer.getKnownTypeTransformer()
							.getKnownTypeClassByName("sc_signal");
				}
				if (cl != null) {
					List<TAVariable> vars = ClassTransformer
							.getAllVisibleMembers(cl);

					addPrefixedList(header, vars, name + DIV);
				} else {
					logger.error(
							"createTemplateHeader could not find class for port {}",
							port);
				}
			}

		}*/
		// for(String socketName : sockets.keySet())
		// {
		// String tlmMod = sockets.get(socketName);
		// String socketPrefix = socketName + DIV; // + i + DIV;
		// if(this.maxSocketConnections.containsKey(socketName) &&
		// this.maxSocketConnections.get(socketName) > 0)
		// {
		// addPrefixedList(header,
		// this.listOfVarsAsArrays(modules.get(tlmMod).getAllVisibleMembers(),this.getMaxSocketConnection(socketName)
		// + 1), socketPrefix);
		// }
		// else
		// addPrefixedList(header, modules.get(tlmMod).getAllVisibleMembers(),
		// socketPrefix);
		// }

		// TODO
		// for (String peqName : peqs.keySet()) {
		// addPrefixedList(header, peqs.get(peqName).getAllVisibleParameters(),
		// peqName + DIV);
		// }

		// local members
		for (SCVariable scvar : scclass.getMembers()) {
			TAVariable tavar = null;
			if (scvar.isSCClassInstance()
					&& (scvar.isSCModule() || scvar.isChannel())) {
				// for modules and channels only add their member variables
				SCClassInstance inst = (SCClassInstance) scvar;
				SCClass iclass = inst.getSCClass();

				addPrefixedList(header, getAllVisibleMembers(iclass),
						inst.getName() + DIV);
			} else if (scvar instanceof SCPeq) {
				// do not add the PEQ, but rather the connection
				SCPeq peq = (SCPeq) scvar;
				header.addAll(PeqTransformer.getConnectionParamters(peq));
			} else {
				tavar = VariableConverter.convertVariable(scvar);
				if (tavar != null) {
					header.add(tavar);
				}
			}
		}

		// create the request-update channel if this class is a sc_prim_channel
		if (scclass.isPrimitiveChannel()) {
			header.add(new TABroadcastChannel(Constants.REQUEST_UPDATE_CHANNEL));
		}

		for (SCFunction fun : scclass.getMemberFunctions()) {
			if (fun.equals(scclass.getConstructor())) {
				// TODO constructor
				continue;
			}

			List<TAVariable> vars = FunctionTransformer
					.getNeededTmplParamsFromFunc(fun, "");
			header.addAll(vars);

			// check if function calls global function
			for (FunctionCallExpression funcExp : fun.getFunctionCalls()) {
				SCFunction calledFunc = funcExp.getFunction();
				if (sc.getGlobalFunctions().contains(calledFunc)) {
					header.addAll(FunctionTransformer
							.getNeededTmplParamsFromFunc(calledFunc, ""));
				}
			}
		}

		// for(ModuleInstance objInst : complexMembers) {
		// addPrefixedList(header, objInst.getModule().getAllVisibleMembers(),
		// objInst.getName() + DIV);
		// }

		for (TATemplate temp : templates) {
			header.addAll(temp.getParameters());
			temp.setParameters(header);
		}
	}

	/**
	 * Copies the TAVariables of the source list into the target list. The names
	 * of the copies are furthermore prefixed with the given prefix.
	 * 
	 * @param target
	 *            The list to copy the values into.
	 * @param source
	 *            The list to get the value out.
	 * @param prefix
	 *            For prefixing the names of the copies.
	 */
	private void addPrefixedList(List<TAVariable> target,
			List<TAVariable> source, String prefix) {
		for (TAVariable val : source) {
			target.add(val.createCopy(prefix));
		}
	}

	/**
	 * Returns all elements of this module which are visible in surrounding
	 * scopes and could therefore be referenced by, for instance, methods in
	 * other modules. Contains all simple variables, the signature variables of
	 * the member methods, and all visible members of the complex variables,
	 * prefixed with their name.
	 * 
	 * @return List of all variables that could be referenced in a surrounding
	 *         scope.
	 */
	public static List<TAVariable> getAllVisibleMembers(SCClass cl) {

		LinkedList<TAVariable> list = new LinkedList<TAVariable>();

		// we can see all members because we want to be able to copy
		for (SCVariable scvar : cl.getMembers()) {
			TAVariable tavar = VariableConverter.convertVariable(scvar);
			if (tavar != null) {
				list.add(tavar);
			}
		}

		for (SCFunction f : cl.getMemberFunctions()) {
			if (f.equals(cl.getConstructor())) {
				continue;
			}
			list.addAll(FunctionTransformer.getNeededTmplParamsFromFunc(f, ""));
		}

		// for(ModuleInstance objInst: complexMembers) {
		// List<TAVariable> objMembers =
		// objInst.getModule().getAllVisibleMembers();
		// for(TAVariable var: objMembers) {
		// list.add(var.createCopy(objInst.getName() + DIV));
		// }
		// }
		return list;
	}
}
