/*****************************************************************************
 *
 * Copyright (c) 2008-14, Joachim Fellmuth, Holger Gross, Florian Greiner, 
 * Bettina Hünnemeyer, Paula Herber, Verena Klös, Timm Liebrenz, 
 * Tobias Pfeffer, Marcel Pockrandt, Rolf Schröder
 * Technische Universitaet Berlin, Software Engineering for Embedded
 * Systems Group, Ernst-Reuter-Platz 7, 10587 Berlin, Germany.
 * All rights reserved.
 * 
 * This file is part of STATE (SystemC to Timed Automata Transformation Engine).
 * 
 * STATE is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * STATE is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with STATE.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Please report any problems or bugs to: state@pes.tu-berlin.de
 *
 ****************************************************************************/

package de.tub.pes.sc2uppaal.engine;

import java.util.EnumSet;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.tub.pes.sc2uppaal.tamodel.Constants;
import de.tub.pes.sc2uppaal.tamodel.SCConstants;
import de.tub.pes.sc2uppaal.tamodel.TABroadcastChannel;
import de.tub.pes.sc2uppaal.tamodel.TAChannel;
import de.tub.pes.sc2uppaal.tamodel.TAInteger;
import de.tub.pes.sc2uppaal.tamodel.TAModel;
import de.tub.pes.sc2uppaal.tamodel.TATemplate;
import de.tub.pes.sc2uppaal.tamodel.TATemplateInstance;
import de.tub.pes.sc2uppaal.tamodel.TAVariable;
import de.tub.pes.sc2uppaal.tamodel.VariableConverter;
import de.tub.pes.syscir.analysis.data_race_analyzer.AtomicBlock;
import de.tub.pes.syscir.engine.util.Pair;
import de.tub.pes.syscir.sc_model.SCClass;
import de.tub.pes.syscir.sc_model.SCConnectionInterface;
import de.tub.pes.syscir.sc_model.SCFunction;
import de.tub.pes.syscir.sc_model.SCMODIFIER;
import de.tub.pes.syscir.sc_model.SCPORTSCSOCKETTYPE;
import de.tub.pes.syscir.sc_model.SCPROCESSTYPE;
import de.tub.pes.syscir.sc_model.SCPort;
import de.tub.pes.syscir.sc_model.SCPortInstance;
import de.tub.pes.syscir.sc_model.SCProcess;
import de.tub.pes.syscir.sc_model.SCSocketInstance;
import de.tub.pes.syscir.sc_model.SCSystem;
import de.tub.pes.syscir.sc_model.SCVariable;
import de.tub.pes.syscir.sc_model.expressions.AccessExpression;
import de.tub.pes.syscir.sc_model.expressions.BinaryExpression;
import de.tub.pes.syscir.sc_model.expressions.ConstantExpression;
import de.tub.pes.syscir.sc_model.expressions.EventNotificationExpression;
import de.tub.pes.syscir.sc_model.expressions.Expression;
import de.tub.pes.syscir.sc_model.expressions.FunctionCallExpression;
import de.tub.pes.syscir.sc_model.expressions.SCVariableExpression;
import de.tub.pes.syscir.sc_model.variables.SCClassInstance;
import de.tub.pes.syscir.sc_model.variables.SCEvent;
import de.tub.pes.syscir.sc_model.variables.SCKnownType;
import de.tub.pes.syscir.sc_model.variables.SCPeq;

/**
 * Transformer for SystemC class instances. Handles the instantiation of all
 * members, the instantiation of all necessary function templates and the
 * binding of all variables and parameters. Also contains
 * ClassInstanceTransformer for all class instances which are members of the
 * instance represented by this transformer.
 * 
 * @author Timm Liebrenz, Paula Herber
 * 
 */
public class ClassInstanceTransformer extends HierarchicalTransformer {

	private static Logger logger = LogManager
			.getLogger(ClassInstanceTransformer.class.getName());

	private static final String DIV = Constants.PREFIX_DELIMITER;

	private final SCClassInstance scclassIns;
	private final SCClass scclass;
	private final Transformer mainTransformer;

	public ClassInstanceTransformer(SCClassInstance scclassIns, String prefix,
			Transformer mainTransformer,
			HierarchicalTransformer outerClInstTrans) {
		super(outerClInstTrans, prefix);

		this.scclassIns = scclassIns;
		this.scclass = scclassIns.getSCClass();
		this.mainTransformer = mainTransformer;

		mainTransformer.addClassInstTrans(this);
	}

	public SCClassInstance getClassInstance() {
		return scclassIns;
	}

	public void createChildClassInstanceTransformers() {
		for (SCVariable scvar : scclass.getMembers()) {
			String prefix = hPrefix + scclassIns.getName() + DIV;
			// SCKnownTypes
			if (scvar instanceof SCKnownType) {
				mainTransformer.getKnownTypeTransformer().transformKnownType(
						(SCKnownType) scvar, prefix, mainTransformer, this);
				// SCKnownTypes are also SCClassInstance and do not need to
				// created twice
				continue;
			}
			// Class Instances
			if (scvar instanceof SCClassInstance) {
				if (scvar.isSCModule() || scvar.isChannel()) {
					ClassInstanceTransformer modInsTrans = new ClassInstanceTransformer(
							(SCClassInstance) scvar, prefix, mainTransformer,
							this);
					// FIXME: Endless recursion.
					modInsTrans.createChildClassInstanceTransformers();
				}
			}
		}
		// Class Instances in member functions
		for (SCFunction memFunc : scclass.getMemberFunctions()) {
			String prefix = hPrefix + scclassIns.getName() + DIV
					+ memFunc.getName() + DIV;
			// use just the instance name as prefix for the constructor
			if (memFunc.equals(scclass.getConstructor())) {
				prefix = hPrefix + scclassIns.getName() + DIV;
			}
			for (SCVariable locVar : memFunc.getLocalVariables()) {
				// SCKnownTypes
				if (locVar instanceof SCKnownType) {
					mainTransformer.getKnownTypeTransformer()
							.transformKnownType((SCKnownType) locVar, prefix,
									mainTransformer, this);
					// SCKnownTypes are also SCClassInstance and do not need to
					// created twice
					continue;
				}
				// Class Instances
				if (locVar instanceof SCClassInstance) {
					ClassInstanceTransformer modInsTrans = new ClassInstanceTransformer(
							(SCClassInstance) locVar, prefix, mainTransformer,
							this);
					modInsTrans.createChildClassInstanceTransformers();
				}
			}
		}
	}

	@SuppressWarnings("unchecked")
	public void createClassInstance(SCSystem sc, TAModel ta) {

		String fullPrefix = hPrefix + scclassIns.getName() + DIV;

		// TODO
		// Hierarchy
		List<SCClass> classFathers = scclass.getInheritFrom();

		HashMap<String, String> params = new HashMap<String, String>();

		params = createParamMap(fullPrefix, "", "", classFathers, ta, sc);

		// get initial values of some members
		HashMap<SCVariable, Expression> initVars = getInitValForMembers(scclassIns);

		// create the simple members with proper prefix, put them into the TA
		// model
		for (SCVariable scvar : scclass.getMembers()) {
			// Class Instances
			if (scvar.isSCModule() || scvar.isChannel()) {
				// SCKnownTypes are also SCClassInstance
				ClassInstanceTransformer modInsTrans = this
						.getChildClassInstanceTransformerByInstance((SCClassInstance) scvar);
				modInsTrans.createClassInstance(sc, ta);
			} else if (scvar instanceof SCPeq) {
				// don't create the peq just yet - wait till it's called (see
				// instanciateAndBindMehtod)
				continue;
			} else { // also applies for non-sc_module-instances (e.g. structs)
				// add member vars for this class instance to model
				TAVariable tavar = VariableConverter.convertVariable(scvar);
				tavar.addPrefix(fullPrefix);
				Expression initValue = initVars.get(scvar);
				if (initValue != null) {
					tavar.setInitValue(initValue);
				}
				ta.addVariable(tavar);
			}
		}

		// TODO: Test this
		// Events
		for (SCEvent event : scclass.getEvents()) {
			String evName = fullPrefix + event.getName();
			TATemplate ev = ta.getTemplates().get(
					Constants.TEMPLATE_NAME_SC_EVENT);
			TATemplateInstance evInst = ev.getInstance(evName);
			evInst.connectParameter(Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY,
					evName + DIV + Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY);
			evInst.connectParameter(
					Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM, evName + DIV
							+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM);
			evInst.connectParameter(Constants.SC_EVENT_TEMPLATE_PARAM_WAIT,
					evName + DIV + Constants.SC_EVENT_TEMPLATE_PARAM_WAIT);
			evInst.connectParameter(
					Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME, evName + DIV
							+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME);
			ta.addTemplateInstance(evInst);

			ta.addVariable(new TABroadcastChannel(evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT));
			ta.addVariable(new TAChannel(evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY));
			ta.addVariable(new TAChannel(evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM));
			ta.addVariable(new TAInteger(evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME));
		}

		// TODO: Test this
		// if this is a primitive channel -
		// create the request-update logic
		// and all methods visible in the primitive channel (such that the
		// update method can use them?)
		handleRequestUpdate(scclassIns.getName() + DIV, hPrefix, ta, sc,
				params, classFathers);

		// // instantiate & bind peqs
		// for(String peqName : module.getPeqs().keySet()) {
		// // String instName = threadPrefix + peqName;
		// Peq p = this.instancePeqs.get(peqName);
		// createPeqFunctions(p, parents, sc, ta, params, instPrefix + peqName +
		// DIV);
		// }

		// Processes
		int i = 0;
		for (SCProcess modProcess : scclass.getProcesses()) {
			// TODO: instantiate and bind process templates
			HashMap<String, String> threadParams = (HashMap<String, String>) params
					.clone();
			String threadPrefix = "thread" + i + DIV;
			// get unique parameter map (different for threads, but same for all
			// templates in a thread)

			// create the sensitivity automaton for this thread, and the
			// corresponding variables
			createSensitivity(fullPrefix, modProcess, i, classFathers, ta, sc);

			threadParams.put(Constants.LOCAL_SENSITIVE_KEYWORD, fullPrefix
					+ threadPrefix + Constants.LOCAL_SENSITIVE_CHANNEL);
			createTimeoutEvent(fullPrefix + threadPrefix, threadParams, ta);

			createThreadStarter(modProcess, ta, threadPrefix, fullPrefix);

			SCFunction threadFunction = modProcess.getFunction();

			// instantiate and bind the thread method and recursively all
			// functions called by this method
			instantiateAndBindMethod(threadFunction,
					hPrefix + scclassIns.getName() + DIV + "thread" + i + DIV,
					hPrefix, threadParams, null, classFathers, sc, ta, false,
					-1, modProcess);

			i++;
		}

		// Class Instances
		// for (SCClassInstance classIns : scclass.getInstances()) {
		// // TODO
		// }

		// Class Instances in member functions, needed for hierarchical modules
		for (SCFunction memFunc : scclass.getMemberFunctions()) {
			for (SCVariable locVar : memFunc.getLocalVariables()) {
				// Class Instances
				if (locVar instanceof SCClassInstance) {
					// SCKnownTypes are also SCClassInstance
					ClassInstanceTransformer modInsTrans = this
							.getChildClassInstanceTransformerByInstance((SCClassInstance) locVar);
					modInsTrans.createClassInstance(sc, ta);
				}
			}
		}


		// PortSockets
		// for (SCPort portSocket : scclass.getPortsSockets()) {
		// ?????????????
		// TODO
		// }
	}

	/**
	 * 
	 * @return HashMap with variables and their initial expression
	 */
	public static HashMap<SCVariable, Expression> getInitValForMembers(
			SCClassInstance scclassIns) {
		HashMap<SCVariable, Expression> initVars = new HashMap<SCVariable, Expression>();

		// TODO: This has nothing to do here ... Oo
		SCClass scclass = scclassIns.getSCClass();
		if (scclass.getName().equals("sc_clock")) {
			if (scclassIns.getInitialValueCount() == 3) {
				List<Expression> initExpr = scclassIns.getDeclaration()
						.getInitialValues();
				// convert the last two parameter to one value
				int periode = SCConstants.getTimeUnits(initExpr.get(1)
						.toStringNoSem(), initExpr.get(2).toStringNoSem());
				initExpr.remove(2);
				initExpr.remove(1);
				initExpr.add(new ConstantExpression(null, Integer
						.toString(periode)));
			}
		}
		// put all constant members into the map
		for (SCVariable var : scclass.getMembers()) {
			if (var.isConst()) {
				if (var.hasInitialValue()) {
					initVars.put(var, var.getDeclaration()
							.getFirstInitialValue());
				}
				// TODO: The handling of constants in sc2uppaal will be changed
				// and then this should be removed.
				else {
					// get the initial value from the constructor
					if (scclass.getConstructor() != null) {
						for (Expression expr : scclass.getConstructor()
								.getAllExpressions()) {
							// only check BinaryExpression
							if (expr instanceof BinaryExpression) {
								BinaryExpression be = (BinaryExpression) expr;
								if (be.getLeft().toStringNoSem()
										.equals(var.getName())) {
									initVars.put(var, be.getRight());
								}
							}
						}
					}
				}
			}
		}
		return initVars;
	}

	/**
	 * Creates an instance of the sensitivity template of the given thread and
	 * connects the parameters (corresponding to entries of the sensitivity
	 * list) with actual event wait channels. For that, the event that
	 * represents the sensitivity list entry, has to be computed from the list
	 * entry, which can be one of the following:
	 * <ul>
	 * <li>The entry is the name of an event that is visible in the module
	 * scope. Then the parameter can be connected directly with the wait channel
	 * of the event instance.</li>
	 * <li>The entry is the name of a local primitive channel. The
	 * default_event() method of that channel delivers the default event name,
	 * and the wait channel of that event can then be connected with the
	 * sensitivity list entry.</li>
	 * <li>The entry is a port name, and the port type is a primitive channel.
	 * In that case the channel that is connected to the port has to be found.
	 * This could be a primitive channel instance in any of the higher hierarchy
	 * levels. The list entry can the be connected with the wait channel of the
	 * default event of that channel instance.</li>
	 * <li>The entry is an event that is a member of a port. In that case, the
	 * channel, that is connected with the port, has to be found. The event,
	 * which is a member of that channel, can then be connected with the list
	 * entry.
	 * </ul>
	 * 
	 * @param instPrefix
	 *            The prefix of the current module instance, which the thread is
	 *            a part of.
	 * @param thread
	 *            The current thread object, to get the sensitivity list from.
	 * @param threadCnt
	 *            The number of the thread within the module.
	 * @param ta
	 *            TA model to insert the created objects.
	 * @param sc
	 *            SC model to get channel information from
	 * @param parents
	 */
	private void createSensitivity(String instPrefix, SCProcess process,
			int threadCnt, List<SCClass> classFathers, TAModel ta, SCSystem sc) {
		String sensTempName = scclass.getName() + DIV + "thread" + threadCnt
				+ DIV + Constants.LOCAL_SENSITIVE_KEYWORD + "Temp";
		String sensInstName = instPrefix + "thread" + threadCnt + DIV
				+ Constants.LOCAL_SENSITIVE_KEYWORD;
		String sensChanName = instPrefix + "thread" + threadCnt + DIV
				+ Constants.LOCAL_SENSITIVE_CHANNEL;

		TATemplateInstance instance = new TATemplateInstance(ta.getTemplates()
				.get(sensTempName), sensInstName);

		ta.addVariable(new TABroadcastChannel(sensChanName));
		instance.connectParameter(Constants.LOCAL_SENSITIVE_KEYWORD,
				sensChanName);

		for (SCEvent sensEvent : process.getSensitivity()) {
			String sEventName = sensEvent.getName();
			// is entry a port?
			SCConnectionInterface psi = scclassIns
					.getPortSocketInstanceByName(sEventName);
			if (psi instanceof SCPortInstance) {
				SCPortInstance psInst = (SCPortInstance) psi;
				// TODO: revise this, maybe it is necessary to go up more
				// hierarchy levels
				List<SCKnownType> channelList = psInst.getChannels();
				SCClassInstance pClInst = channelList.get(0);
				// find Transformer for this instance
				HierarchicalTransformer pClInsTrans = getParentTransformer()
						.getChildClassInstanceTransformerByInstance(pClInst);

				for (SCEvent pEvent : pClInst.getSCClass().getEvents()) {
					instance.connectParameter(
							sEventName,
							pClInsTrans.getHierarchyPrefix()
									+ pClInst.getName() + DIV
									+ pEvent.getName() + DIV
									+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT);
				}
			} else if (this.scclass.getEventByName(sEventName) != null) {
				instance.connectParameter(sEventName, instPrefix + sEventName
						+ Constants.PREFIX_DELIMITER
						+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT);
			}
			// TODO: implement more
			// is it a local prim channel?
			// else if(locInst != null){
			// String eventName = locInst.getModule().getDefaultEvent();
			// instance.connectParameter(entry, instPrefix + entry + DIV +
			// eventName + DIV + Constants.SC_EVENT_TEMPLATE_PARAM_WAIT);
			// // some visible event
			// } else if(module.isEvent(entry, sc)) {
			// instance.connectParameter(entry, instPrefix + entry + DIV +
			// Constants.SC_EVENT_TEMPLATE_PARAM_WAIT);
			// // direct port->event connection
			// } else {
			// // vielleicht besser?
			// String[] prefixes =
			// entry.split(Constants.INTERNAL_STRING_DELIMITER);
			//
			// if(prefixes.length == 2) {
			// String channel = this.getPortConnections().get(prefixes[0]);
			// String fineEntry =
			// entry.replaceAll(Constants.INTERNAL_STRING_DELIMITER, DIV);
			// instance.connectParameter(fineEntry, channel + DIV + prefixes[1]
			// + DIV + Constants.SC_EVENT_TEMPLATE_PARAM_WAIT);
			// } else {
			// //TODO events von Kan"alen die weiter weg sind machen....
			// System.out.println("unexpected behaviour: event list entry has more than one prefix "
			// + prefixes.length + " [" + entry + "]");
			// }
			// }
		}
		ta.addTemplateInstance(instance);
	}

	/**
	 * Creates the timeout event that is created for each thread. To achieve
	 * that, it inserts the four global event variables and the event template
	 * into the TA model. The name of the event is constructed from the current
	 * thread prefix and a predefined event name.
	 * 
	 * @param threadPrefix
	 *            The prefix of the event instance name
	 * @param threadParams
	 *            Parameters map to insert the newly created identifiers.
	 * @param ta
	 *            TA model to insert the event
	 */
	private void createTimeoutEvent(String threadPrefix,
			HashMap<String, String> threadParams, TAModel ta) {
		String timeoutInstName = threadPrefix
				+ Constants.LOCAL_TIMEOUT_EVENT_KEYWORD;

		TATemplateInstance instance = new TATemplateInstance(ta.getTemplates()
				.get(Constants.TEMPLATE_NAME_SC_EVENT), timeoutInstName);

		ta.addVariable(new TABroadcastChannel(timeoutInstName + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT));
		ta.addVariable(new TAChannel(timeoutInstName + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY));
		ta.addVariable(new TAChannel(timeoutInstName + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM));
		ta.addVariable(new TAInteger(timeoutInstName + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME));

		instance.connectParameter(Constants.SC_EVENT_TEMPLATE_PARAM_WAIT,
				timeoutInstName + DIV + Constants.SC_EVENT_TEMPLATE_PARAM_WAIT);
		instance.connectParameter(Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY,
				timeoutInstName + DIV
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY);
		instance.connectParameter(Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM,
				timeoutInstName + DIV
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM);
		instance.connectParameter(
				Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME, timeoutInstName
						+ DIV + Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME);

		threadParams.put(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT, timeoutInstName + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT);
		threadParams.put(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY, timeoutInstName
				+ DIV + Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY);
		threadParams.put(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM, timeoutInstName
				+ DIV + Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM);
		threadParams.put(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
				+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME,
				timeoutInstName + DIV
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME);

		ta.addTemplateInstance(instance);
	}

	/**
	 * Inserts a new instance of a thread starter template into the TA model.
	 * Which thread starter template is instantiated depends on the type of
	 * thread and initialization phase setting. See documentation for details.
	 * The name of the instance consists of the thread prefix (see create
	 * method) and the suffix 'starter'. The arguments to the new instance are
	 * the threads' sensitivity channel(only for method threads) and the control
	 * channel of the threads' start method.
	 * 
	 * 
	 * @param thread
	 *            The thread to create a starter for
	 * @param ta
	 *            TA model to insert new thread starter
	 * @param threadPrefix
	 *            The prefix to use for all identifiers.
	 */
	private void createThreadStarter(SCProcess process, TAModel ta,
			String threadPrefix, String localPrefix) {
		SCFunction func = process.getFunction();
		SCPROCESSTYPE proType = process.getType();

		TATemplate starter = null;
		EnumSet<SCMODIFIER> modifier = process.getModifier();
		boolean initializes = !((modifier != null) && (modifier
				.contains(SCMODIFIER.DONTINITIALIZE)));

		switch (proType) {
		case SCTHREAD:
			if (initializes) {
				starter = ta.getTemplates().get(
						Constants.TEMPLATE_NAME_THREAD_STARTER);
			} else {
				starter = ta.getTemplates().get(
						Constants.TEMPLATE_NAME_THREAD_STARTER_NO_INIT);
			}
			break;
		case SCMETHOD:
			if (initializes) {
				starter = ta.getTemplates().get(
						Constants.TEMPLATE_NAME_METHOD_STARTER);
			} else {
				starter = ta.getTemplates().get(
						Constants.TEMPLATE_NAME_METHOD_STARTER_NO_INIT);
			}
			break;
		case SCCTHREAD:
			// TODO
			// Warning/Error message
		default:
			return;
		}

		boolean consumesTime = func.getConsumesTime();

		TATemplateInstance sInstance = new TATemplateInstance(starter,
				localPrefix + threadPrefix + "starter");
		sInstance.connectParameter(Constants.LOCAL_SENSITIVE_KEYWORD,
				localPrefix + threadPrefix + Constants.LOCAL_SENSITIVE_CHANNEL);
		// if the function associated with the thread consumes no time, its
		// instantiation also doesn't use the threadPrefix
		if (!consumesTime) {
			threadPrefix = localPrefix;
		} else {
			threadPrefix = localPrefix + threadPrefix;
		}

		// This prefix extension is used when the flag USE_CLASSNAME_IN_PREFIX
		// is set, it is used in the prefix to add the name of the class in
		// front of function variables of that class
		String classPrefix = "";
		if (Constants.USE_CLASSNAME_IN_PREFIX) {
			classPrefix = scclass.getName() + DIV;
		}

		sInstance.connectParameter(Constants.LOCAL_FUNCTION_CTRL_KEYWORD,
				threadPrefix + classPrefix + func.getName() + DIV
						+ Constants.LOCAL_FUNCTION_CTRL_KEYWORD);

		if (ta.isPorActive()) {

			if (ta.getPor().getAtomicBlockCounter() > 0
					&& process.getName().equals("run")) {
				return;
			}
			sInstance.connectParameter("cln", ta.getPor()
					.getAtomicBlockCounter() + "");

			for (AtomicBlock aB : ta.getPor().dra.getAtomicBlocks()) {
				if (aB.getExpression() == null && aB.getProcess() == process
						&& aB.getTopLevelClassInstance() == scclassIns) {
					aB.setProcessNoInit(!initializes);
					aB.setAtomicBlockNumber(ta.getPor().getAtomicBlockCounter());
					ta.getPor().setAtomicBlockCounter(
							ta.getPor().getAtomicBlockCounter() + 1);
				}
			}

		}

		ta.addTemplateInstance(sInstance);
	}

	private HashMap<String, String> createParamMap(String prefix, String localPrefix, 
			String threadPrefix, List<SCClass> parents, TAModel ta, SCSystem sc) {

		// parents.addFirst(this);

		HashMap<String, String> params = new HashMap<String, String>();

		// create handles for the simple members with proper prefix, put them
		// into the param maps
		for (SCVariable var : scclass.getMembers()) {
			// do not add modules and channels
			if (var.isNotSCModule() && var.isNotChannel()) {
				params.put(var.getName(), prefix + var.getName());
			}
		}

		// create handle for the request-update channel if this class is a
		// sc_prim_channel
		if (scclass.isPrimitiveChannel()) {
			params.put(Constants.REQUEST_UPDATE_CHANNEL, prefix
					+ Constants.REQUEST_UPDATE_CHANNEL);
		}

		// Events
		for (SCEvent event : scclass.getEvents()) {
			String evName = event.getName();
			params.put(evName + DIV + Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY,
					prefix + evName + DIV
							+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY);
			params.put(evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM, prefix
					+ evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM);
			params.put(evName + DIV + Constants.SC_EVENT_TEMPLATE_PARAM_WAIT,
					prefix + evName + DIV
							+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT);
			params.put(evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME, prefix
					+ evName + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME);
		}

		// // create complex members, collect template parameter connections
		// // TODO: buggy -> cope with inner modules, classes and structs
		// for(ModuleInstance modInst : module.getComplexMembers()) {
		// //the module may contain another sc_module (f.e. an inner module, a
		// port...)
		// HashMap<String,String> localParams = modInst.createParamMap(prefix,
		// modInst.getName() + DIV, parents, ta, sc);
		// concatHashMaps(params, localParams);
		// }

		// for(String peqName : module.getPeqs().keySet()) {
		// for(TAVariable var :
		// module.getPeq(peqName).getAllVisibleParameters()) {
		// params.put(peqName + DIV + var.getName(), this.getName() + DIV +
		// peqName + DIV + var.getName());
		// }
		// }

		for (SCFunction fun : scclass.getMemberFunctions()) {
			if (fun.equals(scclass.getConstructor())) {
				// TODO Constructor
				continue;
			}

			String tmpPrefix;
			if (!(fun.getConsumesTime())) {
				// use modinst prefix, for hierarchical classes add localPrefix, too
				tmpPrefix = scclassIns.getName() + DIV + localPrefix;
			} else {
				tmpPrefix = threadPrefix;
			}

			for (TAVariable var : FunctionTransformer
					.getNeededTmplParamsFromFunc(fun, "")) {
				if (var != null) {
					params.put(var.getName(), tmpPrefix + var.getName());
				}
			}
		}

		// TODO
		// add public port members to template parameters (the instances have
		// been
		// created somewhere else, this is just the connection)
		// the port may be connected to a module or to a class
		// for(String port: module.getPorts().keySet()) {
		// Module channel = sc.getModule(module.getPorts().get(port));
		// if (channel != null)
		// {
		// String connect = this.computeChannelName(port,
		// portConnections.get(port), parents, sc);
		// for(TAVariable var: channel.getSimpleMembers()) {
		// params.put(port + DIV + var.getName(), connect + DIV +
		// var.getName());
		// }
		// }
		// else
		// {
		// CClass cc = sc.getClass(module.getPorts().get(port));
		// String connect = this.computeChannelName(port,
		// portConnections.get(port), parents, sc);
		// for(TAVariable var: cc.getMembers()) {
		// params.put(port + DIV + var.getName(), connect + DIV +
		// var.getName());
		// }
		// }
		// }

		// TODO
		// for every socket: add params to the param map
		// so far: only TLM sockets supported
		// name scheme: sMod, then my socket name
		// this creates handles to methods that are reachable over a socket!
		// for(String socket: this.getSocketConnections().keySet()) {
		//
		// LinkedList<Module> sModules = new LinkedList<Module>();
		// for (String channel : socketConnections.get(socket))
		// {
		// Module sModule = sc.getInstance(channel).getModule();
		//
		// if(!sModules.contains(sModule))
		// {
		// sModules.add(sModule);
		// ModuleInstance sModInst = sc.getInstance(channel);
		//
		// String socketPrefix = prefix;
		//
		// for(Function tlmFun :
		// sc.getModules().get("tlm_fw_bw_if").getMethods())
		// {
		// Function myFun = sModule.getMethodByName(tlmFun.getName());
		//
		// if(myFun != null)
		// {
		//
		// if(!myFun.getConsumesTime())
		// {
		// socketPrefix = sModInst.getName() + DIV;
		// }
		// else
		// {
		// socketPrefix = prefix;
		// }
		// for(TAVariable var : myFun.getAsTAVariables())
		// {
		// params.put(socket + DIV + var, socketPrefix + var);
		// }
		// }
		// }
		// } // for each TLM function
		// } // for each instance bound to this socket
		// } // for each socket

		// parents.removeFirst();
		return params;
	}

	/**
	 * Creates instances of all local methods and port methods for the update
	 * phase. As the update phase of the scheduler is a execution basis like the
	 * threads are, separate method template instances are required to prevent
	 * deadlocks. These method instances are prefixed with the instance prefix
	 * instead of the thread prefix.
	 * <p>
	 * Update cycles can only exist in primitive channels, and they are only
	 * called after requesting updates. Hence the update cycle method instances
	 * only have to be created when a call to <code>request_update()</code> can
	 * be found in at least one of the module's member methods.
	 * 
	 * @param instPrefix
	 *            The prefix for all variables and TA's created by this method
	 * @param ta
	 *            TA model to insert into
	 * @param sc
	 * @param params
	 *            Parameter/argument mappings to use for the new TA instances
	 * @param parents
	 *            List of module instances to reflects the higher level of
	 *            hierarchy
	 */
	@SuppressWarnings("unchecked")
	private void handleRequestUpdate(String instPrefix, String localPrefix,
			TAModel ta, SCSystem sc, HashMap<String, String> params,
			List<SCClass> parents) {
		// if it is a primitive channel -
		// create the request-update logic

		String fullPrefix = localPrefix + instPrefix;

		if (scclass.isPrimitiveChannel()) {
			// create the request-update channel for this sc_prim_channel
			ta.addVariable(new TABroadcastChannel(fullPrefix
					+ Constants.REQUEST_UPDATE_CHANNEL));

			SCFunction fun = scclass.getMemberFunctionByName("update");
			// see if there is an update method
			if (fun == null) {
				// TODO: is this possible?
				// TODO: implement null handling

				logger.error("update function of {}  not found",
						scclass.getName());
				return;

				// //if not, create an empty one
				// module.createMethod(ta, "update");
				//
				// TATemplate updateFun = ta.getTemplates().get(module.getName()
				// + DIV + "update");
				// TALocation init = updateFun.createLocation();
				// updateFun.setInitLocation(init);
			}

			String updatePrefix;
			// delete prefix if the update function uses no time
			if (!fun.getConsumesTime()) {
				updatePrefix = localPrefix + scclassIns.getName() + DIV;
			} else {
				updatePrefix = fullPrefix;
			}

			TATemplate update = ta.getTemplates().get(
					Constants.TEMPLATE_NAME_UPDATE_STARTER);

			TATemplateInstance updateInst = update.getInstance(fullPrefix
					+ Constants.TEMPLATE_NAME_UPDATE_STARTER);
			ta.addTemplateInstance(updateInst);

			updateInst.connectParameter(Constants.REQUEST_UPDATE_CHANNEL,
					fullPrefix + Constants.REQUEST_UPDATE_CHANNEL);

			// This prefix extension is used when the flag
			// USE_CLASSNAME_IN_PREFIX is set, it is used in the prefix to add
			// the name of the class in front of function variables of that
			// class
			String classPrefix = "";
			if (Constants.USE_CLASSNAME_IN_PREFIX) {
				classPrefix = scclass.getName() + DIV;
			}

			updateInst.connectParameter(Constants.UPDATE_CTRL_CHANNEL,
					updatePrefix + classPrefix + "update" + DIV + "ctrl");

			if (ta.isUpdateRequestDeterminism()) {
				if (!ta.getUrD().contains(
						fullPrefix + Constants.TEMPLATE_NAME_UPDATE_STARTER)) {

					updateInst.connectParameter(
							Constants.REQUEST_UPDATE_NUMBER, ta.getUrD()
									.getUpdateRequests().size()
									+ "");
					ta.getUrD()
							.getUpdateRequests()
							.add(fullPrefix
									+ Constants.TEMPLATE_NAME_UPDATE_STARTER);

				} else {
					updateInst
							.connectParameter(
									Constants.REQUEST_UPDATE_NUMBER,
									ta.getUrD()
											.getPos(fullPrefix
													+ Constants.TEMPLATE_NAME_UPDATE_STARTER)
											+ "");
				}
			}

			// TAVariable update_ctrl_channel =
			// ta.addVariable(update_ctrl_channel);

			HashMap<String, String> updateParams = (HashMap<String, String>) params
					.clone();

			// connect sensitivity param to dummy-no-sensitivity-channel
			updateParams.put(Constants.LOCAL_SENSITIVE_KEYWORD,
					Constants.DUMMY_BROADCAST_CHANNEL);
			updateParams.put(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT,
					Constants.DUMMY_BROADCAST_CHANNEL);
			updateParams.put(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY,
					Constants.DUMMY_CHANNEL);
			updateParams.put(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM,
					Constants.DUMMY_CHANNEL);
			updateParams.put(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
					+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME,
					Constants.DUMMY_INTEGER);

			// create the update method
			instantiateAndBindMethod(fun, updatePrefix, localPrefix,
					updateParams, null, parents, sc, ta, false, -1, null);
			// TODO: more hierarchy? or update documentation

		}
	}

	/**
	 * Recursively instantiates and binds a given Function fun.
	 * 
	 * @param fun
	 * @param threadPrefix
	 *            : this prefix is the prefix of the thread (mod_inst#threadi#)
	 *            and is used for variable bindings if the function is time
	 *            consuming, otherwise it is ignored
	 * @param localPrefix
	 *            : this additional prefix is always used and also passed to the
	 *            next recursive call, i.e. it can be used for expressing
	 *            hierarchy
	 * @param params
	 *            : these parameters are used and new parameters are added to
	 *            this map
	 * @param threadparams
	 * @param parents
	 * @param sc
	 * @param ta
	 * @param noopt
	 *            : if noopt is true, timeless methods will not be optimized,
	 *            i.e. a prefixed version will be instantiated
	 * @param multiSocketIndex
	 */
	private void instantiateAndBindMethod(SCFunction fun, String threadPrefix,
			String localPrefix, HashMap<String, String> params,
			Map<String, String> threadParams, List<SCClass> parents,
			SCSystem sc, TAModel ta, boolean noopt, int multiSocketIndex,
			SCProcess modProcess) {

		// TODO: Is this enough?
		TATemplate funTemp = FunctionTransformer.getTemplate(fun);
		if (funTemp == null) {
			return;
		}

		// This prefix extension is used when the flag USE_CLASSNAME_IN_PREFIX
		// is set, it is used in the prefix to add the name of the class in
		// front of function variables of that class
		String classPrefix = "";
		if (Constants.USE_CLASSNAME_IN_PREFIX) {
			classPrefix = scclass.getName() + DIV;
		}

		// TODO: das Problem wenn ich hier nur den Modulnamen verwende ist dass
		// ein Modul zwei gleichnamige Funktionen haben kann,
		// von denen eine bspw. über einen Port oder ein socket sichtbar ist.
		// Dies erzeugt dann gleichnamige Variablen.
		String fullPrefix;
		if (!fun.getConsumesTime() && !noopt) {
			// if the function consumes no time, it is only generated once with
			// the module instance and maybe a struct prefix
			fullPrefix = localPrefix + scclassIns.getName() + DIV; // module.getName()
																	// + DIV;
		} else {
			fullPrefix = threadPrefix;
		}

		// this function wasn't created before
		if (!ta.getSystemDeclaration().containsKey(
				fullPrefix + classPrefix + funTemp.getName())) {

			// create parameters for this function
			for (TAVariable funVar : FunctionTransformer
					.getNeededTmplParamsFromFunc(fun, "")) {

				if (multiSocketIndex >= 0) {
					// TODO: implement multisocket (TAVariable.asArray needed)
					logger.warn("multisocket not yet implemented");
					// // if the current function is bound to a multisocket, we
					// need special prefixes and array parameters
					// // note that the prefix does NOT contain the module
					// instance name itself
					// // because the same array is used for multiple instances
					// that are bound to the same multisocket
					// String tmpPrefix;
					// if(fun.getConsumesTime()) {
					// tmpPrefix = prefix + extPrefix;
					// }
					// else {
					// tmpPrefix = extPrefix;
					// }
					// TAVariable varAsArray =
					// funVar.createCopy(tmpPrefix).asArray(multiSocketIndex);
					// params.put(funVar.getName(), varAsArray.getName() + "[" +
					// multiSocketIndex + "]");
				} else {
					ta.addVariable(funVar.createCopy(fullPrefix + classPrefix));
					params.put(funVar.getName(), fullPrefix + classPrefix
							+ funVar.getName());
				}
			}

			TATemplateInstance inst = null;
			// create the template. if the current function is bound to a
			// multisocket,
			// we need to add the module instance name to the template instance
			// name
			// because all instances bound to the same multisocket use the same
			// prefix
			// but have to be instantiated multiple times
			if (multiSocketIndex >= 0) {
				// TODO: implement multisocket
				inst = funTemp.getInstance(fullPrefix + classPrefix
						+ scclassIns.getName() + DIV + funTemp.getName());
			} else {
				inst = funTemp.getInstance(fullPrefix + classPrefix
						+ funTemp.getName());
			}
			inst.setParamConnections(params);

			// TODO: add struct to parameter
			// if(struct != null) {
			// inst.connectParameter(Constants.INSTANCE_NAME, this.name + DIV +
			// struct.getName());
			// }

			if (ta.isPorActive()) {

				inst.connectParameter("cln", ta.getPor()
						.getAtomicBlockCounter() + "");
				int offset = 0;
				for (FunctionCallExpression fc : fun.getFunctionCalls()) {
					if (fc.getFunction().getName().equals("wait")) {
						for (AtomicBlock aB : ta.getPor().dra.getAtomicBlocks()) {
							if (fc != null
									&& fc == aB.getExpression()
									&& this.scclassIns == aB
											.getWaitLevelClassInstance()
									&& modProcess == aB.getProcess()) {
								for (Pair<Integer, Expression> p : ta.getPor()
										.getWaitOffsetList()) {
									if (p.getSecond() == fc) {
										aB.setAtomicBlockNumber(ta.getPor()
												.getAtomicBlockCounter()
												+ p.getFirst());
									}
								}
								offset++;
							}
						}
					}
				}
				ta.getPor().setAtomicBlockCounter(
						ta.getPor().getAtomicBlockCounter() + offset);
			}

			ta.addTemplateInstance(inst);

			boolean functionCallFound;
			// recursively create all methods that are called by this method
			for (FunctionCallExpression fc : fun.getFunctionCalls()) {

				functionCallFound = false;

				// TODO: is this enough?
				// don't handle wait calls
				if (fc.getFunction().getName().equals("wait")) {
					continue;
				}

				LinkedList<Expression> access = new LinkedList<Expression>();
				Expression currentParentExpr = fc.getParent();

				while (currentParentExpr instanceof AccessExpression) {
					AccessExpression acExpr = (AccessExpression) currentParentExpr;
					access.add(acExpr.getLeft());

					currentParentExpr = currentParentExpr.getParent();
				}

				// check for global functions
				if (sc.getGlobalFunction(fc.getFunction().getName()) != null) {
					functionCallFound = true;
					SCFunction f = sc.getGlobalFunction(fc.getFunction()
							.getName());
					instantiateAndBindMethod(f, fullPrefix, "", params,
							threadParams, parents, sc, ta, noopt, -1,
							modProcess);
				}
				// it is a member method of the current module
				else if (access.isEmpty()) {
					SCFunction f = scclass.getMemberFunctionByName(fc
							.getFunction().getName());
					if (f != null) {
						functionCallFound = true;

						instantiateAndBindMethod(f, fullPrefix, localPrefix,
								params, threadParams, parents, sc, ta, noopt,
								-1, modProcess);
					}
				} else {
					String funName = fc.getFunction().getName();
					// TODO handle the rest of the access list
					Expression callThis = access.removeFirst();
					String funPrefix = callThis.toStringNoSem();

					if (callThis instanceof SCVariableExpression
							&& ((SCVariableExpression) callThis).getVar() instanceof SCClassInstance) {
						// it is a member of an inner class
						// get the inner class instance and the appropriate
						// transformer
						SCClassInstance innerClassInst = (SCClassInstance) ((SCVariableExpression) callThis)
								.getVar();
						SCClass innerClass = innerClassInst.getSCClass();
						ClassInstanceTransformer innerClassInstTrans = this
								.getChildClassInstanceTransformerByInstance(innerClassInst);

						// check if the functioncall is a function of the inner
						// class
						SCFunction f = innerClass.getMemberFunctionByName(fc
								.getFunction().getName());
						if (f != null) {
							HashMap<String, String> newParams = new HashMap<String, String>();
							newParams = innerClassInstTrans.createParamMap(
									innerClassInstTrans.getHierarchyPrefix()
											+ innerClassInst.getName() + DIV, innerClassInst.getType() + DIV,
									fullPrefix + funPrefix + DIV, parents, ta,
									sc);
							copyTimeoutAndSensitivity(newParams, params);

							innerClassInstTrans.instantiateAndBindMethod(
									fc.getFunction(), fullPrefix
											+ innerClassInst.getName() + DIV,
									localPrefix, newParams, threadParams,
									parents, sc, ta, noopt, -1, modProcess);

							// here the error happens
							concatHashMapsPref(params, newParams, funPrefix
									+ DIV);

							functionCallFound = true;
						}
					}

					// TODO: port
					// if the prefix is a port, create the necessary parameter
					// bindings
					// and call the instantiateAndBindMethod of the module which
					// is bound to the port

					if (scclassIns.getPortSocketInstanceByName(funPrefix) != null) {
						String portName = funPrefix;
						String portPrefix = funPrefix + DIV;
						SCConnectionInterface psi = scclassIns
								.getPortSocketInstanceByName(portName);
						if (psi instanceof SCPortInstance) {
							SCPortInstance pi = (SCPortInstance) psi;
							List<SCClassInstance> listClassInst = pi
									.getModuleInstances();
							List<SCKnownType> listChannels = pi.getChannels();
							SCClassInstance pClassInst = null;

							SCPort port = pi.getPortSocket();
							SCPORTSCSOCKETTYPE conType = port.getConType();

							// TODO
							if (conType == SCPORTSCSOCKETTYPE.SC_PORT) {
								if (listClassInst.size() < 1) {
									logger.error(
											"instantiateAndBind: could not find class connected to port {}",
											portName);
									return;
								}
								if (listClassInst.size() > 1) {
									logger.error("instantiateAndBind: multisocket not yet implemented");
									return;
								}
								pClassInst = listClassInst.get(0);
							}
							if (SCPORTSCSOCKETTYPE.SC_FIFO.contains(conType)) {
								if (listChannels.size() < 1) {
									logger.error(
											"instantiateAndBind: could not find class connected to fifo port {}",
											portName);
									return;
								}
								pClassInst = listChannels.get(0);
							}
							if (SCPORTSCSOCKETTYPE.SC_SIGNAL.contains(conType)) {
								if (listChannels.size() < 1) {
									logger.error(
											"instantiateAndBind: could not find class connected to signal port {}",
											portName);
									return;
								}
								pClassInst = listChannels.get(0);
							}
							if (pClassInst == null) {
								logger.error(
										"instantiateAndBind: could not find class connected to port {}",
										portName);
								return;
							}

							// create the necessary parameters. in particular,
							// copy
							// the sensitivity and timeout-event parameters
							// of the current thread to the other module
							// instance
							HashMap<String, String> newParams = new HashMap<String, String>();

							// TODO: Maybe it is necessary to go up more
							// hierarchy
							// levels
							ClassInstanceTransformer pClassInsTransf = getParentTransformer()
									.getChildClassInstanceTransformerByInstance(
											pClassInst);
							if (pClassInsTransf == null) {
								logger.error(
										"instantiateAndBind could not find ClassInstanceTransformer for instance {}",
										pClassInst);
								return;
							}
							newParams = pClassInsTransf.createParamMap(
									pClassInsTransf.getHierarchyPrefix()
											+ pClassInst.getName() + DIV, "",
									fullPrefix + portPrefix, parents, ta, sc);
							copyTimeoutAndSensitivity(newParams, params);

							for (SCFunction portFun : pClassInst.getSCClass()
									.getMemberFunctions()) {
								if (portFun.getName().equals(funName)) {
									functionCallFound = true;
									pClassInsTransf.instantiateAndBindMethod(
											portFun, fullPrefix + portPrefix,
											pClassInsTransf
													.getHierarchyPrefix(),
											newParams, threadParams, parents,
											sc, ta, noopt, -1, modProcess);
								}
							}

							concatHashMapsPref(params, newParams, portPrefix);
						} else if (psi instanceof SCSocketInstance) {
							SCSocketInstance si = (SCSocketInstance) psi;
							// multisocket
							for (SCSocketInstance connectedSocket : si
									.getPortSocketInstances()) {
								SCClassInstance sClassInst = connectedSocket
										.getOwner();
								ClassInstanceTransformer sClassInsTransf = getParentTransformer()
										.getChildClassInstanceTransformerByInstance(
												sClassInst);
								if (sClassInsTransf == null) {
									logger.error(
											"instantiateAndBind could not find ClassInstanceTransformer for instance {}",
											sClassInst);
									return;
								}
								HashMap<String, String> newParams = new HashMap<String, String>();
								newParams = sClassInsTransf.createParamMap(
										sClassInsTransf.getHierarchyPrefix()
												+ sClassInst.getName() + DIV, "",
										fullPrefix + portPrefix, parents, ta,
										sc);
								copyTimeoutAndSensitivity(newParams, params);

								sClassInsTransf.instantiateAndBindMethod(
										fc.getFunction(), fullPrefix
												+ sClassInst.getName() + DIV,
										sClassInsTransf.getHierarchyPrefix(),
										newParams, threadParams, parents, sc,
										ta, noopt, -1, modProcess);

								concatHashMapsPref(params, newParams,
										portPrefix);
							}
							functionCallFound = true;
						} else {
							logger.error("expected port or socket but got {}",
									psi);
						}
					}
					if (!functionCallFound)
						logger.error(
								"function call {} for method {} in module instance {} not found.",
								fc, fun.getName(), scclassIns.getName());
				}
			} // end of function call (fc) loop

			// handle eventNotifications
			for (EventNotificationExpression evntNotify : fun
					.getEventNotifications()) {
				Expression event = evntNotify.getEvent();
				if (event instanceof SCVariableExpression
						&& ((SCVariableExpression) event).getVar() instanceof SCPeq) {
					// peq notification
					SCPeq peq = (SCPeq) ((SCVariableExpression) event).getVar();
					PeqTransformer peqTrans = PeqTransformer.getInstance(
							fullPrefix, peq);
					instantiateAndBindMethod(peq.getCallback(), fullPrefix,
							localPrefix, params, threadParams, parents, sc, ta,
							noopt, multiSocketIndex, modProcess);
					peqTrans.connectCallback(params.get(peq.getCallback()
							.getName() + "$ctrl"));
					peqTrans.addParamterConnections(params);
				}
			}

		}
		// if there already exist an instance of a non-time-consuming template,
		// only add params
		else {
			for (TAVariable funVar : FunctionTransformer
					.getNeededTmplParamsFromFunc(fun, "")) {
				params.put(funVar.getName(),
						fullPrefix + classPrefix + funVar.getName());
			}
		}
	}

	public void copyTimeoutAndSensitivity(HashMap<String, String> target,
			HashMap<String, String> source) {
		target.put(Constants.LOCAL_SENSITIVE_KEYWORD,
				source.get(Constants.LOCAL_SENSITIVE_KEYWORD));
		target.put(
				Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
						+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT,
				source.get(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
						+ Constants.SC_EVENT_TEMPLATE_PARAM_WAIT));
		target.put(
				Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY,
				source.get(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY));
		target.put(
				Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM,
				source.get(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_IMM));
		target.put(
				Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME,
				source.get(Constants.LOCAL_TIMEOUT_EVENT_KEYWORD + DIV
						+ Constants.SC_EVENT_TEMPLATE_PARAM_NOTIFY_TIME));
	}

	/**
	 * Inserts all mappings of the source map into the target map after adding
	 * the given prefix to every key. The source map remains unchanged.
	 * 
	 * @param target
	 * @param source
	 * @param prefix
	 */
	private void concatHashMapsPref(HashMap<String, String> target,
			HashMap<String, String> source, String prefix) {
		for (String key : source.keySet()) {
			target.put(prefix + key, source.get(key));
		}
	}

}
