%-------------------------------------------------
%  Solve the IVP
%   
%   x'=t^2-x, x0=1
%
%  over 0<t<5 with the Adams-Bashforth-Moulton predictor-corrector method.
%  The three starting values x1, x2, x3 can be estimated using the classical RK4
%  method.
%
%  The exact solution is
%
%  x(t)=t^2-2t+2-exp(-t).
%--------------------------------------------------
clear all
close all

%-------------------const-------------------------
t=0;
Tend=5; % right end of the time interval
x0=1; %initial condition
h=0.05; % time step
M=Tend/h; % amount of time steps
%------------------------------------------------
T=zeros(1,M); % time array
X=zeros(1,M); % space array
X(1)=x0;
%---------------starting steps with RK4------------
for j=1:3
    k1=feval('pr',T(j),X(j));
    k2=feval('pr',T(j)+h/2,X(j)+h*k1/2);
    k3=feval('pr',T(j)+h/2,X(j)+h*k2/2);
    k4=feval('pr',T(j)+h,X(j)+h*k3);
    X(j+1)=X(j)+h*(k1+2*k2+2*k3+k4)/6;
    T(j+1)=T(j)+h;
end;
%----------------predictor-corrector loop----------
F=zeros(1,4);
F=feval('pr',T(1:4),X(1:4));
for j=4:M-1
    % predictor
    p=X(j)+(h/24)*(F*[-9 37 -59 55]');
    T(j+1)=T(j)+h;
    % corrector
    F=[F(2) F(3) F(4) feval('pr',T(j+1),p)];
    X(j+1)=X(j)+(h/24)*(F*[1 -5 19 9]');
    F(4)=feval('pr',T(j+1),X(j+1));
end;
R=[T' X'];
%-----------------plot exact solution--------------
te=0:0.05:Tend;
xe=te.^2-2.*te+2-exp(-te);
plot(te,xe,'r')
hold on
%----------------plot numerical solution-----------
plot(R(:,1),R(:,2),'bo')
xlabel('t','FontName','Times New Roman','FontSize',16)
ylabel('x','FontName','Times New Roman','FontSize',16)
legend('exact solution','numerical solution')
