!*******************************************************************************
!
!  2D Swift-Hohenberg equation
!
!  Michael Wilczek
!
!  compile:
!  ifort sh.f90 -o sh -L/usr/local/dislin -ldislin -fpp -D comp_w_dislin -L/usr/local/fftw3/lib -lfftw3
!
!  execution:
!  ./sh input_file mask_file (optional)
!
!*******************************************************************************

MODULE para

IMPLICIT NONE

INTEGER :: dmx=128, dmy=128
INTEGER :: i_iter, n_iter=1e9
REAL :: amp=0.00001, eps=0.2, g=0.0
REAL :: k_init=0.0
REAL :: ratio=1.0, length=128.0
REAL :: dt=5e-3
REAL :: k_c=1.0

INTEGER :: screen_every=2000
LOGICAL :: screen=.TRUE.
LOGICAL :: trapdoor1=.TRUE.,trapdoor2=.TRUE.
INTEGER*8 :: plan_forw, plan_back, plan_dummy_back
INTEGER*8 :: plan_rhs_forw, plan_rhs_back
INTEGER :: ps=2500

NAMELIST /parameters/ dmx, dmy, n_iter ,dt, length, k_c, k_init, eps, g, screen, screen_every

END MODULE


!******************************************************************************
!******************************************************************************

MODULE global_arrays

IMPLICIT NONE

COMPLEX, DIMENSION(:,:), ALLOCATABLE  :: feld_copy
REAL, DIMENSION(:,:), ALLOCATABLE :: real_feld_copy

END MODULE


!******************************************************************************
!******************************************************************************


PROGRAM swift_hohenberg

USE para
USE global_arrays

IMPLICIT NONE

#INCLUDE 'fftw3.f'

INTEGER :: i,k
REAL :: zahl, pi=ACOS(-1.0), lin_fac
REAL, DIMENSION(:,:), ALLOCATABLE :: real_feld
REAL, DIMENSION(:,:), ALLOCATABLE :: real_mask
REAL, DIMENSION(:,:,:), ALLOCATABLE :: kvec
COMPLEX, DIMENSION(:,:), ALLOCATABLE  :: dummy
COMPLEX, DIMENSION(:,:), ALLOCATABLE  :: feld,k1,k2,k3,k4
CHARACTER(150) :: fileout="feld"


! Simulation initialisieren
CALL initialize_simulation()

! Dislin initialisieren >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
#ifdef comp_w_dislin
IF(screen) THEN
  CALL METAFL ('XWIN')    ! Dislin Initialisierung
  CALL PAGE(2*ps,ps)
  CALL SCRMOD('REVERS')
  CALL SCLMOD('FULL')
  CALL WINSIZ(1200,600)
  CALL DISINI()
  CALL AX3LEN((2*ps)/3,(2*ps)/3,(2*ps)/3)
END IF
#endif
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<


! Speicher allokieren >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
ALLOCATE(real_feld(dmx,dmy),real_mask(dmx,dmy),kvec(dmx/2+1,dmy,3),dummy(dmx/2+1,dmy))
ALLOCATE(feld(dmx/2+1,dmy),k1(dmx/2+1,dmy),k2(dmx/2+1,dmy),k3(dmx/2+1,dmy),k4(dmx/2+1,dmy))
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

! Variablen initialisieren >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
real_feld=0
kvec=0
dummy=0
feld=0
k1=0
k2=0
k3=0
k4=0
real_mask=0
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

! Rampe initialisieren >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CALL mask_init(real_mask)
!<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

! FFT-Plaene >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CALL sfftw_plan_dft_r2c_2d(plan_forw,dmx,dmy,real_feld,feld,FFTW_ESTIMATE)
CALL sfftw_plan_dft_c2r_2d(plan_back,dmx,dmy,feld,real_feld,FFTW_ESTIMATE)
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

! kvec initialisieren >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CALL kvec_init(kvec)
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

! Feld initialisieren >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CALL init_random_seed()

DO k=1,dmy
  DO i=1,dmx

    CALL RANDOM_NUMBER(zahl)
    real_feld(i,k)=SIN(REAL(i-1)/REAL(dmx)*length*k_init)+amp*(zahl-0.5)

  END DO
END DO


! AB in den FR
CALL sfftw_execute_dft_r2c(plan_forw,real_feld,feld)
feld=feld/REAL(dmx*dmy)**0.5

! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<


#ifdef rk4
! Zeitintegration RK4 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
DO i_iter=0,n_iter

#ifdef comp_w_dislin
  IF(MOD(i_iter,screen_every).EQ.0) THEN

    fileout="feld"
    dummy=feld

    CALL display(dummy,fileout)

    PRINT*,"time ",dt*i_iter
  END IF
#endif

  CALL sh_rhs(feld,k1,kvec,real_mask)
  CALL sh_rhs(feld+dt/2.0*k1,k2,kvec,real_mask)
  CALL sh_rhs(feld+dt/2.0*k2,k3,kvec,real_mask)
  CALL sh_rhs(feld+dt*k3,k4,kvec,real_mask)

  feld=feld+dt/6.0*(k1+2*k2+2*k3+k4)


END DO
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
#else

! Zeitintegration implizites Eulerverfahren >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
DO i_iter=0,n_iter

#ifdef comp_w_dislin
  IF(MOD(i_iter,screen_every).EQ.0) THEN

    fileout="feld"
    dummy=feld

    CALL display(dummy,fileout)

    PRINT*,"time ",dt*i_iter
  END IF
#endif

  ! Aufruf rhs
  CALL sh_rhs(feld,k1,kvec,real_mask)

  ! Zeitschritt durchfuehren
  DO k=1,dmy
    DO i=1,dmx/2+1

      feld(i,k)=(feld(i,k)/dt+k1(i,k))/(1.0/dt-(eps-(k_c**2.0-kvec(i,k,3))**2.0))


    END DO
  END DO


  ! reelle Symmetrie erzwingen
  DO k=2,dmy/2
    feld(1,k)=CMPLX( REAL(feld(1,dmy+2-k)), -AIMAG(feld(1,dmy+2-k)) )
  END DO


END DO
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
#endif


! Speicher freigeben >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
DEALLOCATE(real_feld,real_mask,kvec,dummy)
DEALLOCATE(feld,k1,k2,k3,k4)
DEALLOCATE(feld_copy,real_feld_copy)
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

#ifdef comp_w_dislin
IF(screen)  CALL DISFIN()
#endif


! FFTW Plaene freigeben >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CALL sfftw_destroy_plan(plan_forw)
CALL sfftw_destroy_plan(plan_back)
CALL sfftw_destroy_plan(plan_dummy_back)
CALL sfftw_destroy_plan(plan_rhs_forw)
CALL sfftw_destroy_plan(plan_rhs_back)
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

END PROGRAM


!***************************************************************************
!***************************************************************************


SUBROUTINE kvec_init(kvec)

USE para

IMPLICIT NONE

INTEGER :: i,k
REAL, DIMENSION(dmx/2+1,dmy,3) :: kvec
REAL :: pi=ACOS(-1.0)

! initialiue k-vectors
DO k=1,dmy
  DO i=1,dmx/2+1

    IF(k.LE.dmy/2+1) THEN
       kvec(i,k,1:2)=(/i-1,k-1/)
    ELSE
       kvec(i,k,1:2)=(/i-1,-dmy-1+k/)
    END IF

  END DO
END DO


! normalize to system length
kvec(:,:,1:2)=(2.0*pi*kvec(:,:,1:2))/length
! allow for aspect ration
kvec(:,:,2)=ratio*kvec(:,:,2)


! calculate square
DO k=1,dmy
  DO i=1,dmx/2+1

    kvec(i,k,3)=kvec(i,k,1)**2+kvec(i,k,2)**2

  END DO
END DO


PRINT*,""
PRINT*,"  kvec initialized"
PRINT*,""

END SUBROUTINE


!***************************************************************************
!***************************************************************************


#ifdef comp_w_dislin
SUBROUTINE display(feld_in,filename)

USE para

IMPLICIT NONE

#INCLUDE 'fftw3.f'

! Variablen >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
INTEGER :: i,k
REAL :: mini, maxi
COMPLEX, DIMENSION(dmx/2+1,dmy) :: feld_in
REAL, DIMENSION(dmx,dmy) :: real_feld
CHARACTER(150)  ::  filename
!<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

IF(trapdoor2) THEN
  CALL sfftw_plan_dft_c2r_2d(plan_dummy_back,dmx,dmy,feld_in,real_feld,FFTW_ESTIMATE)
  trapdoor2=.FALSE.
END IF

real_feld=0

IF(.NOT.screen) THEN
  CALL METAFL ('PNG')    ! Dislin Initialisierung
  CALL PAGE(2*ps,ps)
  CALL SCRMOD('REVERS')
  CALL SCLMOD('FULL')
  CALL WINSIZ(ps/2,ps/4)
  filename=TRIM(filename)//".png"
  CALL SETFIL(TRIM(filename))
  CALL DISINI()
  CALL AX3LEN((2*ps)/3,(2*ps)/3,(2*ps)/3)
END IF


! Initialisierung >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
mini=10
maxi=-10
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<



! Fouriertransformierte plotten >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
DO k=1,dmy
  DO i=1,dmx/2+1
        IF(k.LE.dmy/2+1)  THEN
          real_feld(i+dmx/2-1,k+dmy/2-1)=REAL(feld_in(i,k))**2+AIMAG(feld_in(i,k))**2
        ELSE
          real_feld(i+dmx/2-1,k-dmy/2-1)=REAL(feld_in(i,k))**2+AIMAG(feld_in(i,k))**2
        END IF
  END DO
END DO

DO k=1,dmy
  DO i=1,dmx/2
     real_feld(i,k)=real_feld(dmx-i,k)
  END DO
END DO

!real_feld=LOG(real_feld)
real_feld=(real_feld)**0.5

! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

! Fuer z-Skalierung Minimum und Maximum bestimmen >>>>>>>>>>>>>>>>>>>>>>>>>>>>>
maxi=MAXVAL(real_feld)
mini=MINVAL(real_feld)
!<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

! Graph formatieren, ausgeben und beenden >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CALL ERASE()
CALL SETVLT ('TEMP')
CALL AXSPOS(ps+ps/10,(5*ps)/6)
CALL AXSSCL('LIN','XY')
CALL NAME('','X')
CALL NAME('','Y')
CALL GRAF3(0.0, REAL(dmx/2), 0.0, REAL(dmx/4) , 0.0, REAL(dmy/2) ,0.0, REAL(dmy/4), mini, maxi, mini, (maxi-mini)/10)
!CALL GRAF3(0.0, REAL(dmx), 0.0, REAL(dmx/4) , 0.0, REAL(dmy) ,0.0, REAL(dmy/4), mini, maxi, mini, (maxi-mini)/10)
CALL TITLE()
CALL CRVMAT(real_feld(dmx/4+1:3*dmx/4,dmy/4+1:3*dmy/4),dmx/2,dmy/2,8,8)
!CALL CRVMAT(real_feld,dmx,dmy,8,8)
CALL COLOR('WHITE')
CALL ENDGRF()
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<


! FFT in den Ortsraum >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CALL sfftw_execute_dft_c2r(plan_dummy_back,feld_in,real_feld)
real_feld=real_feld/REAL(dmx*dmy)**0.5
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

! Fuer z-Skalierung Minimum und Maximum bestimmen >>>>>>>>>>>>>>>>>>>>>>>>>>>>>
maxi=MAXVAL(real_feld)
mini=MINVAL(real_feld)
!<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

! Graph formatieren, ausgeben und beenden >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CALL SETVLT ('RAIN')
CALL AXSPOS(ps/10,(5*ps)/6)
CALL AXSSCL('LIN','XY')
CALL NAME('','X')
CALL NAME('','Y')
CALL GRAF3(0.0, REAL(dmx), 0.0, REAL(dmx/4) , 0.0, REAL(dmy) ,0.0, REAL(dmy/4), mini, maxi, mini, (maxi-mini)/10)
CALL TITLE()
CALL CRVMAT(real_feld,dmx,dmy,8,8)
CALL COLOR('WHITE')
CALL ENDGRF()
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<


IF(.NOT.screen)  CALL DISFIN()


END SUBROUTINE
#endif


!***************************************************************************
!***************************************************************************


SUBROUTINE sh_rhs(feld_in,feld_out,kvec,real_mask)

USE para
USE global_arrays

IMPLICIT NONE

#INCLUDE 'fftw3.f'

INTEGER :: i,k
COMPLEX, DIMENSION(dmx/2+1,dmy), INTENT(IN)  :: feld_in
COMPLEX, DIMENSION(dmx/2+1,dmy), INTENT(OUT)  :: feld_out
REAL, DIMENSION(dmx/2+1,dmy,3), INTENT(IN)  :: kvec
REAL, DIMENSION(dmx,dmy), INTENT(IN)  :: real_mask


! Felder ggf allokieren
IF(.NOT.ALLOCATED(feld_copy)) ALLOCATE(feld_copy(dmx/2+1,dmy),real_feld_copy(dmx,dmy))

IF(trapdoor1) THEN
  CALL sfftw_plan_dft_r2c_2d(plan_rhs_forw,dmx,dmy,real_feld_copy,feld_copy,FFTW_ESTIMATE)
  CALL sfftw_plan_dft_c2r_2d(plan_rhs_back,dmx,dmy,feld_copy,real_feld_copy,FFTW_ESTIMATE)
  trapdoor1=.FALSE.
END IF


real_feld_copy=0
feld_copy=feld_in
feld_out=0

! FFT in den Ortsraum
CALL sfftw_execute_dft_c2r(plan_rhs_back,feld_copy,real_feld_copy)
real_feld_copy=real_feld_copy/REAL(dmx*dmy)**0.5


! Nichtlinearitaet berechnen >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

real_feld_copy=g*real_feld_copy**2.0-real_feld_copy**3.0+real_mask*real_feld_copy

! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<


! FFT in den Fourierraum
CALL sfftw_execute_dft_r2c(plan_rhs_forw,real_feld_copy,feld_copy)
feld_copy=feld_copy/REAL(dmx*dmy)**0.5


! Nichtlinearitaet zusammenbauen >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
#ifdef rk4
DO k=1,dmy
  DO i=1,dmx/2+1

    feld_out(i,k)=feld_copy(i,k)+eps*feld_in(i,k)-(k_c**2-kvec(i,k,3))**2*feld_in(i,k)

  END DO
END DO
#else
DO k=1,dmy
  DO i=1,dmx/2+1

    feld_out(i,k)=feld_copy(i,k)

  END DO
END DO
#endif
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

END SUBROUTINE


!***************************************************************************
!***************************************************************************


SUBROUTINE initialize_simulation()

USE para

IMPLICIT NONE

CHARACTER(150) :: parafile

CALL getarg(1,parafile)

OPEN(16,FILE=TRIM(parafile))
  READ(16,NML=parameters)
CLOSE(16)

CALL SYSTEM('clear')

PRINT*, ""
PRINT*, "  *********************************************"
PRINT*, "  *                                           *"
PRINT*, "  *        Swift-Hohenberg equation           *"
PRINT*, "  *                                           *"
PRINT*, "  *********************************************"
PRINT*, ""
PRINT*, "  parameters:"
PRINT*, ""
PRINT*, "  dmx ", dmx
PRINT*, "  dmy ", dmy
PRINT*, ""
PRINT*, "  n_iter ", n_iter
PRINT*, "  dt", dt
PRINT*, ""
PRINT*, "  length", length
PRINT*, "  k_c", k_c
PRINT*, ""
PRINT*, "  eps", eps
PRINT*, "  g", g
PRINT*, ""
PRINT*, "  screen", screen
PRINT*, "  screen_every", screen_every
PRINT*, ""
PRINT*, "  *********************************************"


END SUBROUTINE


!***************************************************************************
!***************************************************************************


SUBROUTINE init_random_seed()

INTEGER :: i, n, clock
INTEGER, DIMENSION(:), ALLOCATABLE :: seed

CALL RANDOM_SEED(size = n)
ALLOCATE(seed(n))

CALL SYSTEM_CLOCK(COUNT=clock)

seed = clock + 37 * (/ (i - 1, i = 1, n) /)

CALL RANDOM_SEED(PUT = seed)
DEALLOCATE(seed)

END SUBROUTINE


!***************************************************************************
!***************************************************************************


SUBROUTINE mask_init(real_mask)

USE para

IMPLICIT NONE

INTEGER :: i,k
INTEGER :: ierr
REAL, DIMENSION(dmx,dmy) :: real_mask
CHARACTER(150) :: maskfile

CALL getarg(2,maskfile)

real_mask=0

IF(.NOT.(maskfile.EQ."nomask".OR.maskfile.EQ."")) THEN 

! Maske laden >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
ierr=-1

PRINT*,""

OPEN(11,FILE=TRIM(maskfile), ACCESS='direct', FORM='unformatted', RECL=dmx*dmy,IOSTAT=ierr)

IF(ierr.NE.0) THEN
  PRINT*,"  Fehler beim Oeffnen von ", TRIM(maskfile),"!!"
  STOP
END IF

READ(11,rec=1,IOSTAT=ierr) real_mask

IF(ierr.NE.0) THEN
  PRINT*,"  Fehler beim Lesen von ", TRIM(maskfile),"!!"
  STOP
ELSE
  PRINT*,"  ",TRIM(maskfile)," gelesen."
END IF

CLOSE(11)
PRINT*,""
! <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

DO k=1,dmy
  DO i=1,dmx

    IF(real_mask(i,k).NE.0.0) THEN
       real_mask(i,k)=-3.0*eps
    END IF

  END DO
END DO

END IF

END SUBROUTINE