
using namespace std;
#include<iostream>
#include<complex>

#include "nefft.h"


#include <fftw.h>


// REALTYPE is float or double
// defined in makefile or nefft.h

/* fills x[0..N-1]
   with random numbers from [0,1)
 */

void FillVectorRandom(complex<REALTYPE> *x,int N) {
  int i;
  
  for(i=0;i<N;i++) 
    x[i]=complex<REALTYPE>(drand48(),drand48());
}

/* fills x[0..N-1]
   with random numbers in [-\pi,\pi]
 */ 
void FillVectorRandomNodes(REALTYPE *x,int N) {
  int i;

  for(i=0;i<N;i++) 
    x[i]=drand48()*M_PI*2.0 - M_PI;
  
}

/* 
   calculates the difference between the
   complex vectors x and y.
   defined as

             max_i | x[i]-y[i] |
   e_infty = -------------------
             max_i | x[i] |

 */

double e_infty(complex<REALTYPE> *x,complex<REALTYPE> *y,int N) {

  int i;
  double d;
  double num_max=0;
  double den_max=0;

  for(i=0;i<N;++i) {
    d = abs(x[i]-y[i]);
    
    if(d > num_max)
      num_max =d;
    
    d = abs( x[i] );
    if(d > den_max)
      den_max =d;
  }
  
  return num_max / den_max;
}


/* 
   calculates the difference between the
   complex vectors x and y.
   defined as

         sum_i | x[i]-y[i] |^2
   e_2 = -------------------
         sum_i | x[i] |^2

 */

double e_2(complex<REALTYPE> *x,complex<REALTYPE> *y,int N) {

  int i;
  double d;
  double num_sum=0;
  double den_sum=0;

  for(i=0;i<N;++i) {
    d = abs(x[i]-y[i]);
    num_sum += d*d;

    d = abs( x[i] );
    den_sum += d*d;

  }
  
  return num_sum / den_sum;
}

/* calcs
   y_j = \sum_{k=-N/2}^{N/2-1} x_k e^{i Nodes_j k}
   using direct computation.

   Note that k=-N/2..N/2-1 rather than 0..N
   but x_{-N/2} 0 data[0]
   j is 0..M-1
   internally the computation are always done in
   double precision.

*/

void ner_ft( complex<REALTYPE> *direct_result
	    ,complex<REALTYPE> *data
	    ,REALTYPE *nodes
	    ,int N,int M) {
  
  int j,k;
  complex<double> t,e,ee;
  
  for(j=0;j<M;++j) { 
    t=0;
    ee = complex<double> (cos((double)nodes[j]*(-N/2)), sin((double)nodes[j]*(-N/2)));
    e  = complex<double> (cos((double)nodes[j]) , sin((double)nodes[j]));
    for(k=0;k<N;++k) {
      t+=complex<double>((double)real(data[k]),(double)imag(data[k]))
	*ee;
      ee*=e;      
    }
    direct_result[j]=complex<REALTYPE>((REALTYPE)real(t),(REALTYPE)imag(t));
  }

}

void ned_ft(complex<REALTYPE> *direct_result
	    ,complex<REALTYPE> *data
	    ,REALTYPE *nodes
	    ,int N,int M) {
  
  int j,k;
  complex<double> t;
  
  for(j=0 ; j<N ; ++j) { 
    t = 0;
    for(k=0 ; k< M ; ++k)
      t += complex<double> ((double)real(data[k]),(double)imag(data[k]))
	* complex<double> ( cos((double)nodes[k]*(j-N/2)),sin((double)nodes[k]*(j-N/2)));
    direct_result[j]=complex<REALTYPE>((REALTYPE)real(t),(REALTYPE)imag(t));
  }
  
}


/* some debugging functions to print a vector */

void print_vector(complex<REALTYPE> *x,int N) {
  for(int i=0;i<N;++i)
    cout << x[i] << ' ';
  cout << endl;

}

void print_vector(REALTYPE *x,int N) {
  for(int i=0;i<N;++i)
    cout << i << ':' << x[i] << ' ';
  cout << endl;

}

void print_vector(int *x,int N) {
  for(int i=0;i<N;++i)
    cout << x[i] << ' ';
  cout << endl;

}

int main() {

  const int N_MAX=1024;

  const int L_MAX=13;

  int L=13; /* L=2K+1 is the length of interpolation. L=13 is for double prec.
	       for float precision L=7 will do. For these two special cases
	       (L=7, 13) the code has been optimized */

  int N,M,FFT_N;
  double E_2,E_infty;
  
  REALTYPE nodes[N_MAX];

  // input, fast_output and direct_output
  complex<REALTYPE> data[N_MAX];
  complex<REALTYPE> fast_result[N_MAX];
  complex<REALTYPE> direct_result[N_MAX];

  // precomputed stuff
  int      intnodes[N_MAX];  // intnodes is of size M
  REALTYPE weight[N_MAX]; // weight is of size N 
  REALTYPE coef[N_MAX*L_MAX]; // coef is of size M*L

  // tempspace
  REALTYPE temp [2*(2*N_MAX+L_MAX)]; // tempspace for FFT_N+L complex values
  REALTYPE temp2[4*N_MAX]; // FFT_N complex values, needed for 
                           // output of not in place FFT

  fftw_plan plan;
  
  for (N=32 ; N<=N_MAX ; N *= 2) {
    /* M= number of nonequispaced nodes. 
       In this demo we choose M=N
    */

    M=N; // number of equispaced nodes equals number of nonequispaced nodes

    cout << "N=" << N << endl;

    FillVectorRandomNodes(nodes,N);
    FillVectorRandom(data,N); 

    FFT_N=2*N; // oversampling c=2

    cout << "\tprecomputations...";
    /* precompute the coefficents for the nonequispaced fourier transform
       the coefficients can be used for ned as well as ner 
    */
    init_nefft_kb(N,FFT_N,L,M, nodes,
		  intnodes,
		  weight,
		  coef);

    // make FFTW plan
    fftw_plan plan=fftw_create_plan(FFT_N,FFTW_BACKWARD,FFTW_ESTIMATE);

    cout << "done" << endl;

    /* ned transform */

    cout << "\tned direct ned transform...";
    ned_ft(direct_result,data,nodes,N,M);

    cout << "done\n fast transform...";



    /* step 1: gridding */
    nedgridding((REALTYPE*)temp,(REALTYPE*)data,M,FFT_N,L,intnodes,coef);
    /* step 2: fft. use fftw library */
    fftw_one(plan,(fftw_complex*)temp,(fftw_complex*)temp2);
    /* step 3: scaling */
    nedscaling((REALTYPE*)fast_result,temp2,weight,N,FFT_N);
    
    cout << "done   ";

    E_2=e_2(direct_result,fast_result,N);
    E_infty=e_infty(direct_result,fast_result,N);

    cout << "\n\t E_2: " << E_2 << "  E_infty:" << E_infty << endl;


    /* ner transform */

    cout << "\tner direct ner transform...";
    ner_ft(direct_result,data,nodes,N,M);
    cout << "done   fast transform...";

    nerscaling(temp,(REALTYPE*)data,weight, N, FFT_N);

    fftw_one(plan,(fftw_complex*)temp,(fftw_complex*)temp2);
    
    nergridding((REALTYPE*)fast_result,temp2,M,FFT_N,L,intnodes,coef);  
    
    cout << "done   ";

    E_2=e_2(direct_result,fast_result,M);
    E_infty=e_infty(direct_result,fast_result,M);

    cout << "\n\t E_2: " << E_2 << "  E_infty:" << E_infty << endl;

  }

  return 0;
}
