%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% function u = rof(f, lambda, epsilon, dt, itermax, scmtype, difftype, u0)
%%
%% ROF: u_t = curv(u) + 2*lambda*(f-u)
%% 
%% input: f:        noisy signal/image, vector or matrix
%%        lambda:   scale parameter; default = 0.1
%%        epsilon:  tiny value number for approximating 
%%             |grad u| = sqrt{|grad u|^2+epsilon^2}, default = 1.0e-6
%%        dt:       timestep; default = 0.1
%%        itermax:  maximum iteration numbers; default = 1000;
%%        scmtype:  scheme: 1 = explicit, 2 = simple semi-implicit
%%                  default = 1
%%        difftype: 1 = D-D+; -1 = D+D-; 0 = D0; default: 1
%%        u0:       initial guess. optional
%%
%% output: u: denoised image
%%
%% Neumann boundary condition
%% 
%% required file: curvature.m
%% function [curv, normgrad, dxu, dyu] = curvature(u, dirtype, epsilon)
%%
%% created: 03/24/2005
%% last modified: 04/26/2005
%% author: jjxu@ucla
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function u = rof(f, lambda, epsilon, dt, itermax, scmtype, difftype, u0)

%% check input arguments
if exist('difftype') ~= 1 | ~isnumeric(difftype), 
    difftype = 1;  %% default: D-D+
else
    difftype = sign(difftype);
end

if exist('scmtype') ~= 1 | ~isnumeric(scmtype),
    scmtype = 1;
else 
    if scmtype ~= 2, scmtype = 1; end   %% explicit
end

if exist('itermax') ~= 1 | ~isnumeric(itermax) | itermax < 0, itermax = 1000; end;
if exist('dt') ~= 1 | ~isnumeric(dt) | dt < 0, dt = 0.1; end;
if exist('epsilon') ~= 1 | ~isnumeric(epsilon) | epsilon < 0, epsilon = 1.0e-3; end;
if exist('lambda') ~= 1 | ~isnumeric(lambda) | lambda < 0, lambda = 0.1; end;

if scmtype == 1, lambda2 = 2.0 * lambda; end;
if scmtype == 2, 
    i1p2lambdadt = 1.0 / (1.0+2.0*lambda*dt);
    lambda2f = (2.0 * lambda) * f;
end;

iter = 0;

if exist('u0') & size(u0)==size(f), u = u0; 
else u = f;
end

while iter < itermax
    iter = iter + 1;

    oldu = u;
    
    curvu = curvature(u, difftype, epsilon);
    
    switch scmtype 
        case 1
            u = oldu + dt * ( curvu + lambda2 * (f - oldu) );
        case 2
            u = i1p2lambdadt * ( oldu + dt * (curvu + lambda2f) );
    end
end

return  %% end